// -*- C++ -*-
// --------------------------------------------------------------------
// Bitmaps
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPEBITMAP_H
#define IPEBITMAP_H

#include "ipebase.h"
#include "ipexml.h"

// --------------------------------------------------------------------

class IPE_EXPORT IpeBitmap {
public:
  enum TFilter { EDirect, EFlateDecode, EDCTDecode };
  enum TColorSpace { EDeviceRGB, EDeviceGray, EDeviceCMYK };

  class IPE_EXPORT MRenderData {
  public:
    virtual ~MRenderData();
  };

  IpeBitmap();
  IpeBitmap(int width, int height,
	    TColorSpace colorSpace, int bitsPerComponent,
	    IpeBuffer data, TFilter filter, bool deflate = false);
  IpeBitmap(const IpeXmlAttributes &attr, IpeString data);
  IpeBitmap(const IpeXmlAttributes &attr, IpeBuffer data);

  IpeBitmap(const IpeBitmap &rhs);
  ~IpeBitmap();
  IpeBitmap &operator=(const IpeBitmap &rhs);

  void SaveAsXml(IpeStream &stream, int id, int pdfObjNum = -1) const;

  inline bool IsNull() const;
  bool Equal(IpeBitmap rhs) const;

  inline TColorSpace ColorSpace() const;
  inline TFilter Filter() const;
  inline int Components() const;
  inline int BitsPerComponent() const;
  inline int Width() const;
  inline int Height() const;

  inline const char *Data() const;
  inline int Size() const;

  inline int ObjNum() const;
  inline void SetObjNum(int objNum) const;

  inline MRenderData *RenderData() const;
  void SetRenderData(MRenderData *data) const;

  IpeBuffer PixelData() const;

  inline bool operator==(const IpeBitmap &rhs) const;
  inline bool operator!=(const IpeBitmap &rhs) const;
  inline bool operator<(const IpeBitmap &rhs) const;

private:
  int Init(const IpeXmlAttributes &attr);
  void ComputeChecksum();

private:
  struct Imp {
    int iRefCount;
    TColorSpace iColorSpace;
    int iBitsPerComponent;
    int iWidth;
    int iHeight;
    int iComponents;
    IpeBuffer iData;
    TFilter iFilter;
    int iChecksum;
    mutable int iObjNum;           // Object number (e.g. in PDF file)
    mutable MRenderData *iRender;  // cached pixmap to render it fast
  };

  Imp *iImp;
};

// --------------------------------------------------------------------

//! Is this a null bitmap?
inline bool IpeBitmap::IsNull() const
{
  return (iImp == 0);
}

//! Return the color space of the image.
inline IpeBitmap::TColorSpace IpeBitmap::ColorSpace() const
{
  return iImp->iColorSpace;
}

//! Return number of components per pixel.
inline int IpeBitmap::Components() const
{
  return iImp->iComponents;
}

//! Return the number of bits per component.
inline int IpeBitmap::BitsPerComponent() const
{
  return iImp->iBitsPerComponent;
}

//! Return width of pixel array.
inline int IpeBitmap::Width() const
{
  return iImp->iWidth;
}

//! Return height of pixel array.
inline int IpeBitmap::Height() const
{
  return iImp->iHeight;
}

//! Return the data filter of the image data.
inline IpeBitmap::TFilter IpeBitmap::Filter() const
{
  return iImp->iFilter;
}

//! Return a pointer to the image data (in PDF arrangement).
inline const char *IpeBitmap::Data() const
{
  return iImp->iData.data();
}

//! Return size (number of bytes) of image data (in PDF arrangement).
inline int IpeBitmap::Size() const
{
  return iImp->iData.size();
}

//! Return object number of the bitmap.
inline int IpeBitmap::ObjNum() const
{
  return iImp->iObjNum;
}

//! Set object number of the bitmap.
inline void IpeBitmap::SetObjNum(int objNum) const
{
  iImp->iObjNum = objNum;
}

//! Return cached bitmap for rendering.
inline IpeBitmap::MRenderData *IpeBitmap::RenderData() const
{
  return iImp->iRender;
}

//! Two bitmaps are equal if they share the same data.
inline bool IpeBitmap::operator==(const IpeBitmap &rhs) const
{
  return iImp == rhs.iImp;
}

//! Two bitmaps are equal if they share the same data.
inline bool IpeBitmap::operator!=(const IpeBitmap &rhs) const
{
  return iImp != rhs.iImp;
}

//! Less operator, to be able to sort bitmaps.
/*! The checksum is used, when it is equal, the shared address.
  This guarantees that bitmaps that are == (share their implementation)
  are next to each other, and blocks of them are next to blocks that
  are identical in contents. */
inline bool IpeBitmap::operator<(const IpeBitmap &rhs) const
{
  return (iImp->iChecksum < rhs.iImp->iChecksum ||
	  (iImp->iChecksum == rhs.iImp->iChecksum && iImp < rhs.iImp));
}

// --------------------------------------------------------------------
#endif
