// -*- C++ -*-
// --------------------------------------------------------------------
// The path object.
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPEPATH_H
#define IPEPATH_H

#include "ipeobj.h"

// --------------------------------------------------------------------

class IpeEllipse;
class IpeClosedSpline;
class IpeSegmentSubPath;

class IPE_EXPORT IpePathSegment {
public:
  enum TType { EArc, ESegment, EQuad, EBezier, ESpline };

  //! Type of segment.
  inline TType Type() const { return iType; }
  //! Number of control points.
  inline int NumCP() const { return iNumCP; }
  //! Return control point.
  inline IpeVector CP(int i) const { return iCP[i]; }
  //! Return last control point.
  inline IpeVector Last() const { return iCP[iNumCP - 1]; }
  //! Matrix (if Type() == EArc).
  inline IpeMatrix Matrix() const { return *iM; }

  IpeBezier Bezier() const;
  IpeArc Arc() const;
  void Beziers(std::vector<IpeBezier> &bez) const;

  void Draw(IpePainter &painter) const;
  void AddToBBox(IpeRect &box, const IpeMatrix &m) const;
  double Distance(const IpeVector &v, const IpeMatrix &m, double bound) const;
  void SnapVtx(const IpeVector &mouse, const IpeMatrix &m,
	       IpeVector &pos, double &bound) const;
  void SnapBnd(const IpeVector &mouse, const IpeMatrix &m,
	       IpeVector &pos, double &bound) const;
private:
  IpePathSegment(TType type, int num, const IpeVector *cp,
		 const IpeMatrix *m = 0);
private:
  TType iType;
  const IpeVector *iCP;
  int iNumCP;
  const IpeMatrix *iM;

  friend class IpeSegmentSubPath;
};

class IPE_EXPORT IpeSubPath {
public:
  //! The subpath types.
  enum TType { ESegments, EEllipse, EClosedSpline };
  virtual ~IpeSubPath() = 0;
  //! Return type of this subpath.
  virtual TType Type() const = 0;
  virtual bool Closed() const;

  //! Return a copy of this subpath.
  virtual IpeSubPath *Clone() const = 0;

  //! Return a transformed copy of this subpath.
  virtual IpeSubPath *Transform(const IpeMatrix &m) const = 0;

  virtual const IpeEllipse *AsEllipse() const;
  virtual IpeEllipse *AsEllipse();
  virtual const IpeClosedSpline *AsClosedSpline() const;
  virtual IpeClosedSpline *AsClosedSpline();
  virtual const IpeSegmentSubPath *AsSegs() const;
  virtual IpeSegmentSubPath *AsSegs();

  //! Save subpath to XML stream.
  virtual void SaveAsXml(IpeStream &stream) const = 0;
  //! Draw subpath (does not call DrawPath()).
  virtual void Draw(IpePainter &painter) const = 0;
  //! Add subpath to box.
  virtual void AddToBBox(IpeRect &box, const IpeMatrix &m) const = 0;
  //! Return distance from \a v to subpath transformed by \a m.
  virtual double Distance(const IpeVector &v, const IpeMatrix &m,
			  double bound) const = 0;
  //! Snap to vertex.
  virtual void SnapVtx(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const = 0;
  //! Snap to boundary of subpath.
  virtual void SnapBnd(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const = 0;
};

class IPE_EXPORT IpeEllipse : public IpeSubPath {
public:
  IpeEllipse(const IpeMatrix &m);
  virtual TType Type() const;
  virtual const IpeEllipse *AsEllipse() const;
  virtual IpeEllipse *AsEllipse();
  virtual IpeSubPath *Clone() const;
  virtual IpeSubPath *Transform(const IpeMatrix &m) const;
  //! Return matrix that transforms unit circle to the ellipse.
  inline IpeMatrix Matrix() const { return iM; }
  //! Set matrix.
  inline void SetMatrix(const IpeMatrix &m) { iM = m; }

  virtual void SaveAsXml(IpeStream &stream) const;
  virtual void Draw(IpePainter &painter) const;
  virtual void AddToBBox(IpeRect &box, const IpeMatrix &m) const;
  virtual double Distance(const IpeVector &v, const IpeMatrix &m,
			  double bound) const;
  virtual void SnapVtx(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const;
  virtual void SnapBnd(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const;
private:
  IpeMatrix iM;
};

class IPE_EXPORT IpeClosedSpline : public IpeSubPath {
public:
  IpeClosedSpline(const std::vector<IpeVector> &v);
  virtual TType Type() const;
  virtual const IpeClosedSpline *AsClosedSpline() const;
  virtual IpeClosedSpline *AsClosedSpline();
  virtual IpeSubPath *Clone() const;
  virtual IpeSubPath *Transform(const IpeMatrix &m) const;
  void Beziers(std::vector<IpeBezier> &bez) const;
  virtual void SaveAsXml(IpeStream &stream) const;
  virtual void Draw(IpePainter &painter) const;
  virtual void AddToBBox(IpeRect &box, const IpeMatrix &m) const;
  virtual double Distance(const IpeVector &v, const IpeMatrix &m,
			  double bound) const;
  virtual void SnapVtx(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const;
  virtual void SnapBnd(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const;
public:
  std::vector<IpeVector> iCP; // control points
};

class IPE_EXPORT IpeSegmentSubPath : public IpeSubPath {
public:
  IpeSegmentSubPath();
  virtual TType Type() const;
  inline virtual bool Closed() const { return iClosed; }
  virtual const IpeSegmentSubPath *AsSegs() const;
  virtual IpeSegmentSubPath *AsSegs();
  virtual IpeSubPath *Clone() const;
  virtual IpeSubPath *Transform(const IpeMatrix &m) const;
  virtual void SaveAsXml(IpeStream &stream) const;
  virtual void Draw(IpePainter &painter) const;
  virtual void AddToBBox(IpeRect &box, const IpeMatrix &m) const;
  virtual double Distance(const IpeVector &v, const IpeMatrix &m,
			  double bound) const;
  virtual void SnapVtx(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const;
  virtual void SnapBnd(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const;

  //! Return number of segments.
  /*! This does not include the closing segment for a closed path. */
  int NumSegments() const { return iSeg.size(); }
  IpePathSegment Segment(int i) const;
  IpePathSegment ClosingSegment(IpeVector u[2]) const;

  void Append(const IpePathSegment &seg);
  void Append(const IpeMatrix &m, const IpePathSegment &seg);
  void AppendReversed(const IpeMatrix &m, const IpePathSegment &seg);
  void AppendSegment(const IpeVector &v0, const IpeVector &v1);
  void AppendArc(const IpeMatrix &m, const IpeVector &v0,
		 const IpeVector &v1);
  void AppendQuad(const IpeVector &v0, const IpeVector &v1,
		  const IpeVector &v2);
  void AppendBezier(const IpeVector &v0, const IpeVector &v1,
		    const IpeVector &v2, const IpeVector &v3);
  void AppendSpline(const std::vector<IpeVector> &v);

  void SetClosed(bool closed);
  void MoveCP(int seg, int cp, const IpeVector &pos);
  void InsertCP(int seg, int cp, const IpeVector &pos);
  void InsertSegment(int seg);
  void DeleteSegment(int seg);
  void DeleteCP(int seg, int cp);
  void Straighten(int seg);
  void SetMatrix(int seg, const IpeMatrix &m);

private:
  int BeginCP(int seg) const;
  void RecomputeMatrix(int seg);

private:
  struct Seg {
    IpePathSegment::TType iType;
    int iLastCP;
    int iMatrix;
  };
  bool iClosed;
  std::vector<Seg> iSeg;
  std::vector<IpeVector> iCP; // control points
  std::vector<IpeMatrix> iM;  // for arcs
};

// --------------------------------------------------------------------

class IPE_EXPORT IpePath : public IpeFillable {
private:
  typedef std::vector<IpeSubPath *> IpeSubPathSeq;
public:
  explicit IpePath(IpeRepository *rep, const IpeXmlAttributes &attr,
		   IpeString data);
  explicit IpePath(const IpeAllAttributes &attr);
  explicit IpePath(const IpeAllAttributes &attr, const IpeRect &rect);
  explicit IpePath(const IpeAllAttributes &attr, const IpeSegment &seg);
  explicit IpePath(const IpeAllAttributes &attr,
		   const IpeVector &center, double radius);
  explicit IpePath(const IpeAllAttributes &attr,
		   const IpeVector &center, double radius,
		   double alpha0, double alpha1);

  IpePath(const IpePath &rhs);
  ~IpePath();
  IpePath &operator=(const IpePath &rhs);
  virtual IpeObject *Clone() const;

  virtual IpePath *AsPath();

  virtual void SaveAsXml(IpePainter &painter, IpeStream &stream,
			 IpeString layer) const;
  virtual void Draw(IpePainter &painter) const;
  virtual void Accept(IpeVisitor &visitor) const;
  virtual void AddToBBox(IpeRect &box, const IpeMatrix &m) const;
  virtual double Distance(const IpeVector &v, const IpeMatrix &m,
			  double bound) const;
  virtual void SnapVtx(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const;
  virtual void SnapBnd(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const;

  virtual void CheckStyle(const IpeStyleSheet *sheet,
			  IpeAttributeSeq &seq) const;

  //! Return arrow of the object.
  inline IpeAttribute ForwardArrow() const {return iForwardArrow; }
  //! Return backward arrow of the object.
  inline IpeAttribute BackwardArrow() const {return iBackwardArrow; }
  void SetForwardArrow(IpeAttribute size);
  void SetBackwardArrow(IpeAttribute size);

  static void DrawArrow(IpePainter &painter, IpeVector pos, IpeAngle alpha,
			IpeAttribute size);

  void DrawSpline(IpePainter &painter, int n,  const IpeVector *v) const;
  void DrawClosedSpline(IpePainter &painter, int n,
			const IpeVector *v) const;

  //! Return number of subpaths.
  inline int NumSubPaths() const { return iImp->iSubPaths.size(); }
  //! Return subpath.
  inline const IpeSubPath *SubPath(int i) const { return iImp->iSubPaths[i]; }
  void AddSubPath(IpeSubPath *sp);

private:
  void MakeArrowData();

  IpeAttribute iForwardArrow;
  IpeAttribute iBackwardArrow;

  struct SArrow {
    bool iOk;
    IpeVector iPos;
    IpeAngle iDir;
  };

  struct Imp {
    ~Imp();
    int iRefCount;
    IpeSubPathSeq iSubPaths;
    SArrow iForward;
    SArrow iBackward;
  };
  Imp *iImp;
};

// --------------------------------------------------------------------
#endif
