#ifndef CLS_H__
#define CLS_H__

#include <jni.h>
#include <jvmpi.h>
#include <jmp.h>

typedef struct cls_usage {
    /** Current count of bytes used by instances of this class. */
    long totalAlloced;
    /** Current count of instances of this class. */
    long totalInstances;
    /** The maximum number of simultanious instances of this class. */
    long maximumInstances;
    /** The total number of garbage collected instances of this class. */
    long totalGC;
    /** The total count of allocation levels. */
    long totalAllocLevel;
} cls_usage;

typedef struct field {
    char* field_name;
    unsigned char type;
    char* field_type;   /** the type of the field if object or array. */
} field;

/** The structure for a class instance.
 */
struct cls {
    char* class_name;     /** The name of the class. */
    char* source_name;    /** The source file of the class. */
    char* name;           /** The human readable name of the class. */
    jobjectID class_id;   /** The id of the class. */
    jint n_interfaces;    /** The number of interfaces this class implements. */
    cls** interfaces;     /** The interfaces this class implements. */
    jint n_statics;       /** The number of static variables. */
    field* statics;       /** The (types of the) static variables. */
    jint n_instances;     /** The number of instance variables. */
    field* instances;     /** The (types of the) instance variables. */
    cls_usage usage;      /** The current usage. */
    cls_usage restorer;   /** here we store data while under reset counting.. */
    cls* super;           /** The super class for this class. */
    int filtered;         /** if true, this class passes the current filter. */
    int modified;         /** Hack to avoid updating ui to much. */
};

/** Create a new class */
cls* cls_new (const char* class_name, 
	      const char* source_name, 
	      const jobjectID class_id,
	      const jint n_interfaces, 
	      const jint n_statics, 
	      const JVMPI_Field* statics,    /* static variables. */
	      const jint n_instances, 
	      const JVMPI_Field* instances); /* local variables. */

/** Delete a class. */
void cls_free (cls* c);

/** The hash function for classes. */
size_t cls_jmphash_func (void* c, size_t len);

/** compare the two classes.
 *  return 0 if they are equal. 
 *  return -1 if c1 is smaller
 *  return 1 if c1 is bigger.
 */
int cls_cmp_func (void* c1, void* c2);

/** print the given class on stdout. */
void cls_print (cls* c);

/** Get the class name of the class. */
inline char* cls_get_class_name (cls* c);

/** Get the source file name for the class. */
inline char* cls_get_source_name (cls* c);

/** Get the human readable name of the class. */
inline char* cls_get_name (cls* c);

/** Get the id of the class. */
inline jobjectID cls_get_class_id (cls* c);

/** Set the class id to the new id. */
inline void cls_set_class_id (cls* c, jobjectID class_id);

/** Get the current number of instances of this class. */
inline long cls_get_instances (cls* c);

/** Set the current number of instances of this class. 
 * Useful for relative counting. 
 */
inline void cls_set_instances (cls* c, long l);

/** Get the maximum number of simultanious instances of this class.. */
inline long cls_get_max_instances (cls* c);

/** Set the maximum number of simultanious instances of this class. 
 *  Useful for relative counting. 
 */
inline void cls_set_max_instances (cls* c, long l);

/** Get the number of bytes instances of this class currently use. */
inline long cls_get_size (cls* c);

/* Set the number of bytes this class is currently using. 
 * Useful for relative counting. 
 */
inline void cls_set_size (cls* c, long l);

/** Get the total number of instances of this class that 
 *  have been garbage collected.
 */
inline long cls_get_total_gc (cls* c);

/** Get the tenure number for this class. 
 *  The returned number is the sum of the gc-level for all 
 *  instances of this class.
 */
inline long cls_get_tenure (cls* c);

/** Add a new instance with given size to the current statistics. */
inline void cls_object_alloc (cls* c, jint size, jint gc_level);

/** An object has been freed. */
inline void cls_object_free (cls* c, jint size, jint gc_level);

/** An object has been freed. 
 *  The object was not GC:ed but rather removed due to heap dump 
 *  or event enable/disable.
 */
inline void cls_object_free_not_gc (cls* c, jint size);

/** Reset counters to zero... */
inline void cls_reset_count (cls* c);

/** Restore counters to real values. */
inline void cls_restore_count (cls* c);

/** Set the super class. */
inline void cls_set_super (cls* c, cls* super);

/** Get the super class */
inline cls* cls_get_super (cls* c);

/** Set the filtered mode of the given class. */
inline void cls_set_filtered (cls* c, int filtered);

/** Get the filtered mode of the given class. */
inline int cls_get_filtered (cls* c);

/** Check if this class has changed since last call. */
inline int cls_check_modified (cls* c);

/** Set updated status of this class. */
inline void cls_set_modified (cls* c, int flag);

/** Get the field type as a readable string. */
char* get_field_type (field* f);

#endif /* CLS_H__ */

/* Emacs Local Variables: */
/* Emacs mode:C */
/* Emacs c-indentation-style:"gnu" */
/* Emacs c-hanging-braces-alist:((brace-list-open)(brace-entry-open)(defun-open after)(substatement-open after)(block-close . c-snug-do-while)(extern-lang-open after)) */
/* Emacs c-cleanup-list:(brace-else-brace brace-elseif-brace space-before-funcall) */
/* Emacs c-basic-offset:4 */
/* Emacs End: */
