#include <Python.h>
#include <datetime.h>
#include <errno.h>

#include <time.h>
#include "pytype_timezone.h"
#include "pytype_basics.h"
#include "structmember.h"
#include "python_compatibility.h"

/*******************************************************
 * Create and delloc methods for objects 
 ******************************************************/

/**
 * Create a new palm timezone.
 *
 * @param type the type of object to create
 *
 * @param args the arguments to the constructor
 *
 * @param kwds the keyword arguments
 *
 * @return a new jppy calendar event object
 */
extern PyObject* PyPiTimezone_New(PyTypeObject *type, PyObject *args, PyObject *kwds) {
  int i;
  PyPiTimezone* self;

  /* Why do we have to do this here? The one in the swig init doesn't seem
     to work ?! */
  PyDateTime_IMPORT;
  
  TimezoneType.ob_type = &PyType_Type;
  self = (PyPiTimezone *)type->tp_alloc(type, 0);
  new_Timezone(&(self->tz));

  return (PyObject*)self;
}

/**
 * Initialize a palm timezone event.
 *
 * @param args the arguments to the constructor
 *
 * @param kwds the keyword arguments: timezone the timezone to copy
 *
 * @return -1 on failure, 0 on success
 */
extern int PyPiTimezone_Init(PyObject *self, PyObject *args, PyObject *kwds) {
  PyPiTimezone* fromtimezone = NULL;
  PyPiTimezone* timezone = NULL;
  int i;
  
  static char *kwlist[] = {"timezone", NULL};
  if (!PyArg_ParseTupleAndKeywords(args, kwds, "|O", kwlist, 
				   &fromtimezone)) {
    return -1;
  }
  
  timezone = (PyPiTimezone*)self;
  /* we have to support calling __init__ more than once */
  free_Timezone(&(timezone->tz));

  if ((fromtimezone == NULL) || ((PyObject *)fromtimezone == Py_None)) {
    /* Initialise attributes custom to this type of object */
    new_Timezone(&(timezone->tz));
  } else {
    if (!PyPiTimezone_Check(fromtimezone)) {
      PyErr_SetString(PyExc_TypeError,"Must provide a Timezone object to share");
      return -1;
    }

    /* Copy the record */
    if(copy_Timezone(&(fromtimezone->tz), &(timezone->tz)) < 0) {
      PyErr_SetFromErrno(PyExc_SystemError);
      return -1;
    }
  }

  return 0;
}

static PyObject * PyPiTimezone_Allocate(PyTypeObject *type, int nitems) {
  PyPiTimezone *timezone;
  if (type == &TimezoneType) {
    timezone = PyObject_New(PyPiTimezone, &TimezoneType);
    return (PyObject *) timezone;
  } else {
    /* Is this for subclasses ? */
    timezone = (PyPiTimezone *)PyType_GenericAlloc(type, nitems);
    return (PyObject*)timezone;
  }
}

/**
 * Wrap an existing timezone in a python accessible version. This makes a copy
 * of fromtimezone into a newly allocated struct inside the python object.
 *
 * @param a the palm timezone object
 * @return a newly created python event object that represents the data in a
 */
extern PyObject* PyPiTimezone_Wrap(Timezone_t* fromtimezone) {
  PyPiTimezone* timezone;
  int i;

  timezone = (PyPiTimezone*)PyPiTimezone_New(&TimezoneType,NULL,NULL);

  copy_Timezone(fromtimezone, &(timezone->tz));
  
  return (PyObject*)timezone;
}

static void PyPiTimezone_Dealloc(PyPiTimezone* self) {
  free_Timezone(&(self->tz));
  self->ob_type->tp_free((PyObject*)self);
}


/**
 * Compare two timezone objects.
 *
 * @param self the first object
 * @param other the second object
 * @return standard comparison values (-1, 0, 1)
 *
 */
static int PyPiTimezone_Compare(PyPiTimezone* self,PyPiTimezone *other) {
  int s, o;

  s = self->tz.offset;
  o = other->tz.offset;

  if (s == o) {
    return 0;
  } else if (s > 0) {
    return 1;
  } else {
    return -1;
  }
}

static char *PyPiTimezone_key_list[] = {
  "offset", /* int */
  "t2", /* int */
  "dstStart.dayOfWeek", /* int */
  "dstStart.weekOfMonth", /* int */
  "dstStart.month", /* int */
  "dstStart.unknown", /* int */
  "dstEnd.dayOfWeek", /* int */
  "dstEnd.weekOfMonth", /* int */
  "dstEnd.month", /* int */
  "dstEnd.unknown", /* int */
  "dstObserved", /* boolean */
  "unknown", /* char */
  "name", /* unicode string */
  NULL};

static PyObject* PyPiTimezone_keys(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiTimezone_key_list[n]) {
    PyObject *value;
    value = PyString_FromString(PyPiTimezone_key_list[n++]);
    PyList_Append(list, value);
    Py_DECREF(value);
  }
  return list;
}

/* forward declaration */
PyObject *PyPiTimezone_GetItem(PyPiTimezone* self,  PyObject* key);

static PyObject* PyPiTimezone_values(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiTimezone_key_list[n]) {
    PyObject *key;
    PyObject *value;
    key   = PyString_FromString(PyPiTimezone_key_list[n++]);
    value = PyPiTimezone_GetItem((PyPiTimezone *)self, key);
    PyList_Append(list, value);
    Py_DECREF(key);
    Py_DECREF(value);
  }
  return list;
}

static PyObject* PyPiTimezone_items(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiTimezone_key_list[n]) {
    PyObject *key, *value, *tuple;
    key = PyString_FromString(PyPiTimezone_key_list[n++]);
    value = PyPiTimezone_GetItem((PyPiTimezone *)self, key);
    tuple = Py_BuildValue("(OO)", key, value);
    PyList_Append(list, tuple); /* get it's own ref */
    Py_DECREF(key);
    Py_DECREF(value);
    Py_DECREF(tuple);
  }
  return list;
}

/*******************************************************
 * 
 ******************************************************/

static PyMethodDef PyPiTimezone_Methods[] = {
  { "keys", (PyCFunction)PyPiTimezone_keys, METH_NOARGS, "Return a list of available keys"},
  { "items",(PyCFunction)PyPiTimezone_items, METH_NOARGS, "Return a list of available items"},
  { "values",(PyCFunction)PyPiTimezone_values, METH_NOARGS, "Return a list of available items"},
  {NULL,NULL,0,NULL} /* Sentinel */
};

static PyMemberDef PyPiTimezone_Members[] = {
  PYPI_MEMBERS_HEAD,
  {NULL}  /* Sentinel */
};

static PyGetSetDef PyPiTimezone_Getseters[] = {
  PYPI_GETSETERS_HEAD,
  {NULL}  /* Sentinel */
};

/**** mapping interface ****/
int PyPiTimezone_Len(self) {
  int len=0;
  while(PyPiTimezone_key_list[len] != NULL) {
    ++len;
  }
  return len;
}

PyObject *PyPiTimezone_GetItem(PyPiTimezone* self,  PyObject* key) {
  char *keystring;
  if (!PyString_Check(key)) {
    Py_INCREF(Py_None);
    return Py_None;  
  }

  Py_INCREF(key);
  keystring = PyString_AsString(key);

  GET_INT_ATTR(keystring, "offset", tz.offset);
  GET_INT_ATTR(keystring, "t2", tz.t2);
  GET_INT_ATTR(keystring, "dstStart.dayOfWeek", tz.dstStart.dayOfWeek);
  GET_INT_ATTR(keystring, "dstStart.weekOfMonth", tz.dstStart.weekOfMonth);
  GET_INT_ATTR(keystring, "dstStart.month", tz.dstStart.month);
  GET_INT_ATTR(keystring, "dstStart.unknown", tz.dstStart.unknown);
  GET_INT_ATTR(keystring, "dstEnd.dayOfWeek", tz.dstEnd.dayOfWeek);
  GET_INT_ATTR(keystring, "dstEnd.weekOfMonth", tz.dstEnd.weekOfMonth);
  GET_INT_ATTR(keystring, "dstEnd.month", tz.dstEnd.month);
  GET_INT_ATTR(keystring, "dstEnd.unknown", tz.dstEnd.unknown);
  GET_BOOL_ATTR(keystring, "dstObserved", tz.dstObserved);
  GET_INT_ATTR(keystring, "unknown", tz.unknown);
  GET_STRING_ATTR(keystring,"name", tz.name);

  PyErr_Format(PyExc_KeyError,"no such key '%s'", keystring);
  Py_DECREF(key);
  return NULL;
}

int PyPiTimezone_SetItem(PyPiTimezone* self, PyObject* key, PyObject* value) {
  char buf[255];
  char *keystring;

  if (!PyString_Check(key)) {
    PyErr_SetString(PyExc_TypeError,"key must be a String");
    return -1;
  }
  
  Py_INCREF(key);
  keystring = PyString_AsString(key);

  if (value == NULL) {
    PyErr_Format(PyExc_ValueError,"Can't delete value %s", keystring);
    return -1;
  }

  SET_INT_ATTR(keystring, "offset", tz.offset, value, buf, 255);
  SET_INT_ATTR(keystring, "t2", tz.t2, value, buf, 255);
  SET_INT_ATTR(keystring, "dstStart.dayOfWeek", tz.dstStart.dayOfWeek, value, buf, 255);
  SET_INT_ATTR(keystring, "dstStart.weekOfMonth", tz.dstStart.weekOfMonth, value, buf, 255);
  SET_INT_ATTR(keystring, "dstStart.month", tz.dstStart.month, value, buf, 255);
  SET_INT_ATTR(keystring, "dstStart.unknown", tz.dstStart.unknown, value, buf, 255);
  SET_INT_ATTR(keystring, "dstEnd.dayOfWeek", tz.dstEnd.dayOfWeek, value, buf, 255);
  SET_INT_ATTR(keystring, "dstEnd.weekOfMonth", tz.dstEnd.weekOfMonth, value, buf, 255);
  SET_INT_ATTR(keystring, "dstEnd.month", tz.dstEnd.month, value, buf, 255);
  SET_INT_ATTR(keystring, "dstEnd.unknown", tz.dstEnd.unknown, value, buf, 255);
  SET_BOOL_ATTR(keystring, "dstObserved", tz.dstObserved, value, buf, 255);
  SET_INT_ATTR(keystring, "unknown", tz.unknown, value, buf, 255);
  SET_STRING_ATTR(keystring,"name", tz.name, value, 256);

  PyErr_SetString(PyExc_KeyError,"no such key");
  Py_DECREF(key);
  return -1;
}

static PyMappingMethods PyPiTimezone_Mapping = {
  (lenfunc)PyPiTimezone_Len,
  (binaryfunc)PyPiTimezone_GetItem, 
  (objobjargproc)PyPiTimezone_SetItem,
};

/*******************************************************
 * Declare the type
 ******************************************************/


PyTypeObject TimezoneType = {
  PyObject_HEAD_INIT(NULL)
  0,
  "jppy._jpilot.__jpilot._Timezone",
  sizeof(PyPiTimezone),
  0, /* tp_itemsize*/
  (destructor)PyPiTimezone_Dealloc,      /*tp_dealloc*/
  0,                                /*tp_print*/
  0, /*tp_getattr*/
  0, /*tp_setattr*/
  (cmpfunc)PyPiTimezone_Compare,     /*tp_compare*/
  0,       /*tp_repr*/
  0,                                /*tp_as_number*/
  0,                                /*tp_as_sequence*/
  &PyPiTimezone_Mapping,                                /*tp_as_mapping*/
  0,                                /*tp_hash */
  0,                         /*tp_call*/
  0,                         /*tp_str*/
  0,                         /*tp_getattro*/
  0,                         /*tp_setattro*/
  0,                         /*tp_as_buffer*/
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
  "Timezone object",           /* tp_doc */
  0,		               /* tp_traverse */
  0,		               /* tp_clear */
  0,		               /* tp_richcompare */
  0,		               /* tp_weaklistoffset */
  0,		               /* tp_iter */
  0,		               /* tp_iternext */
  PyPiTimezone_Methods,             /* tp_methods */
  PyPiTimezone_Members,            /* tp_members */
  PyPiTimezone_Getseters,          /* tp_getset */
  0,                         /* tp_base */
  0,                         /* tp_dict */
  0,                         /* tp_descr_get */
  0,                         /* tp_descr_set */
  0,                         /* tp_dictoffset */
  (initproc)PyPiTimezone_Init,      /* tp_init */
  /*PyType_GenericAlloc*/(allocfunc)PyPiTimezone_Allocate,                 /* tp_alloc */
  /*PyType_GenericNew*/(newfunc)PyPiTimezone_New,                 /* tp_new */
  _PyObject_Del, /* Low-level free-memory routine, tp_free */
};
