#include <Python.h>

#include <time.h>
#include "pytype_todos.h"
#include "pytype_basics.h"
#include "structmember.h"
#include "python_compatibility.h"

/*******************************************************
 * Create and delloc methods for objects
 ******************************************************/

extern PyObject* PyPiTodo_New(PyTypeObject *type, PyObject *args, PyObject *kwds) {
  int i;
  PyPiTodo* self;

  /* Why do we have to do this here? The one in the swig init doesn't seem
     to work ?! */
  mxDateTime_ImportModuleAndAPI();

  TodoType.ob_type = &PyType_Type;
  self = (PyPiTodo *)type->tp_alloc(type, 0);
  new_ToDo(&(self->a));
  SetBasicRecordObjectAttributeDefaults((PyObject*) self, pack_ToDo);
  self->filters = NULL;

  return (PyObject*)self;
}

extern int PyPiTodo_Init(PyObject *self, PyObject *args, PyObject *kwds) {
  PyPiTodo* fromtodo = NULL;
  PyPiTodo* todo = NULL;
  PyObject* filters = NULL;
  int i;

  static char *kwlist[] = {"todo","record_field_filters", NULL};
  if (!PyArg_ParseTupleAndKeywords(args, kwds, "|OO", kwlist,
				   &fromtodo, &filters)) {
    return -1;
  }

  todo = (PyPiTodo*)self;
  /* we have to support calling __init__ more than once */
  if (todo->filters != NULL) {
    Py_DECREF(todo->filters);
    todo->filters = NULL;
  }
  if (filters != NULL) {
    todo->filters = filters;
    Py_INCREF(filters);
  }
  free_ToDo(&(todo->a));
  if (todo->saved_br.size > 0 && todo->saved_br.buf) {
    free(todo->saved_br.buf);
  }

  if ((fromtodo == NULL) || ((PyObject *)fromtodo == Py_None)) {
    /* Initialise attributes custom to this type of object */
    new_ToDo(&(todo->a));
    SetBasicRecordObjectAttributeDefaults((PyObject*) todo, pack_ToDo);
  } else {
    if (!PyPiTodo_Check(fromtodo)) {
      PyErr_SetString(PyExc_TypeError,"Must provide a Todo object to share");
      return -1;
    }

    /* copy all the database agnostic record attributes */
    todo->saved_br.size = fromtodo->saved_br.size;
    todo->saved_br.attrib = fromtodo->saved_br.attrib;
    todo->saved_br.rt = fromtodo->saved_br.rt;
    todo->saved_br.unique_id = fromtodo->saved_br.unique_id;

    todo->rt = fromtodo->rt;
    todo->unique_id = fromtodo->unique_id;

    todo->saved_br.buf = malloc(fromtodo->saved_br.size);
    memcpy(todo->saved_br.buf,
	   fromtodo->saved_br.buf,
	   fromtodo->saved_br.size);

    todo->category = fromtodo->category;
    todo->unsaved_changes = fromtodo->unsaved_changes;
    todo->deleted = fromtodo->deleted;
    todo->modified = fromtodo->modified;
    todo->busy = fromtodo->busy;
    todo->secret = fromtodo->secret;

    /* copy indefinite, tm, priority and complete */
    memcpy(&(todo->a), &(fromtodo->a), sizeof(struct ToDo));

    /* Now do the pointers */

    pyint_strcpy(todo->a.description, fromtodo->a.description);
    pyint_strcpy(todo->a.note, fromtodo->a.note);

  }

  return 0;
}

static PyObject * PyPiTodo_Allocate(PyTypeObject *type, int nitems) {
  PyPiTodo *todo;
  if (type == &TodoType) {
    todo = PyObject_New(PyPiTodo, &TodoType);
    return (PyObject *) todo;
  } else {
    /* Is this for subclasses ? */
    todo = (PyPiTodo *)PyType_GenericAlloc(type, nitems);
    return (PyObject*)todo;
  }
}

/**
 * Wrap an existing todo in a jppy todo. This makes
 * a copy of a into a newly allocated struct inside the python object.
 *
 * @param a the palm object
 * @param rt
 * @param unique_id
 * @param attrib
 * @param size
 * @param buf
 * @return a newly created python event object that represents the data in a
 */
extern PyObject* PyPiTodo_Wrap(struct ToDo* a, PCRecType rt,
				 unsigned int unique_id, unsigned char attrib,
				 int size, void* buf, PyObject *filters) {
  PyPiTodo* todo;
  int i;

  PyObject *python_mod, *python_mdict, *todo_class, *python_args, *python_kw;
  python_mod = PyImport_Import(PyString_FromString("jppy.jpilot.legacy"));
  if (python_mod == NULL) {
    PyErr_Print();
    return NULL;
  }
  python_mdict = PyModule_GetDict(python_mod);
  if (python_mdict == NULL) {
    PyErr_Print();
    Py_DECREF(python_mod);
    return NULL;
  }
  Py_INCREF(python_mdict);
  Py_DECREF(python_mod);
  todo_class = PyDict_GetItemString(python_mdict, "Todo"); /* borrowed reference */
  if (todo_class == NULL) {
    PyErr_Print();
    Py_DECREF(python_mdict);
    return NULL;
  }
  Py_INCREF(todo_class);
  python_args = Py_BuildValue("()");
  python_kw = Py_BuildValue("{s:O}","record_field_filters",filters);
  todo = (PyPiTodo*) PyObject_Call(todo_class, python_args, python_kw);
  Py_DECREF(todo_class);
  Py_DECREF(python_args);
  Py_DECREF(python_kw);
  if (todo == NULL) {
    PyErr_Print();
    return NULL;
  }
  Py_INCREF(todo);

  /* copy indefinite, tm, priority and complete */
  memcpy(&(todo->a), a, sizeof(struct ToDo));


  /* set saved_br stuff, and rt and unique_id, and attrib derived
     details for the current todo */
  SetSavedBrAndRTandUniqueIDandAttribs(rt, unique_id, attrib, size, buf, (PyObject *)todo);

  pyp_strcpy(todo->a.description, a->description);
  pyp_strcpy(todo->a.note, a->note);

  return (PyObject*)todo;
}

static void PyPiTodo_Dealloc(PyPiTodo* self) {
  free_ToDo(&(self->a));
  if (self->filters != NULL) {
    Py_DECREF(self->filters);
    self->filters = NULL;
  }
  if (self->saved_br.size > 0 && self->saved_br.buf) {
    free(self->saved_br.buf);
  }
  self->ob_type->tp_free((PyObject*)self);
}


static int PyPiTodo_Compare(PyPiTodo* self,PyPiTodo *other) {
  int res;

  if ((self->a.description) &&
      (other->a.description)) {

    res = strcasecmp(self->a.description,
		 other->a.description);
    if (res > 0) {
      res = 1;
    } else if (res < 0) {
      res = -1;
    } else {
      res = 0;
    }
  } else if (self->a.description) {
    res = -1;
  } else if (other->a.description) {
    res = 1;
  } else {
    res = 0;
  }

  return res;

}

static char *PyPiTodo_key_list[] = {
  "description",
  "note",
  "due",
  "complete",
  "priority",
  NULL};

static PyObject* PyPiTodo_keys(PyObject* self) {
  PyObject *list = PyList_New(0);
  int n = 0;

  while (PyPiTodo_key_list[n]) {
    PyObject *value;
    value = PyString_FromString(PyPiTodo_key_list[n++]);
    PyList_Append(list, value);
    Py_DECREF(value);
  }
  PyPi_extend_keys_from_filters((PyPiBase*)self, list);

  return list;
}

/* forward declaration */
PyObject *PyPiTodo_GetItem(PyPiTodo* self,  PyObject* key);

static PyObject* PyPiTodo_values(PyObject* self) {
  PyObject *list = PyList_New(0);
  PyObject *keys = PyPiTodo_keys(self);
  int n = 0;

  for (n=0;n<PyList_Size(keys);n++) {
    PyObject *value;
    value = PyPiTodo_GetItem((PyPiTodo *)self, PyList_GET_ITEM(keys, n));
    if (value == NULL) {
      Py_DECREF(keys);
      return NULL;
    }
    PyList_Append(list, value);
    Py_DECREF(value);
  }
  Py_DECREF(keys);
  return list;
}

static PyObject* PyPiTodo_items(PyObject* self) {
  PyObject *list = PyList_New(0);
  PyObject *keys = PyPiTodo_keys(self);
  int n = 0;

  for (n=0;n<PyList_Size(keys);n++) {
    PyObject *key, *value, *tuple;
    key = PyList_GET_ITEM(keys, n);
    Py_INCREF(key);
    value = PyPiTodo_GetItem((PyPiTodo *)self, key);
    if (value == NULL) {
      Py_DECREF(keys);
      Py_DECREF(key);
      return NULL;
    }
    tuple = Py_BuildValue("(OO)", key, value);
    PyList_Append(list, tuple); /* get it's own ref */
    Py_DECREF(key);
    Py_DECREF(value);
    Py_DECREF(tuple);
  }
  Py_DECREF(keys);
  return list;
}

/*******************************************************
 *
 ******************************************************/

static PyMethodDef PyPiTodo_Methods[] = {
  { "keys", (PyCFunction)PyPiTodo_keys, METH_NOARGS, "Return a list of available keys"},
  { "items",(PyCFunction)PyPiTodo_items, METH_NOARGS, "Return a list of available items"},
  { "values",(PyCFunction)PyPiTodo_values, METH_NOARGS, "Return a list of available items"},
  {NULL,NULL,0,NULL} /* Sentinel */
};

static PyMemberDef PyPiTodo_Members[] = {
  PYPI_MEMBERS_HEAD,
  {NULL}  /* Sentinel */
};

static PyGetSetDef PyPiTodo_Getseters[] = {
  PYPI_GETSETERS_HEAD,
  {NULL}  /* Sentinel */
};

/**** mapping interface ****/
int PyPiTodo_Len(PyObject* self) {
  int len=0;
  // quite expensive way to do it, but we need to get filters too
  PyObject *keys = PyPiTodo_keys(self);
  len = PySequence_Size(keys);
  Py_DECREF(keys);
  return len;
}

PyObject *PyPiTodo_GetItem(PyPiTodo* self,  PyObject* key) {
  char *keystring;
  PyObject *result;

  if (!PyString_Check(key)) {
    Py_INCREF(Py_None);
    return Py_None;
  }

  if ((result = PyPi_GetItem_from_filters((PyPiBase *)self, key)) != NULL)
    return result;
  else if (PyErr_Occurred() != NULL)
    return NULL;

  Py_INCREF(key);
  keystring = PyString_AsString(key);

  GET_STRING_ATTR(keystring,"description", a.description);
  GET_STRING_ATTR(keystring,"note", a.note);

  GET_DATE_AND_FLAG(keystring,"due",a.due,a.indefinite);

  GET_INT_ATTR(keystring,"complete",a.complete);
  GET_INT_ATTR(keystring,"priority",a.priority);

  PyErr_Format(PyExc_KeyError,"no such key '%s'", keystring);
  Py_DECREF(key);
  return NULL;
}

int PyPiTodo_SetItem(PyPiTodo* self, PyObject* key, PyObject* value) {
  char buf[255];
  char *keystring;

  if (!PyString_Check(key)) {
    PyErr_SetString(PyExc_TypeError,"key must be a String");
    return -1;
  }

  if (PyPi_SetItem_from_filters((PyPiBase *)self, key, value) > 0)
    return 0;
  else if (PyErr_Occurred() != NULL)
    return -1;

  Py_INCREF(key);
  keystring = PyString_AsString(key);

  if (value == NULL) {
    PyErr_Format(PyExc_ValueError,"Can't delete value %s", keystring);
    return -1;
  }

  SET_STRING_ATTR(keystring,"description",a.description,value, 256);
  SET_STRING_ATTR(keystring,"note",a.note,value, 4096);

  SET_DATE_AND_FLAG_ATTR(keystring,"due",a.due,a.indefinite,value);

  SET_BOUNDED_INT_ATTR(keystring,"complete",a.complete,value,0,1,buf,255);
  SET_BOUNDED_INT_ATTR(keystring,"priority",a.priority,value,1,5,buf,255);

  PyErr_SetString(PyExc_KeyError,"no such key");
  Py_DECREF(key);
  return -1;
}

static PyMappingMethods PyPiTodo_Mapping = {
  (lenfunc)PyPiTodo_Len,
  (binaryfunc)PyPiTodo_GetItem,
  (objobjargproc)PyPiTodo_SetItem,
};



/*******************************************************
 * Provide a repr method
 ******************************************************/
static PyObject *PyPiTodo_Repr(PyPiTodo* self) {
  static PyObject *format = NULL;
  PyObject *attrib, *args, *result;
  int len1;

  if (format == NULL) {
    format = PyString_FromString("<%s %r %s>");
    if (format == NULL)
      return NULL;
  }

  if (self->a.description) {
    len1 = strlen(self->a.description) > 25 ? 25 : strlen(self->a.description);
  } else {
    len1 = 0;
  }

  args = Py_BuildValue("ss#O",
		       (self->ob_type)->tp_name,
		       self->a.description,
		       len1,
		       Attribute_Repr((PyObject *)self));

  if (args == NULL)
    return NULL;

  result = PyString_Format(format, args);
  Py_DECREF(args);
  return result;
}


/*******************************************************
 * Declare the type
 ******************************************************/


PyTypeObject TodoType = {
  PyObject_HEAD_INIT(NULL)
  0,
  "jppy._jpilot.__jpilot.Todo",
  sizeof(PyPiTodo),
  0,
  (destructor)PyPiTodo_Dealloc,      /*tp_dealloc*/
  0,                                /*tp_print*/
  0, /*tp_getattr*/
  0, /*tp_setattr*/
  (cmpfunc)PyPiTodo_Compare,     /*tp_compare*/
  (reprfunc)PyPiTodo_Repr,       /*tp_repr*/
  0,                                /*tp_as_number*/
  0,                                /*tp_as_sequence*/
  &PyPiTodo_Mapping,                                /*tp_as_mapping*/
  0,                                /*tp_hash */
  0,                         /*tp_call*/
  0,                         /*tp_str*/
  0,                         /*tp_getattro*/
  0,                         /*tp_setattro*/
  0,                         /*tp_as_buffer*/
  Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /*tp_flags*/
  "Todo objects",           /* tp_doc */
  0,		               /* tp_traverse */
  0,		               /* tp_clear */
  0,		               /* tp_richcompare */
  0,		               /* tp_weaklistoffset */
  0,		               /* tp_iter */
  0,		               /* tp_iternext */
  PyPiTodo_Methods,             /* tp_methods */
  PyPiTodo_Members,            /* tp_members */
  PyPiTodo_Getseters,          /* tp_getset */
  0,                         /* tp_base */
  0,                         /* tp_dict */
  0,                         /* tp_descr_get */
  0,                         /* tp_descr_set */
  0,                         /* tp_dictoffset */
  (initproc)PyPiTodo_Init,      /* tp_init */
  (allocfunc)PyPiTodo_Allocate,                 /* tp_alloc */
  (newfunc)PyPiTodo_New,                 /* tp_new */
  0, /* Low-level free-memory routine */
};
