#ifndef K3DUI_PROPERTY_BUTTON_H
#define K3DUI_PROPERTY_BUTTON_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares property_button, which provides a standard MVC UI for linking properties to channels
		\author Tim Shead (tshead@k-3d.com)
*/

#include <k3dsdk/idag.h>

#include "dynamic_menu.h"
#include "k3dcontrol.h"

#include <sdpgtk/sdpgtkloopevents.h>

// Forward declarations
class sdpGtkIObjectContainer;
class sdpxml::Element;

namespace k3d
{

// Forward declarations
class idocument;
class iobject;
class iplugin_factory;
class iproperty;
class istate_recorder;

namespace property_button
{

/////////////////////////////////////////////////////////////////////////////
// idata_proxy

/// Abstract interface for an object that proxies a data source for a property_button (i.e. the "model" in model-view-controller)
class idata_proxy
{
public:
	virtual ~idata_proxy() {}

	/// Called to return the document containing the underlying data property
	virtual idocument& document() = 0;
	/// Called to return the underlying data property
	virtual iproperty& property() = 0;

protected:
	idata_proxy() {}
	idata_proxy(const idata_proxy& RHS) {}
	idata_proxy& operator=(const idata_proxy& RHS) { return *this; }
};

/////////////////////////////////////////////////////////////////////////////
// control

/// Provides a UI for manipulating boolean quantities (i.e. the view and the controller from model-view-controller)
class control :
	public k3dControl
{
public:
	control(iunknown* const CommandNodeParent, const std::string CommandNodeName);
	~control();

	/// Called by the framework when instantiating the button from a GTKML (XML) document
	bool Create(sdpGtkIObjectContainer* const ObjectContainer, sdpxml::Document& Document, sdpxml::Element& Element);

	/// Attaches the button to the data it fronts for
	bool attach(std::auto_ptr<idata_proxy> Data, istate_recorder* const StateRecorder, const std::string StateChangeName);
	const std::string CustomType() const;
	bool execute_command(const std::string& Command, const std::string& Arguments);

private:
	typedef k3dControl base;

	/// Called whenever the the button state needs to be updated
	void update();
	/// Called by the signal system whenever property dependencies change
	void on_dependencies_changed(const idag::dependencies_t& Dependencies);

	/// Called to handle SDPGTK events
	void OnEvent(sdpGtkEvent* Event);
	/// Called when the GTK+ widgets are about to disappear
	void on_destroy();
	/// Called when the user clicks the button
	void on_clicked();

	// Context menu event handlers
	void on_create_object_set_connection(iplugin_factory* Factory);
	void on_set_connection();
	void on_edit_connected(iobject* Object);
	void on_remove_connection();
	void on_delete_connected(iobject* Object);

	// Popup context menus
	dynamic_menu::control<> m_context_menu;
	dynamic_menu::control<> m_context_new_object_menu;

	/// Storeas a reference to the underlying data object
	std::auto_ptr<idata_proxy> m_data;
};

/// Provides an implementation of property_button::idata_proxy that supports any data source that supports the property() concept
template<typename data_t>
class data_proxy :
	public idata_proxy
{
public:
	data_proxy(idocument& Document, data_t& Data) :
		m_document(Document),
		m_data(Data)
	{
	}

	idocument& document()
	{
		return m_document;
	}

	iproperty& property()
	{
		return m_data.property();
	}

private:
	data_proxy(const data_proxy& RHS);
	data_proxy& operator=(const data_proxy& RHS);
	~data_proxy() {}

	idocument& m_document;
	data_t& m_data;
};

/// Specialization of property_button::data_proxy for use directly with iproperty objects
template<>
class data_proxy<iproperty> :
	public idata_proxy
{
public:
	typedef iproperty data_t;
	
	data_proxy(idocument& Document, data_t& Data) :
		m_document(Document),
		m_data(Data)
	{
	}

	idocument& document()
	{
		return m_document;
	}

	iproperty& property()
	{
		return m_data;
	}

private:
	data_proxy(const data_proxy& RHS);
	data_proxy& operator=(const data_proxy& RHS);
	~data_proxy() {}

	idocument& m_document;
	data_t& m_data;
};

/// Convenience factory function for creating property_button::idata_proxy objects
template<typename data_t>
std::auto_ptr<idata_proxy> proxy(idocument& Document, data_t& Data)
{
	return std::auto_ptr<idata_proxy>(new data_proxy<data_t>(Document, Data));
}

} // namespace property_button

} // namespace k3d

#endif // !K3DUI_PROPERTY_BUTTON_H


