// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Romain Behar (romainbehar@yahoo.com)
*/

#include <k3dsdk/imaterial.h>
#include <k3dsdk/object.h>
#include <k3dsdk/persistence.h>
#include <k3dsdk/material.h>
#include <k3dsdk/material_collection.h>
#include <k3dsdk/measurement.h>
#include <k3dsdk/mesh_source.h>
#include <k3dsdk/module.h>
#include <k3dsdk/plugins.h>
#include <k3dsdk/transform.h>

namespace libk3dnurbs
{

/////////////////////////////////////////////////////////////////////////////
// nurbs_polygon_implementation

class nurbs_polygon_implementation :
	public k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > >
{
	typedef k3d::material_collection<k3d::mesh_source<k3d::persistent<k3d::object> > > base;

public:
	nurbs_polygon_implementation(k3d::idocument& Document) :
		base(Document),
		m_u_segments(k3d::init_name("u_segments") + k3d::init_description("Sides [integer]") + k3d::init_value(4) + k3d::init_constraint(k3d::constraint::minimum(1UL)) + k3d::init_precision(0) + k3d::init_step_increment(1) + k3d::init_units(typeid(k3d::measurement::scalar)) + k3d::init_document(Document)),
		m_radius(k3d::init_name("radius") + k3d::init_description("Radius [number]") + k3d::init_value(5.0) + k3d::init_precision(2) + k3d::init_step_increment(0.1) + k3d::init_units(typeid(k3d::measurement::distance)) + k3d::init_document(Document))
	{
		enable_serialization(k3d::persistence::proxy(m_u_segments));
		enable_serialization(k3d::persistence::proxy(m_radius));

		register_property(m_u_segments);
		register_property(m_radius);

		m_material.changed_signal().connect(SigC::slot(*this, &nurbs_polygon_implementation::on_reset_geometry));

		m_u_segments.changed_signal().connect(SigC::slot(*this, &nurbs_polygon_implementation::on_reset_geometry));
		m_radius.changed_signal().connect(SigC::slot(*this, &nurbs_polygon_implementation::on_reset_geometry));

		m_output_mesh.need_data_signal().connect(SigC::slot(*this, &nurbs_polygon_implementation::on_create_geometry));
	}

	void on_reset_geometry()
	{
		m_output_mesh.reset();
	}

	k3d::mesh* on_create_geometry()
	{
		k3d::mesh* const mesh = new k3d::mesh();
		k3d::nucurve_group* const nucurve_group = new k3d::nucurve_group();
		k3d::nucurve* const nucurve = new k3d::nucurve();

		mesh->nucurve_groups.push_back(nucurve_group);
		nucurve_group->curves.push_back(nucurve);
		nucurve_group->material = m_material.interface();

		// Create a NURBS polygon ...
		nucurve->order = 2;

		// Get side number ...
		const unsigned long u_segments = m_u_segments.property_value();

		// Build knot vector ...
		nucurve->knots.push_back(0);
		for(unsigned long n = 0; n <= u_segments; ++n)
			nucurve->knots.push_back(n);
		nucurve->knots.push_back(u_segments);

		// Build control vertices ...
		const double radius = m_radius.property_value();

		for(unsigned long n = 0; n != u_segments; ++n)
			{
				const double angle = k3d::pi_over_2() + k3d::pi_times_2() / static_cast<double>(u_segments) * static_cast<double>(n);

				k3d::point* const new_point = new k3d::point(radius * cos(angle), radius * sin(angle), 0.0);
				mesh->points.push_back(new_point);
				nucurve->control_points.push_back(k3d::nucurve::control_point(new_point));
			}
		nucurve->control_points.push_back(k3d::nucurve::control_point(mesh->points.front()));

		// Do some sanity-checking on ourselves ...
		assert_warning(is_valid(*nucurve));

		return mesh;
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<k3d::document_plugin<nurbs_polygon_implementation>, k3d::interface_list<k3d::imesh_source > > factory(
		k3d::uuid(0x49796f52, 0x2e29f346, 0xbd47e3c9, 0xfb899a81),
			"NurbsPolygon",
			"Generates a NURBS polygon",
			"Objects",
			k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_measurement_property(unsigned long, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::with_constraint) m_u_segments;
	k3d_measurement_property(double, k3d::immutable_name, k3d::change_signal, k3d::with_undo, k3d::local_storage, k3d::no_constraint) m_radius;
};

/////////////////////////////////////////////////////////////////////////////
// nurbs_polygon_factory

k3d::iplugin_factory& nurbs_polygon_factory()
{
	return nurbs_polygon_implementation::get_factory();
}

} // namespace libk3dnurbs


