// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "imesh_source.h"
#include "mesh.h"
#include "persistent_lookup.h"
#include "selection.h"

namespace k3d
{

namespace selection
{
	
namespace detail
{

persistent_lookup node_lookup;
	
} // namespace detail

std::ostream& operator<<(std::ostream& Stream, const type& RHS)
{
	switch(RHS)
	{
		case NONE:
			Stream << "none";
			break;
		case NODE:
			Stream << "node";
			break;
		case MESH:
			Stream << "mesh";
			break;
		case POINT_GROUP:
			Stream << "point_group";
			break;
		case POINT:
			Stream << "point";
			break;
		case ABSOLUTE_POINT:
			Stream << "absolute_point";
			break;
		case POLYHEDRON:
			Stream << "polyhedron";
			break;
		case FACE:
			Stream << "face";
			break;
		case ABSOLUTE_FACE:
			Stream << "absolute_face";
			break;
		case FACE_HOLE:
			Stream << "face_hole";
			break;
		case SPLIT_EDGE:
			Stream << "split_edge";
			break;
		case ABSOLUTE_SPLIT_EDGE:
			Stream << "absolute_split_edge";
			break;
		case LINEAR_CURVE_GROUP:
			Stream << "linear_curve_group";
			break;
		case LINEAR_CURVE:
			Stream << "linear_curve";
			break;
		case ABSOLUTE_LINEAR_CURVE:
			Stream << "absolute_linear_curve";
			break;
		case CUBIC_CURVE_GROUP:
			Stream << "cubic_curve_group";
			break;
		case CUBIC_CURVE:
			Stream << "cubic_curve";
			break;
		case ABSOLUTE_CUBIC_CURVE:
			Stream << "absolute_cubic_curve";
			break;
		case NUCURVE_GROUP:
			Stream << "nucurve_group";
			break;
		case NUCURVE:
			Stream << "nucurve";
			break;
		case ABSOLUTE_NUCURVE:
			Stream << "absolute_nucurve";
			break;
		case BILINEAR_PATCH:
			Stream << "bilinear_patch";
			break;
		case BICUBIC_PATCH:
			Stream << "bicubic_patch";
			break;
		case NUPATCH:
			Stream << "nupatch";
			break;
		case USER1:
			Stream << "user1";
			break;
	}

	return Stream;
}

std::istream& operator>>(std::istream& Stream, type& RHS)
{
	std::string buffer;
	Stream >> buffer;

	if(buffer == "none")
		RHS = NONE;
	else if(buffer == "node")
		RHS = NODE;
	else if(buffer == "mesh")
		RHS = MESH;
	else if(buffer == "point_group")
		RHS = POINT_GROUP;
	else if(buffer == "point")
		RHS = POINT;
	else if(buffer == "absolute_point")
		RHS = ABSOLUTE_POINT;
	else if(buffer == "polyhedron")
		RHS = POLYHEDRON;
	else if(buffer == "face")
		RHS = FACE;
	else if(buffer == "absolute_face")
		RHS = ABSOLUTE_FACE;
	else if(buffer == "face_hole")
		RHS = FACE_HOLE;
	else if(buffer == "split_edge")
		RHS = SPLIT_EDGE;
	else if(buffer == "absolute_split_edge")
		RHS = ABSOLUTE_SPLIT_EDGE;
	else if(buffer == "linear_curve_group")
		RHS = LINEAR_CURVE_GROUP;
	else if(buffer == "linear_curve")
		RHS = LINEAR_CURVE;
	else if(buffer == "absolute_linear_curve")
		RHS = ABSOLUTE_LINEAR_CURVE;
	else if(buffer == "cubic_curve_group")
		RHS = CUBIC_CURVE_GROUP;
	else if(buffer == "cubic_curve")
		RHS = CUBIC_CURVE;
	else if(buffer == "absolute_cubic_curve")
		RHS = ABSOLUTE_CUBIC_CURVE;
	else if(buffer == "nucurve_group")
		RHS = NUCURVE_GROUP;
	else if(buffer == "nucurve")
		RHS = NUCURVE;
	else if(buffer == "absolute_nucurve")
		RHS = ABSOLUTE_NUCURVE;
	else if(buffer == "bilinear_patch")
		RHS = BILINEAR_PATCH;
	else if(buffer == "bicubic_patch")
		RHS = BICUBIC_PATCH;
	else if(buffer == "nupatch")
		RHS = NUPATCH;
	else if(buffer == "user1")
		RHS = USER1;
	else
		log() << error << k3d_file_reference << ": could not extract value [" << buffer << "]" << std::endl;

	return Stream;
}

const id null_id()
{
	return static_cast<id>(-1);
}

token::token() :
	type(NONE),
	id(null_id())
{
}

/////////////////////////////////////////////////////////////////
// token

token::token(const selection::type Type, const selection::id ID) :
	type(Type),
	id(ID)
{
}

bool operator==(const token& LHS, const token& RHS)
{
	return LHS.type == RHS.type && LHS.id == RHS.id;
}

bool operator!=(const token& LHS, const token& RHS)
{
	return !(LHS == RHS);
}

std::ostream& operator<<(std::ostream& Stream, const token& RHS)
{
	Stream << RHS.type << " " << RHS.id;
	return Stream;
}

std::istream& operator>>(std::istream& Stream, token& RHS)
{
	Stream >> RHS.type >> RHS.id;
	return Stream;
}

/////////////////////////////////////////////////////////////////
// record

record::record() :
	zmin(0),
	zmax(0)
{
}

const record record::empty_record()
{
	return record();
}

const bool record::empty() const
{
	return tokens.empty();
}

const id record::get_id(const type Type) const
{
	for(tokens_t::const_iterator token = tokens.begin(); token != tokens.end(); ++token)
	{
		if(token->type == Type)
			return token->id;
	}

	return null_id();
}

const token record::get_token(const type Type) const
{
	for(tokens_t::const_iterator token = tokens.begin(); token != tokens.end(); ++token)
	{
		if(token->type == Type)
			return *token;
	}

	return token();
}

std::ostream& operator<<(std::ostream& Stream, const record& RHS)
{
	Stream << RHS.zmin << " " << RHS.zmax << " " << RHS.tokens.size() << " ";
	std::copy(RHS.tokens.begin(), RHS.tokens.end(), std::ostream_iterator<token>(Stream, " "));

	return Stream;
}

std::istream& operator>>(std::istream& Stream, record& RHS)
{
	size_t count = 0;
	
	Stream >> RHS.zmin >> RHS.zmax >> count;

	selection::token token;
	while(Stream && count)
	{
		Stream >> token;
		RHS.tokens.push_back(token);
		--count;
	}
	
	return Stream;
}

//////////////////////////////////////////////////////////////////////////////////
// node_id

const id node_id(inode* Node)
{
	return detail::node_lookup.lookup_id(Node);
}

//////////////////////////////////////////////////////////////////////////////////
// make_record

const record make_record(inode* Node)
{
	record result;
	result.zmin = 0;
	result.zmax = 0;

	result.tokens.push_back(token(NODE, node_id(Node)));

	return result;
}

///////////////////////////////////////////////////////////////////////////////////
// make_records

const records make_records(inode* Node)
{
	return records(1, make_record(Node));
}

///////////////////////////////////////////////////////////////////////////////////
// get_node

inode* get_node(const record& Record)
{
	const selection::id id = Record.get_id(NODE);
	if(id != null_id())
		return dynamic_cast<inode*>(k3d::selection::detail::node_lookup.lookup_object(id));

	return 0;
}

///////////////////////////////////////////////////////////////////////////////////
// get_mesh

mesh* get_mesh(const record& Record)
{
	inode* const node = get_node(Record);
	return_val_if_fail(node, 0);

	const selection::id id = Record.get_id(MESH);
	return_val_if_fail(id == 0, 0); // Should never get a node with more than one mesh!

	imesh_source* const mesh_source = dynamic_cast<imesh_source*>(node);
	return_val_if_fail(mesh_source, 0);

	return boost::any_cast<k3d::mesh*>(mesh_source->mesh_source_output().property_value());
}

///////////////////////////////////////////////////////////////////////////////////
// get_point_group

point_group* get_point_group(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(POINT_GROUP);
	if(id == null_id())
		return 0;
	return_val_if_fail(id < Mesh.point_groups.size(), 0);

	return Mesh.point_groups[id];
}

///////////////////////////////////////////////////////////////////////////////////
// get_point

point* get_point(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(ABSOLUTE_POINT);
	if(id == null_id())
		return 0;
	return_val_if_fail(id < Mesh.points.size(), 0);

	return Mesh.points[id];
}

///////////////////////////////////////////////////////////////////////////////////
// get_polyhedron

polyhedron* get_polyhedron(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(POLYHEDRON);
	if(id == null_id())
		return 0;
	return_val_if_fail(id < Mesh.polyhedra.size(), 0);

	return Mesh.polyhedra[id];
}

///////////////////////////////////////////////////////////////////////////////////
// get_face

face* get_face(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(FACE);
	if(id == null_id())
		return 0;

	k3d::polyhedron* const polyhedron = get_polyhedron(Mesh, Record);
	if(!polyhedron)
		return 0;
	return_val_if_fail(id < polyhedron->faces.size(), 0);

	return polyhedron->faces[id];
}

///////////////////////////////////////////////////////////////////////////////////
// get_split_edge

split_edge* get_split_edge(mesh& Mesh, const record& Record)
{
	const selection::id edge_id = Record.get_id(SPLIT_EDGE);
	if(edge_id == null_id())
		return 0;

	k3d::face* const face = get_face(Mesh, Record);
	if(!face)
		return 0;
	
	const selection::id hole_id = Record.get_id(FACE_HOLE);
	if(hole_id == null_id())
	{
		selection::id id = 0;
		for(split_edge* edge = face->first_edge; edge; edge = edge->face_clockwise)
		{
			if(edge_id == id++)
				return edge;

			if(edge->face_clockwise == face->first_edge)
				break;
		}
	}
	else
	{
		return_val_if_fail(hole_id < face->holes.size(), 0);
		
		selection::id id = 0;
		for(split_edge* edge = face->holes[hole_id]; edge; edge = edge->face_clockwise)
		{
			if(edge_id == id++)
				return edge;

			if(edge->face_clockwise == face->holes[hole_id])
				break;
		}
	}
	
	return 0;
}

///////////////////////////////////////////////////////////////////////////////////
// get_linear_curve_group

linear_curve_group* get_linear_curve_group(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(LINEAR_CURVE_GROUP);
	if(id == null_id())
		return 0;
	return_val_if_fail(id < Mesh.linear_curve_groups.size(), 0);

	return Mesh.linear_curve_groups[id];
}

///////////////////////////////////////////////////////////////////////////////////
// get_linear_curve

linear_curve* get_linear_curve(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(LINEAR_CURVE);
	if(id == null_id())
		return 0;

	linear_curve_group* const group = get_linear_curve_group(Mesh, Record);
	if(!group)
		return 0;
	return_val_if_fail(id < group->curves.size(), 0);
	
	return group->curves[id];
}

///////////////////////////////////////////////////////////////////////////////////
// get_cubic_curve_group

cubic_curve_group* get_cubic_curve_group(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(CUBIC_CURVE_GROUP);
	if(id == null_id())
		return 0;
	return_val_if_fail(id < Mesh.cubic_curve_groups.size(), 0);

	return Mesh.cubic_curve_groups[id];
}

///////////////////////////////////////////////////////////////////////////////////
// get_cubic_curve

cubic_curve* get_cubic_curve(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(CUBIC_CURVE);
	if(id == null_id())
		return 0;

	cubic_curve_group* const group = get_cubic_curve_group(Mesh, Record);
	if(!group)
		return 0;
	return_val_if_fail(id < group->curves.size(), 0);
	
	return group->curves[id];
}

///////////////////////////////////////////////////////////////////////////////////
// get_nucurve_group

nucurve_group* get_nucurve_group(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(NUCURVE_GROUP);
	if(id == null_id())
		return 0;

	return_val_if_fail(id < Mesh.nucurve_groups.size(), 0);
	return Mesh.nucurve_groups[id];
}

///////////////////////////////////////////////////////////////////////////////////
// get_nucurve

nucurve* get_nucurve(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(NUCURVE);
	if(id == null_id())
		return 0;

	nucurve_group* const group = get_nucurve_group(Mesh, Record);
	if(!group)
		return 0;
	return_val_if_fail(id < group->curves.size(), 0);
	
	return group->curves[id];
}

///////////////////////////////////////////////////////////////////////////////////
// get_bilinear_patch

bilinear_patch* get_bilinear_patch(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(BILINEAR_PATCH);
	if(id == null_id())
		return 0;
	return_val_if_fail(id < Mesh.bilinear_patches.size(), 0);

	return Mesh.bilinear_patches[id];
}

///////////////////////////////////////////////////////////////////////////////////
// get_bicubic_patch

bicubic_patch* get_bicubic_patch(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(BICUBIC_PATCH);
	if(id == null_id())
		return 0;
	return_val_if_fail(id < Mesh.bicubic_patches.size(), 0);

	return Mesh.bicubic_patches[id];
}

///////////////////////////////////////////////////////////////////////////////////
// get_nupatch

nupatch* get_nupatch(mesh& Mesh, const record& Record)
{
	const selection::id id = Record.get_id(NUPATCH);
	if(id == null_id())
		return 0;
	return_val_if_fail(id < Mesh.nupatches.size(), 0);

	return Mesh.nupatches[id];
}

///////////////////////////////////////////////////////////////////////////////////
// select

void select(inode* Node)
{
	if(iselectable* const selectable = dynamic_cast<iselectable*>(Node))
		selectable->set_selection_weight(1.0);
}

///////////////////////////////////////////////////////////////////////////////////
// deselect

void deselect(inode* Node)
{
	if(iselectable* const selectable = dynamic_cast<iselectable*>(Node))
		selectable->set_selection_weight(0.0);
}

} // namespace selection

namespace gl
{

///////////////////////////////////////////////////////////////////////////////////
// push_selection_token

void push_selection_token(inode* Node)
{
	::glPushName(selection::NODE);
	::glPushName(k3d::selection::detail::node_lookup.lookup_id(Node));
}

///////////////////////////////////////////////////////////////////////////////////
// push_selection_token

void push_selection_token(const selection::token& Token)
{
	::glPushName(Token.type);
	::glPushName(Token.id);
}

///////////////////////////////////////////////////////////////////////////////////
// push_selection_token

void push_selection_token(const selection::type Type, const selection::id ID)
{
	::glPushName(Type);
	::glPushName(ID);
}

///////////////////////////////////////////////////////////////////////////////////
// pop_selection_token

void pop_selection_token()
{
	::glPopName();
	::glPopName();
}
	
} // namespace gl
	
} // namespace k3d

