#ifndef K3D_SDS_BINDING
#define K3D_SDS_BINDING

// K-3D SDS preview
// Copyright (c) 2005, Bart Janssens
//
// Contact: bart.janssens@lid.kviv.be
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/**	\file K-3D interface for Catmull-Clark algorithm
		\author Bart Janssens <bart.janssens@lid.kviv.be>
 */

#include "subdivision_algorithms.h"

namespace k3d
{

namespace sds
{

/// The size of a segment. There is one segment per OpenGL display list, so the optimum size may vary across systems.
const unsigned long segment_size = 100;

typedef std::set<unsigned long> segments_t;

/// convert k3d::mesh to the internal sds cache structure. This one does a lot of work, it should be looked into if part of this can be moved to the generic SDS part. (doubtful, since it deals heavily with k3d::edge and k3d::face)
class k3d_cache_input : public cache_input<k3d::mesh>
{
public:
	typedef std::vector<facevertices_t*> faces_per_polyhedron_t;
	
	k3d_cache_input(const k3d::mesh* Input);
	~k3d_cache_input();

	//////
	// cache_input<k3d::mesh> implementation
	//////
	bool set_input(const k3d::mesh* Input);

	void update(bool all, facevertices_t& updated_maps);

	//////////
	// Extra K3D specific methods
	//////////

	/// Get the input mesh
	const k3d::mesh& mesh()
	{
		return *m_input_mesh;
	}
	
	/// Get the number of segments
	inline unsigned long get_n_segments()
	{
		return m_n_segments;
	}

	/// Get the segments that were modified by the last update
	segments_t& get_modified_segments()
	{
		return m_modified_segments;
	}

	/// Get the faces per polyhedron
	const faces_per_polyhedron_t& faces_per_polyhedron() {
		return m_faces_per_polyhedron;
	}

private:
	const k3d::mesh* m_input_mesh;

	typedef std::map<k3d::split_edge*, edge_vertex*> edgemap_t;
	typedef std::map<k3d::point*, sds_point*> pointmap_t;
	edgemap_t m_edge_vertices;
	pointmap_t m_sds_vertices;
	faces_per_polyhedron_t m_faces_per_polyhedron;
	unsigned long m_n_segments;
	segments_t m_modified_segments;
	unsigned long m_npoints;
	unsigned long m_npolyhedra;
	std::vector<unsigned long> m_nfaces;
	// track selected edges in case only faces were selected
	std::set<k3d::split_edge*> m_selected_edges;
	
	// get or create the edge vertex belonging to Edge
	edge_vertex* get_ev(k3d::split_edge* Edge, face_vertex* Fv);
	// get the corner associated with the given point
	sds_point* get_corner(k3d::point* Point);
	// True if the given face contains selected elements
	bool selected(k3d::split_edge* first_edge, bool recurse = true);
	// Get the segment of a mesh to which the face vertex with index i belongs.
	inline unsigned long get_segment(unsigned long i)
	{
		return (i - (i % segment_size)) / segment_size;
	}
	// initialise point, face and plohedra counters
	void init_counters()
	{
		return_if_fail(m_input_mesh);
		m_npoints = m_input_mesh->points.size();
		m_npolyhedra = m_input_mesh->polyhedra.size();
		m_nfaces.clear();
		for (unsigned long i = 0; i < m_npolyhedra; ++i)
			m_nfaces.push_back(m_input_mesh->polyhedra[i]->faces.size());
	}
};

/// Base class for the caches used by k3d, handling only the input side of things.
class k3d_sds_cache_base : public catmull_clark_cache<k3d::mesh>
{
protected:
	////////////
	// catmull_clark_cache<k3d::mesh> implementation
	////////////
	cache_input<k3d::mesh>* create_cache_input(const k3d::mesh* Input)
	{
		return new k3d_cache_input(Input);
	}

	
};

/// SDS cache that outputs to OpenGL
class k3d_opengl_sds_cache : public k3d_sds_cache_base
{
public:

	k3d_opengl_sds_cache() : m_nurbs_renderer(0), m_display_lists(0), m_render_state(0) {}

	~k3d_opengl_sds_cache()
	{
		if (m_nurbs_renderer)
			gluDeleteNurbsRenderer(m_nurbs_renderer);

		if (m_display_lists)
		{
			for (unsigned long i = 0; i < dynamic_cast<k3d_cache_input*>(m_first_level_cache)->get_n_segments(); ++i)
				glDeleteLists(m_display_lists[i], 1);
	
			delete[] m_display_lists;
		}
	}
	
	/// (k-3d specific) draw patch borders
	void draw_borders(bool draw)
	{
		m_draw_borders = draw;
	}

	/// Set render state
	void set_render_state(const k3d::gl::render_state& State)
	{
		m_render_state = &State;
	}
	
protected:
	////////
	// catmull_clark_cache<k3d::mesh> implementation
	///////
	void client_output(k3d::mesh* Output = 0);

	void client_output_nurbs(k3d::mesh* Output = 0);

private:
	GLUnurbsObj* m_nurbs_renderer;
	GLuint* m_display_lists;
	bool m_draw_borders;
	const k3d::gl::render_state* m_render_state;

	// initialise display lists
	void init_lists()
	{
		if (m_display_lists != 0)
			return;
		m_display_lists = new GLuint[dynamic_cast<k3d_cache_input*>(m_first_level_cache)->get_n_segments()];
		for (unsigned long i = 0; i < dynamic_cast<k3d_cache_input*>(m_first_level_cache)->get_n_segments(); ++i)
		{
			m_display_lists[i] = glGenLists(1);
		}
	}
	// setup opengl renderer
	void gl_setup();
	// draw faces of level 1
	void draw_faces_first_level(unsigned long start, unsigned long end);
	// draw borders of level 1
	void draw_borders_first_level(unsigned long start, unsigned long end);
	// draw faces of higher levels
	void draw_faces_higher_level(unsigned long start, unsigned long end, unsigned long Level);
	// draw borders of higher levels
	void draw_borders_higher_level(unsigned long start, unsigned long end, unsigned long Level);
	// draw nurbs patches
	void draw_nurbs_patches(unsigned long start, unsigned long end, unsigned long Level);
	// draw nurbs borders
	void draw_nurbs_borders(unsigned long start, unsigned long end, unsigned long Level);
};

/// Output to a k3d::mesh
class k3d_mesh_sds_cache : public k3d_sds_cache_base
{
public:
	k3d_mesh_sds_cache()
	{
		set_levels(1);
	}
protected:
	////////
	// catmull_clark_cache<k3d::mesh> implementation
	///////
	/** \todo make efficient */
	void client_output(k3d::mesh* Output = 0);

	/** \todo implement this */
	void client_output_nurbs(k3d::mesh* Output = 0) {}
};

} // namespace sds

} // namespace k3d

#endif // #ifndef K3D_SDS_BINDING
