#include <string>
#include <vector>
#if (__GNUC__ < 3)
#include <hash_map>
#else
#include <ext/hash_map>
#endif
using namespace std;

#include <qfiledialog.h>
#include <qinputdialog.h>
#include <qapplication.h>
#include <qmessagebox.h>
#include <qclipboard.h>
#include <qtimer.h>

#include "category.h"
#include "currents.h"
#include "globals.h"
#include "browser.h"
#include "categorylabel.h"
#include "navigation.h"
#include "chatbox.h"
#include "sourcebox.h"
#include "mailbox.h"
#include "preferences.h"
#include "utility.h"
#include "errors.h"
#include "keyboard.h"
#include "parser.h"
#include "separator.h"
#include "bookmarks.h"
#include "mailinglists.h"
#include "choicebox.h"
#include "infobox.h"
#include "addressing.h"
#include "files.h"
#include "tracing.h"

Browser::Browser() : QMainWindow()
{ 
	initDialog();
	
	disabled = TRUE;
	keyboardaction = FALSE;
}

void Browser::showPreferences()
{
	config->show();
}

void Browser::showMailingLists()
{
	mailinglists->update();
}

void Browser::showBookmarks()
{
	bookmarks->update();	
}

void Browser::openFile()
{
	if( !disabled )
		update( action_openfile() );
}

void Browser::openURL()
{
	if( !disabled )
		update( action_openurl() );
}

void Browser::bookmarkSelected( int nr ) 
{
	if( !disabled )
		for( unsigned int n = 3; ; n++ )
			if( bmmenu->idAt( n ) == nr )
			{
				update( action_jump( config->bookmarkv[2*(n-3)+1] ) );
				break;
			}
}

void Browser::enterCategory( Category *category )
{
	if( disabled )
		return;

	disabled = TRUE;
	if( update( action_enter( category ) ) )
		highlight = -1;
	disabled = FALSE;
}

void Browser::backPressed()
{
	if( !disabled )
		action_backtrace();
}

void Browser::discussPressed()
{
	if( !disabled )
		tryToSelect( cCategory->discussv, "Select Newsgroup", SLOT( choseDiscuss( string ) ) );
}

void Browser::choseDiscuss( string newsgroup )
{
	if( !disabled )
		launchExtension( config->newsClient(), newsgroup );
}

void Browser::linksPressed()
{
	if( !disabled )
		action_showlinks();
}

void Browser::choseLink( string link )
{
	if( !disabled )
		launchExtension( link );
}

void Browser::chatPressed()
{
	if( !disabled )
		action_chat();
}

void Browser::action_chat()
{
	if( config->ircServer() == "" )
	{
		QMessageBox::warning( this, "Error",
				      "Please enter IRC server" );

		config->show();
		return;
	}
	
	Category *c = cCategory->calcChat();

	if( c->chatv )
	{	
		createVectors( c->chatv );
		if( namev.size() != 1 )
		{
			ChoiceBox *choiceBox = new ChoiceBox( namev, memoryv, descv, "Select Channel", this );
			QObject::connect( choiceBox, SIGNAL( choiceMade( string ) ), this, SLOT( choseChannel( string ) ) ); 
		}
		else
			choseChannel( memoryv[0] );
	}	
	else	
	{
		string path = fullPath( aPath, c );
        	hash<const char *> hashstring;
        	string channel = "#" + itos( hashstring( path.c_str() ) );
		
		string title = c->calcTitle();
		if( title.find( ";" ) != -1 )
			title = trim( title.substr( 0, title.find( ";" ) ) );

		chatBox->join( title, channel );
	}
}

void Browser::choseChannel( string channel )
{
	if( !disabled )
	{
		if( config->ircClient() != "" )
			launchExtension( config->ircClient(), channel );
		else	
			chatBox->join( channel );
	}	
}

void Browser::mlistPressed()
{
	if( !disabled &&
 	    cCategory->mlistv )
	{
		createVectors( cCategory->mlistv );
		ChoiceBox *choiceBox = new ChoiceBox( namev, memoryv, descv, "Select Mailing List", this );
		QObject::connect( choiceBox, SIGNAL( choiceMade( string, string ) ), this, SLOT( choseMlist( string, string ) ) );
	}	
}

void Browser::choseMlist( string name, string memory )
{
	if( !disabled )
	{
		mailinglists->addMlist( name, memory );
		mailinglists->update();
	}	
}

void Browser::faqPressed()
{
	if( !disabled )
		tryToSelect( cCategory->faqv, "Select FAQ", SLOT( choseFaq( string ) ) );
}

void Browser::choseFaq( string faq )
{
	if( !disabled )
		launchExtension( faq );	
}

void Browser::stopPressed()
{
	emit stopSignal();
}

void Browser::basePressed()
{
	if( !disabled )
		tryToSelect( cCategory->basev, "Select Internet Base", SLOT( choseBase( string ) ) );
}

void Browser::choseBase( string base )
{
	if( !disabled )
		launchExtension( base );
}

void Browser::markPressed()
{
	if( !disabled )
	{
		markaddress = fullPath( cPath, cCategory );

		if( markcurrents )
			delete markcurrents;
		markcurrents = new Currents();
		markcurrents->save();

		movemenu->setItemEnabled( movetomarkid, TRUE );
		TBtomark->setEnabled( TRUE );
	}	
}

void Browser::tomarkPressed()
{
	if( !disabled )
	{
		if( update( action_jump( markaddress ) ) )
			markcurrents->load();
	}
}

void Browser::infoPressed()
{
	if( !disabled &&
	    ( cCategory->info != "" ) )
		action_showinfo();
}

void Browser::newsPressed()
{
	if( !disabled &&
	    ( cCategory->news != "" ) )
	    	action_shownews();
}

void Browser::forwardPressed()
{
	if( !disabled )
		action_forwardtrace();
}

void Browser::rootPressed()
{
	if( !disabled )
	    update( action_jump( config->startAddress() ) );
}

void Browser::showRightMenuExtension( int x, int y, Category *c )
{
	if( disabled )
		return;
	
	rightmenu->setItemEnabled( rightopenid, TRUE );
	rightmenu->setItemEnabled( rightopenwid, TRUE );
	rightmenu->setItemEnabled( rightcopyid, TRUE );
		
	extcat = c; 
	rightmenu->exec( QPoint( x-30, y ), 9 );
}

void Browser::showRightMenu( int x, int y )
{
	if( disabled )
		return;
	
	rightmenu->setItemEnabled( rightopenid, FALSE );
	rightmenu->setItemEnabled( rightopenwid, FALSE );
	rightmenu->setItemEnabled( rightcopyid, FALSE );

	rightmenu->exec( QPoint( x-30, y ), 9 );
}

void Browser::extensionOpen()
{
	if( !disabled )
		enterCategory( extcat );
}

void Browser::extensionOpenWith()
{
	bool ok;
	QString text = QInputDialog::getText( "Open With", "Enter application name:", \
			                              QLineEdit::Normal, s_app, &ok, this, "Open" );
	
	if( ok )
	{
		s_app = text;

		string save = config->webBrowser();
		config->setWebBrowser( (const char *)text );
		enterCategory( extcat );
		config->setWebBrowser( save );
	}	
}

void Browser::copyLinkLocation()
{
	Currents c; c.save();

	loopCatv.clear();

	checkingstructure = TRUE;
	enter( extcat );
	checkingstructure = FALSE;

    	int p_dii;
    	string address = cCategory->target, p_extdir, p_file, p_intdir;
    	parseAddress( address, p_dii, p_extdir, p_file, p_intdir );

	QApplication::clipboard()->setText( p_extdir.c_str() );

	c.load();
}

void Browser::viewPageSource()
{
	if( !disabled )
	{
		if( config->sourceViewer() != "" )
			launchExtension( config->sourceViewer(), slashConcat( cPath, cFile ) );
		else	
			sourceBox->show();
	}	
}

void Browser::sendEmail()
{
	if( disabled )
		return;

	if( ( config->emailAddress() == "" ) ||
	    ( config->smtpServer() == "" ) )
	{
		QMessageBox::warning( this, "Error", "Please enter your email address and/or SMTP server" ); 	
		config->show();
		return;
	}	
		
	if( config->emailClient() != "" )
		launchExtension( config->emailClient(), cCategory->calcEmail() );
	else	
		new MailBox( this );
}

void Browser::gotoHomepage()
{
	if( !disabled )
		launchExtension( cCategory->calcHome() );
}	

void Browser::keyPressEvent( QKeyEvent *event )
{
	if( !disabled )
		keyboard->handleKeyPress( event );
}

void Browser::clearCachePressed()
{
	if( !disabled )
	{
		clearCache();
		showNewCategory();
	}	
}

void Browser::debugWindowPressed()
{
	debugWindow->show();
}

void Browser::addBookmark()
{
	if( disabled )
		return;

	config->bookmarkv.push_back( cCategory->calcTitle() );
	config->bookmarkv.push_back( fullPath( cPath, cCategory ) );

	buildBookmarkMenu();
	bookmarks->save();
}


void Browser::tryToSelect( vector<string> *choicev, string title, string slot )
{
	if( !choicev || !choicev->size() )
		return;
	
	createVectors( choicev );
	ChoiceBox *choiceBox = new ChoiceBox( namev, memoryv, descv, title, this );
	QObject::connect( choiceBox, SIGNAL( choiceMade( string ) ), this, slot.c_str() );
}

int Browser::update( int okay )
{
	if( okay )
	{
		if( !keyboardaction )
			highlight = -1;
		showNewCategory();
	}
	else
		setCatLabelColors();

	return okay;
}

int Browser::action_openfile()
{
	QString file = QFileDialog::getOpenFileName( QString::null, "*.dii" );

	if( ( file != QString::null ) && 
	    action_jump( (const char *) file ) )
		return TRUE;

	return FALSE;
}

int Browser::action_openurl()
{
	bool ok;
	QString text = QInputDialog::getText( "Open URL", "Enter category location:", \
			 								QLineEdit::Normal, s_url, &ok, this, "Open" );

	if( !ok )
		return FALSE;
	s_url = text;

	return action_jump( (const char *)text );
}		

void Browser::action_showlinks()
{
	vector<string> linkv;
	if( cCategory->basev )
		for( unsigned int n = 0; n < cCategory->basev->size(); n++ )
			linkv.push_back( (*(cCategory->basev))[n] );
	if( cCategory->linkv )
		for( unsigned int n = 0; n < cCategory->linkv->size(); n++ )
			linkv.push_back( (*(cCategory->linkv))[n] );
	
	tryToSelect( &linkv, "Select Link", SLOT( choseLink( string ) ) );
}

void Browser::action_showinfo()
{
	if( cCategory->info.find( "http://" ) == 0 )
		launchExtension( cCategory->info );
	else	
		new InfoBox( cCategory->calcTitle(), cCategory->info, this );
}

void Browser::action_shownews()
{
	if( cCategory->news.find( "http://" ) == 0 )
		launchExtension( cCategory->news );
	else	
		new InfoBox( cCategory->calcTitle(), cCategory->news, this );
}

int Browser::action_backtrace()
{
	if( backTrace() )
	{
		showNewCategory();
		return TRUE;
	}	

	return FALSE;
}

int Browser::action_forwardtrace()
{
	if( fwdTrace() )
	{
		showNewCategory();
		return TRUE;
	}	

	return FALSE;
}

void Browser::createVectors( vector<string> *choosev )
{
	string name, memory, desc;

	namev.clear();
	memoryv.clear();
	descv.clear();

	for( unsigned int n = 0; n < choosev->size(); n++ )
	{
		name = (*choosev)[n];
		memory = (*choosev)[n];
		desc = "No description given";

		int i;
		if( ( i = name.find( ";" ) ) != -1 )	
		{
			desc = trim( name.substr( i+1, name.length()-i-1 ) );
			name = trim( name.substr( 0, i ) );
			memory = name;
		}

		if( ( i = name.find( "," ) ) != -1 )
		{
			memory = trim( name.substr( i+1, name.length()-i-1 ) ); 
			name = trim( name.substr( 0, i ) );
		}

		namev.push_back( name );
		memoryv.push_back( memory );
		descv.push_back( desc );
	}	
}		

