/**
    Kaya run-time system
    Copyright (C) 2004, 2005 Edwin Brady

    This file is distributed under the terms of the GNU Lesser General
    Public Licence. See COPYING for licence.
*/

#ifndef WIN32
#include <pthread.h>
#define GC_THREADS
#else
#define GC_WIN32_THREADS
#endif
#include <gc/gc_cpp.h>
#include <setjmp.h>

#include <gc/gc_allocator.h>
#include <assert.h>
#include <stack>
#include <iostream>
#include <map>
#include <cstring>

using namespace std;

#include "VM.h"
#include "VMState.h"

VMState* initstack()
{
#ifdef __arm__
  /* This works around a very wierd behaviour on arm with g++ -O2 where
     this function segfaults (see Debian build logs for 0.2.7) */
  if (getenv("DISPLAY_KAYA_ARM_WIERDNESS") != NULL) {
    if (getenv("DISPLAY_KAYA_ARM_WIERDNESS") == NULL) {
      cout << "Starting" << endl;
    }
  }
#endif
    return new VMState();
}

void finish(VMState* vm)
{
    vm->finish();
}

Value* mkstr(const wchar_t* str)
{
    return MKVAL((void*)(NEWSTRING(str)),KVT_STRING);
}
Value* mkstr2(VMState* vm,const wchar_t* str)
{
    return MKPVAL(vm->getVPool(),(void*)(NEWPSTRING(vm->getSPool(),str)),KVT_STRING);
}

Value* mkint(void* i)
{
    return MKVAL(i,KVT_INT);
}

Value* mkint2(VMState* vm, void* i)
{
    return MKPVAL(vm->getVPool(),i,KVT_INT);
}

Value* mkarrayval(void* i)
{
    return MKVAL(i,KVT_ARRAY);
}

/// Maths operators

kint intpower(kint x, kint y)
{
    return (kint)pow((double)x,(double)y);
}

double realpower(double x, double y)
{
    return pow(x,y);
}

/// String conversions

wchar_t* strtowc(const char* str) {
  int inlen = 0;
  if (str == NULL) {
    inlen = 0;
  } else {
    inlen = strlen(str);
  }
  wchar_t* dest = (wchar_t*)GC_MALLOC_ATOMIC(sizeof(wchar_t)*(inlen+1));  
  mkUCS(str,dest,inlen);
  return dest;
}

void strtovwc(VMState* vm, const char* str) {
  int inlen = 0;
  if (str == NULL) {
    inlen = 0;
  } else {
    inlen = strlen(str);
  }
  wchar_t* dest = (wchar_t*)GC_MALLOC_ATOMIC(sizeof(wchar_t)*(inlen+1));  
  mkUCS(str,dest,inlen);
  vm->strbuffer = dest;
  vm->strbuflen = inlen+1;
}

wchar_t chrtowc(const char chr) {
  return *strtowc(&chr);
}

char* wctostr(const wchar_t* wc) {
  ukint len = wcslen(wc);
  return wcntostr(wc,len);
}

void wcnptostr(const wchar_t* wc, kint len, char* target) {
  mkUTF8(wc,target,len);
}

char* wcntostr(const wchar_t* wc, kint len) {
  char* dest = (char*)GC_MALLOC_ATOMIC(sizeof(char)*(++len)*4);  
  mkUTF8(wc,dest,len);
  return dest;
}

// note that even one wchar_t may be bigger than a char...
char wctochr(const wchar_t c) {
  // result is therefore undefined if wchar_t represents something outside 0-127
  return *wctostr(&c);
}

/* UTF8 mbc constructor based on Jeff Bezanson's public domain code */
void mkUTF8(const wchar_t* raw, char* dest, ukint len) {
  //  char* dest = (char*)GC_MALLOC_ATOMIC(sizeof(char)*(1+wcslen(raw))*4);  
  int dpos = -1;
  for (ukint i=0;i<len;++i) {
    ukint ch = (ukint)raw[i];
    if (ch < 0x80) {
      dest[++dpos] = (char)ch;
    } else if (ch < 0x800) {
      dest[++dpos] = (ch>>6) | 0xC0;
      dest[++dpos] = (ch & 0x3F) | 0x80;
    } else if (ch < 0x10000) {
      dest[++dpos] = (ch>>12) | 0xE0;
      dest[++dpos] = ((ch>>6) & 0x3F) | 0x80;
      dest[++dpos] = (ch & 0x3F) | 0x80;
    } else if (ch < 0x200000) {
      dest[++dpos] = (ch>>18) | 0xF0;
      dest[++dpos] = ((ch>>12) & 0x3F) | 0x80;
      dest[++dpos] = ((ch>>6) & 0x3F) | 0x80;
      dest[++dpos] = (ch & 0x3F) | 0x80;
    }
  }
  dest[++dpos] = '\0';
}

/* UTF8 mbc unconstructor loosely based on Jeff Bezanson's public domain code */
void mkUCS(const char* utf8, wchar_t* dest, kint inlen) {
  int cpos = 0;
  int dpos = -1;

  while (cpos < inlen) {
    unsigned char first = (unsigned char)utf8[cpos];
    //    cout << first << endl;
    if (first < 128) {
      // single byte character
      dest[++dpos] = (wchar_t)first;
      ++cpos;
    } else if (first < 192) {
      // bad continuation character, skip it.
      ++cpos;
    } else if (first < 224 && cpos+1 < inlen) {
      // two-byte character
      dest[++dpos] = (wchar_t)((((int)utf8[cpos] & 31) << 6) + ((int)utf8[cpos+1] & 63));
      cpos+=2;
    } else if (first < 240 && cpos+2 < inlen) {
      // three-byte character
      //      cout << "three-byte = " << first << endl;
      dest[++dpos] = (wchar_t)((((int)utf8[cpos] & 15) << 12) + (((int)utf8[cpos+1] & 63) << 6) + (((int)utf8[cpos+2] & 63)));
      cpos+=3;
    } else if (first < 248 && cpos+3 < inlen) {
      // four-byte character
      dest[++dpos] = (wchar_t)((((int)utf8[cpos] & 7) << 18) + (((int)utf8[cpos+1] & 63) << 12) + (((int)utf8[cpos+2] & 63) << 6) + ((int)utf8[cpos+3] & 63));
      cpos+=4;
    } else {
      // start of illegal 5+-byte character? skip it
      ++cpos;
    }
  }
  dest[++dpos] = '\0';
}
