/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003-2004 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie <ashie@homa.ne.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-bookmark-menu-item.c,v 1.8 2006/02/16 14:47:32 furyo Exp $
 */

#include "kz-bookmark-menu-item.h"

#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include "gobject-utils.h"
#include "kazehakase.h"
#include "kz-icons.h"
#include "kz-actions.h"

static void     kz_bookmark_menu_item_class_init       (KzBookmarkMenuItemClass *klass);
static void     kz_bookmark_menu_item_init             (KzBookmarkMenuItem *menu);

/* object class methods */
static GObject* kz_bookmark_menu_item_constructor      (GType           type,
						        guint           n_props,
						        GObjectConstructParam *props);
static void     kz_bookmark_menu_item_dispose          (GObject        *object);

static gboolean kz_bookmark_menu_item_button_release (GtkWidget *widget, GdkEventButton *event);
static void     kz_bookmark_menu_item_activate       (GtkMenuItem *item);

static KzBookmarkBaseMenuItemClass *parent_class = NULL;


KZ_OBJECT_GET_TYPE(kz_bookmark_menu_item, "KzBookmarkMenuItem", KzBookmarkMenuItem,
		   kz_bookmark_menu_item_class_init, kz_bookmark_menu_item_init,
		   KZ_TYPE_BOOKMARK_BASE_MENU_ITEM)

static void
kz_bookmark_menu_item_class_init (KzBookmarkMenuItemClass *klass)
{
	GObjectClass *gobject_class;
	GtkWidgetClass *widget_class;
	GtkMenuItemClass *menu_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class   = (GObjectClass *) klass;
	widget_class    = (GtkWidgetClass *) klass;
	menu_class      = (GtkMenuItemClass *) klass;

	/* GObject signals */
	gobject_class->dispose      = kz_bookmark_menu_item_dispose;
	gobject_class->constructor  = kz_bookmark_menu_item_constructor;

	widget_class->button_release_event = kz_bookmark_menu_item_button_release;

	menu_class->activate = kz_bookmark_menu_item_activate;
}


static void
kz_bookmark_menu_item_init (KzBookmarkMenuItem *menu)
{
	menu->is_releasing_button = FALSE;
}


static GObject*
kz_bookmark_menu_item_constructor (GType                  type,
			      guint                  n_props,
			      GObjectConstructParam *props)
{
	KzBookmarkMenuItem *item;
	KzBookmarkBaseMenuItem *parent;
	GObject *object;
	const gchar *title = NULL;
	gint bookmark_max_strnum = 0;
	GtkWidget *image = NULL;
	GtkWidget *accel_label;	
	KzFavicon *kzfav;
	KzBookmark *bookmark;
	GObjectClass *klass = G_OBJECT_CLASS(parent_class);

	object = klass->constructor(type, n_props, props);
	item = KZ_BOOKMARK_MENU_ITEM(object);
	parent = KZ_BOOKMARK_BASE_MENU_ITEM(object);

	bookmark = parent->bookmark;
#if 1
	if (kz_bookmark_is_folder(bookmark))
		title = kz_bookmark_get_document_title(bookmark);
	if (!title || !* title)
		title = kz_bookmark_get_title(bookmark);
	if (!title)
		title = "";

	KZ_CONF_GET("Global" , "bookmark_cutoff_strnum" , bookmark_max_strnum , INT);

	if(bookmark_max_strnum> 0 && (g_utf8_strlen(title , -1) > bookmark_max_strnum))
	{
		gint len = g_utf8_offset_to_pointer(title, bookmark_max_strnum) - title;
		gchar *short_title = g_alloca (len + 3 + 1);
		/* 3 for "..." , 1 for '\0' */

		g_utf8_strncpy(short_title , title , bookmark_max_strnum);
		short_title[len-3] = '.';
		short_title[len-2] = '.';
		short_title[len-1] = '.';
		short_title[len] = '\0';
		accel_label = gtk_accel_label_new(short_title);
	}
	else
		accel_label = gtk_accel_label_new(title);

	gtk_misc_set_alignment(GTK_MISC(accel_label), 0.0, 0.5);

	gtk_container_add(GTK_CONTAINER(item), accel_label);
	gtk_accel_label_set_accel_widget(GTK_ACCEL_LABEL(accel_label),
			GTK_WIDGET (item));
	gtk_widget_show(accel_label);
#endif

	/* favicon  */
	/* kzfav = parent->kz->kzfav; */
	kzfav = kz_favicon_get_instance();
	image = kz_favicon_get_widget(kzfav,
			kz_bookmark_get_link(bookmark),
			KZ_ICON_SIZE_BOOKMARK_MENU);
	g_object_unref(kzfav);
	if (image)
	{
		gtk_widget_show(image);
		gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(item),
					      image);
	}

	return object;
}


static void
kz_bookmark_menu_item_dispose (GObject *object)
{
	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}



GtkWidget *
kz_bookmark_menu_item_new (KzWindow *kz, KzBookmark *bookmark)
{
	KzBookmarkMenuItem *menu;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);
	g_return_val_if_fail(KZ_IS_BOOKMARK(bookmark), NULL);

	menu = g_object_new(KZ_TYPE_BOOKMARK_MENU_ITEM,
			   "kz-window", kz,
			   "bookmark",  bookmark,
			   NULL);

	return GTK_WIDGET(menu);
}


static gboolean
kz_bookmark_menu_item_button_release (GtkWidget *widget, GdkEventButton *event)
{
	const gchar *uri;
	gboolean keep = FALSE;
	gint button = event->button;
	KzBookmarkBaseMenuItem *parent;
	KzBookmarkMenuItem *item;;
	GtkAction *action;

	parent = KZ_BOOKMARK_BASE_MENU_ITEM(widget);
	item = KZ_BOOKMARK_MENU_ITEM(widget);

	uri = kz_bookmark_get_link(parent->bookmark);

	switch (button)
	{
	case 2:
		if (uri)
		{
			kz_window_open_new_tab(parent->kz, uri);
		}
		else if(kz_bookmark_is_folder(parent->bookmark))
		{

			kz_actions_set_bookmark_for_action(parent->kz, parent->bookmark);
			action = gtk_action_group_get_action(parent->kz->actions,
							     "OpenAllBookmarks");
			if (action)
				gtk_action_activate(action);
			kz_actions_set_bookmark_for_action(parent->kz, NULL);
		}

		KZ_CONF_GET("Global", "keep_bookmark_menu", keep, BOOL);
		if (!keep)
		{
			item->is_releasing_button = TRUE;
			return FALSE;
		}
		return TRUE;
		break;
	case 3:
		kz_actions_popup_bookmark_menu_modal(parent->kz, parent->bookmark,
						     event->button,
						     event->time);
		/* return TRUE; */
		break;
	default:
		break;
	}

	return FALSE;
}

static void
kz_bookmark_menu_item_activate (GtkMenuItem *item)
{
	const gchar *uri;
	KzBookmarkMenuItem *bitem;
	KzBookmarkBaseMenuItem *parent;

	parent = KZ_BOOKMARK_BASE_MENU_ITEM(item);
	bitem = KZ_BOOKMARK_MENU_ITEM(item);

	if (!bitem->is_releasing_button)
	{
		uri = kz_bookmark_get_link(parent->bookmark);

		kz_window_load_url(parent->kz, uri);
	}
	bitem->is_releasing_button = FALSE;
}
