/***************************************************************************
                          keybled.cpp  -  description
                             -------------------
    begin                : Fri Jul 13 2001
    copyright            : (C) 2001-2003 by Aurelien Jarno
    email                : aurelien@aurel32.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

// application specific includes
#include "keybled.h"
#include "popup.h"
#include "confdialog.h"
#include "kledtoggle.h"
#include "xkeylock.h"

// includes files for Qt
#include <qcombobox.h>
#include <qtooltip.h>
#include <qcheckbox.h>
#include <qradiobutton.h>
#include <qslider.h>
#include <qtabwidget.h>
#include <qfont.h>
#include <qcursor.h>

// includes files for KDE
#include <kglobal.h>
#include <kglobalsettings.h>
#include <klocale.h>
#include <kconfig.h>
#include <kapp.h>
#include <kwin.h>
#include <netwm.h>
#include <kaboutdata.h>
#include <kaboutapplication.h>
#include <kcolorbtn.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <kcolordialog.h>
#include <kaudioplayer.h>

// Pics
#include "pics/keybled.xpm"


#define REFRESHTIME	20

extern "C"
{
  KPanelApplet* init(QWidget *parent, const QString& configFile)
  {
    KGlobal::locale()->insertCatalogue("keybled");
    return new KeybLED(configFile, KPanelApplet::Normal,
                      KPanelApplet::About | KPanelApplet::Help | KPanelApplet::Preferences,
                      parent, "keybled");
  }
}

KeybLED::KeybLED(const QString& configFile, Type type, int actions, QWidget *parent, const char *name)
       : KPanelApplet(configFile, type, actions, parent, name),
       confDialog(0), ledColor("green")
{
  // Get the current application configuration handle
  ksConfig = config();
  readSettings();

  // Xtest stuff
  xKeyLock = new XKeyLock(this->x11Display());
  setLedStartup();

  // NUM Led creation
  numLed = new KLedToggle(this);
  numLed->installEventFilter(this);
  QToolTip::add(numLed, i18n("NUM.LOCK"));
  connect(numLed, SIGNAL(toggled()), this, SLOT(slotToggleNumLed()));

  numLabel = new QLabel(i18n("N"), this);
  numLabel->resize(20, 10);
  numLabel->setAlignment(AlignHCenter | AlignVCenter);
  numLabel->installEventFilter(this);
  QToolTip::add(numLabel, i18n("NUM.LOCK"));

  // CAPS Led creation
  capsLed = new KLedToggle(this);
  capsLed->installEventFilter(this);
  QToolTip::add(capsLed, i18n("CAPS.LOCK"));
  connect(capsLed, SIGNAL(toggled()), this, SLOT(slotToggleCapsLed()));

  capsLabel = new QLabel(i18n("C"), this);
  capsLabel->resize(20, 10);
  capsLabel->setAlignment(AlignHCenter | AlignVCenter);
  capsLabel->installEventFilter(this);
  QToolTip::add(capsLabel, i18n("CAPS.LOCK"));

  // SCROLL Led creation
  scrollLed = new KLedToggle(this);
  scrollLed->installEventFilter(this);
  QToolTip::add(scrollLed, i18n("SCROLL.LOCK"));
  connect(scrollLed, SIGNAL(toggled()), this, SLOT(slotToggleScrollLed()));

  scrollLabel = new QLabel(i18n("S"), this);
  scrollLabel->resize(20, 10);
  scrollLabel->setAlignment(AlignHCenter | AlignVCenter);
  scrollLabel->installEventFilter(this);
  QToolTip::add(scrollLabel, i18n("SCROLL.LOCK"));

  popup = new Popup();
  popup->setFont(QFont("fixed", 12));
  connect(popup, SIGNAL(clicked()), this, SLOT(hidePopup()));
  connect(popup, SIGNAL(savePos()), this, SLOT(savePopupPos()));
  
  if (popupOn)
  {
    ksConfig->setGroup("Popup");
    showPopup(QPoint(300, 200), ksConfig->readNumEntry("desktop", 0));
  } 
  applySettings();
 
  timerEvent();
  timer = new QTimer(this);
  connect(timer, SIGNAL(timeout()), this, SLOT(timerEvent()));
  timer->start(REFRESHTIME, false);
}

KeybLED::~KeybLED()
{
  disconnect(numLed, SIGNAL(toggled()), this, SLOT(slotToggleNumLed()));
  delete numLed;
  delete numLabel;

  disconnect(capsLed, SIGNAL(toggled()), this, SLOT(slotToggleScrollLed()));
  delete capsLed;
  delete capsLabel;

  disconnect(scrollLed, SIGNAL(toggled()), this, SLOT(slotToggleCapsLed()));
  delete scrollLed;
  delete scrollLabel;

  timer->stop();
  delete timer;

  delete popup;

  if (confDialog)
  {
    confDialog->close();
    delete confDialog;
  }
}


// Read the current configuration entries from the data base.
void KeybLED::readSettings()
{
  ksConfig->setGroup("Leds");
  ledColor = ksConfig->readColorEntry("color", &ledColor);
  ledLook = (KLed::Look) ksConfig->readNumEntry("look", KLed::Raised);
  ledShape = (KLed::Shape) ksConfig->readNumEntry("shape", KLed::Circular);
  ledSize = ksConfig->readNumEntry("size", 5);
  ledOrientation = ksConfig->readNumEntry("orientation", Horizontal);

  ksConfig->setGroup("Label");
  labelsEnabled = ksConfig->readBoolEntry("enabled", true);
  labelSize = ksConfig->readNumEntry("size", 4);
  labelVerticalPosition = ksConfig->readNumEntry("vertical_position", Left);
  labelHorizontalPosition = ksConfig->readNumEntry("horizontal_position", Bottom);

  ksConfig->setGroup("Display");
  numLedEnabled = ksConfig->readBoolEntry("num_led", true);
  capsLedEnabled = ksConfig->readBoolEntry("caps_led", true);
  scrollLedEnabled = ksConfig->readBoolEntry("scroll_led", true);
  popupEnabled = ksConfig->readBoolEntry("popup", true);
  mouseToggle = ksConfig->readBoolEntry("mouse_toggle", true);
  tooltipsEnabled = ksConfig->readBoolEntry("tooltips", true);
  QToolTip::setEnabled(tooltipsEnabled);

  ksConfig->setGroup("Popup");
  popupX = ksConfig->readNumEntry("X", 0);
  popupY = ksConfig->readNumEntry("Y", 0);
  popupOn = ksConfig->readBoolEntry("on", false);
  
  ksConfig->setGroup("Startup");
  numLedStartup = ksConfig->readNumEntry("num_led", 0);
  capsLedStartup = ksConfig->readNumEntry("caps_led", 0);
  scrollLedStartup = ksConfig->readNumEntry("scroll_led", 0);

  ksConfig->setGroup("Sounds");
  soundsEnabled = ksConfig->readBoolEntry("enabled", false);
  numSoundFile = ksConfig->readEntry("num_file", "");
  capsSoundFile = ksConfig->readEntry("caps_file", "");
  scrollSoundFile = ksConfig->readEntry("scroll_file", "");
}

// Dump the current configuration entries to the data base.
void KeybLED::writeSettings()
{
  ksConfig->setGroup("Leds");
  ksConfig->writeEntry("color", ledColor);
  ksConfig->writeEntry("look", ledLook);
  ksConfig->writeEntry("shape", ledShape);
  ksConfig->writeEntry("size", ledSize);
  ksConfig->writeEntry("orientation", ledOrientation);

  ksConfig->setGroup("Label");
  ksConfig->writeEntry("enabled", labelsEnabled);
  ksConfig->writeEntry("size", labelSize);
  ksConfig->writeEntry("vertical_position", labelVerticalPosition);
  ksConfig->writeEntry("horizontal_position", labelHorizontalPosition);

  ksConfig->setGroup("Display");
  ksConfig->writeEntry("num_led", numLedEnabled);
  ksConfig->writeEntry("caps_led", capsLedEnabled);
  ksConfig->writeEntry("scroll_led", scrollLedEnabled);
  ksConfig->writeEntry("popup", popupEnabled);
  ksConfig->writeEntry("mouse_toggle", mouseToggle);
  ksConfig->writeEntry("tooltips", tooltipsEnabled);

  ksConfig->setGroup("Startup");
  ksConfig->writeEntry("num_led", numLedStartup);
  ksConfig->writeEntry("caps_led", capsLedStartup);
  ksConfig->writeEntry("scroll_led", scrollLedStartup);

  ksConfig->setGroup("Popup");
  ksConfig->writeEntry("X", popupX);
  ksConfig->writeEntry("Y", popupY);
  ksConfig->writeEntry("on", popupOn);
  NETWinInfo wm_client(qt_xdisplay(), popup->winId(), qt_xrootwin(), NET::WMDesktop);
  ksConfig->writeEntry("desktop", wm_client.desktop());

  ksConfig->setGroup("Sounds");
  ksConfig->writeEntry("enabled", soundsEnabled);
  ksConfig->writeEntry("num_file", numSoundFile);
  ksConfig->writeEntry("caps_file", capsSoundFile);
  ksConfig->writeEntry("scroll_file", scrollSoundFile);
  
  ksConfig->sync();
}

void KeybLED::setLedStartup()
{
  if (xKeyLock->isNumLockReadable() && numLedStartup != unchanged)
    xKeyLock->setNumLock(numLedStartup == on);
    
  if (xKeyLock->isCapsLockReadable() && capsLedStartup != unchanged)
    xKeyLock->setCapsLock(capsLedStartup == on);
    
  if (xKeyLock->isScrollLockReadable() && scrollLedStartup != unchanged)
    xKeyLock->setScrollLock(scrollLedStartup == on);
}

void KeybLED::preferences()
{
  if (confDialog)
  {
    KWin::activateWindow(confDialog->winId());
    return;
  }

  confDialog = new ConfDialog(this, 0, FALSE, WDestructiveClose);

  connect(confDialog, SIGNAL(okClicked()), this, SLOT(slotDialogOkClicked()));
  connect(confDialog, SIGNAL(applyClicked()), this, SLOT(slotDialogApplyClicked()));
  connect(confDialog, SIGNAL(destroyed()), SLOT(slotDialogDeleted()));

  confDialog->setNumEnabled(xKeyLock->isNumLockReadable());
  confDialog->setCapsEnabled(xKeyLock->isCapsLockReadable());
  confDialog->setScrollEnabled(xKeyLock->isScrollLockReadable());

  // Display tab
  confDialog->numLedEnabled->setChecked(numLedEnabled);
  confDialog->capsLedEnabled->setChecked(capsLedEnabled);
  confDialog->scrollLedEnabled->setChecked(scrollLedEnabled);
  confDialog->tooltipsEnabled->setChecked(tooltipsEnabled);
  confDialog->popupEnabled->setChecked(popupEnabled);
  confDialog->mouseToggle->setChecked(mouseToggle);
  confDialog->mouseToggle->setEnabled(xKeyLock->isNumLockReadable() || xKeyLock->isCapsLockReadable() || xKeyLock->isScrollLockReadable());

  // Leds tab
  confDialog->ledSize->setValue(ledSize);

  switch(ledOrientation)
  {
    case Vertical : confDialog->ledVertical->setChecked(true);
      break;
    case Horizontal : confDialog->ledHorizontal->setChecked(true);
      break;
  }

  switch(ledLook)
  {
    case KLed::Flat : confDialog->ledFlat->setChecked(true);
      break;
    case KLed::Raised : confDialog->ledRaised->setChecked(true);
      break;
    case KLed::Sunken : confDialog->ledSunken->setChecked(true);
      break;
  }

  switch(ledShape)
  {
    case KLed::Rectangular : confDialog->ledRectangular->setChecked(true);
      break;
    case KLed::Circular : confDialog->ledCircular->setChecked(true);
      break;
  }

  confDialog->ledColor->setColor(ledColor);

  // Label tab
  confDialog->labelsEnabled->setChecked(labelsEnabled);
  confDialog->labelSize->setValue(labelSize);

  switch(labelVerticalPosition)
  {
    case Left : confDialog->labelLeft->setChecked(true);
      break;
    case Right : confDialog->labelRight->setChecked(true);
      break;
  }

  switch(labelHorizontalPosition)
  {
    case Top : confDialog->labelTop->setChecked(true);
      break;
    case Bottom : confDialog->labelBottom->setChecked(true);
      break;
  }

  // Startup tab
  confDialog->numLedStartup->setCurrentItem(numLedStartup);
  confDialog->capsLedStartup->setCurrentItem(capsLedStartup);
  confDialog->scrollLedStartup->setCurrentItem(scrollLedStartup);

  // Sounds tab
  confDialog->soundsEnabled->setChecked(soundsEnabled);
  confDialog->numLedSoundURL->setURL(numSoundFile);
  confDialog->capsLedSoundURL->setURL(capsSoundFile);
  confDialog->scrollLedSoundURL->setURL(scrollSoundFile);

  confDialog->show();
}

void KeybLED::slotDialogOkClicked()
{
  slotDialogApplyClicked();
  delete confDialog;
}

void KeybLED::slotDialogApplyClicked()
{
  // Display tab
  tooltipsEnabled = confDialog->tooltipsEnabled->isChecked();
  popupEnabled = confDialog->popupEnabled->isChecked();
  mouseToggle = confDialog->mouseToggle->isChecked();
  numLedEnabled = confDialog->numLedEnabled->isChecked();
  capsLedEnabled = confDialog->capsLedEnabled->isChecked();
  scrollLedEnabled = confDialog->scrollLedEnabled->isChecked();

  // Leds tab
  if (confDialog->ledRectangular->isChecked()) ledShape = KLed::Rectangular;
  if (confDialog->ledCircular->isChecked()) ledShape = KLed::Circular;
  if (confDialog->ledFlat->isChecked()) ledLook = KLed::Flat;
  if (confDialog->ledRaised->isChecked()) ledLook = KLed::Raised;
  if (confDialog->ledSunken->isChecked()) ledLook = KLed::Sunken;
  ledColor = confDialog->ledColor->color();
  ledSize = confDialog->ledSize->value();
  if (confDialog->ledVertical->isChecked()) ledOrientation = Vertical;
  if (confDialog->ledHorizontal->isChecked()) ledOrientation = Horizontal;

  // Labels tab
  labelsEnabled = confDialog->labelsEnabled->isChecked();
  labelSize = confDialog->labelSize->value();

  if (confDialog->labelLeft->isChecked()) labelVerticalPosition = Left;
  if (confDialog->labelRight->isChecked()) labelVerticalPosition = Right;

  if (confDialog->labelTop->isChecked()) labelHorizontalPosition = Top;
  if (confDialog->labelBottom->isChecked()) labelHorizontalPosition = Bottom;

  // Startup tab
  numLedStartup = confDialog->numLedStartup->currentItem();
  capsLedStartup = confDialog->capsLedStartup->currentItem();
  scrollLedStartup = confDialog->scrollLedStartup->currentItem();

  // Sounds tab
  soundsEnabled = confDialog->soundsEnabled->isChecked();
  numSoundFile = confDialog->numLedSoundURL->url();
  capsSoundFile = confDialog->capsLedSoundURL->url();
  scrollSoundFile = confDialog->scrollLedSoundURL->url();

  writeSettings();
  applySettings();
  emit updateLayout();
}

void KeybLED::slotDialogCancelClicked()
{
  delete confDialog;
}

void KeybLED::slotDialogDeleted()
{
  confDialog = 0;
}

void KeybLED::applySettings()
{
  numLed->setShape(ledShape);
  numLed->setLook(ledLook);
  numLed->setColor(ledColor);

  capsLed->setShape(ledShape);
  capsLed->setLook(ledLook);
  capsLed->setColor(ledColor);

  scrollLed->setShape(ledShape);
  scrollLed->setLook(ledLook);
  scrollLed->setColor(ledColor);

  QToolTip::setEnabled(tooltipsEnabled);

  if (!popupEnabled && popupOn) hidePopup(); 

  updateGeometry();
}

void KeybLED::about()
{
  KAboutData aboutData ("keybled", I18N_NOOP("KeybLED"), VERSION,
                        I18N_NOOP("This application displays a keyboard's LED status. It is particularly\n"
                                  "useful for keyboards which lack their own LED status indicators (e.g.\n" 
                                  "cordless keyboards), or just for the fun!"),
                        KAboutData::License_GPL, "(c) 2001-2003, Aurelien Jarno", 0, 0, "aurelien@aurel32.net");
  aboutData.addAuthor("Aurelien Jarno", 0, "aurelien@aurel32.net", "http://www.aurel32.net");
  KAboutApplication dialog(&aboutData);
  dialog.exec();
}

void KeybLED::help()
{
  KApplication *app = (KApplication*) KGlobal::instance();
  app->invokeHelp("", "keybled");
}

void KeybLED::openContextMenu()
{
  QPixmap keybled_icon((const char**) keybled_icon_data);
  
  KPopupMenu *menu = new KPopupMenu();
  menu->insertTitle(keybled_icon, i18n("KeybLED"));
  menu->insertItem(i18n("&About KeybLed"), 10);
  menu->insertItem(SmallIcon("help"), i18n("&Help..."), 20);
  menu->insertSeparator();
  menu->insertItem(SmallIcon("configure"), i18n("&Preferences..."), 30);

  switch (menu->exec(QCursor::pos()))
  {
    case 10: about();
             break;
    case 20: help();
             break;
    case 30: preferences();
             break;
    default: break;	     
  }

  delete menu;
}

void KeybLED::updateGeometry()
{
  int led = 0; int label = 0; int intled = 0; int xled = 0; int xlabel = 0; int y = 0;
  int nbLed = (numLedEnabled && xKeyLock->isNumLockReadable()) +
              (capsLedEnabled && xKeyLock->isCapsLockReadable()) +
              (scrollLedEnabled && xKeyLock->isScrollLockReadable());

  QFont f(KGlobalSettings::generalFont());

  if (nbLed > 0)
  {
    if (orientation() == KPanelApplet::Horizontal)
    {
       if (ledOrientation == Vertical)
       {
        led = ((height() / nbLed) * ledSize) / 6;
        label = ((height() / nbLed) * labelSize) / 6;
        intled = (height() - (nbLed * led)) / (nbLed + 1);

        if (labelVerticalPosition == Right || !labelsEnabled)
        {
          xled = 1;
          xlabel = 2 + led;
        }
        else
        {
          xled = 2 + label;
          xlabel = 1;
        }
      }
      else
      {
        led = (height() * ledSize) / 12;
        if (labelsEnabled)
        {
          label = (height() * labelSize) / 12;
          if (label > led) intled = 1 + (label - led); else intled = 1;
          if (labelHorizontalPosition == Top)
          {
            xlabel = ((height() - (led + label)) / 4);
            xled = ((height() - (led + label)) / 2) + label;
          }
          else
          {
            xled = ((height() - (led + label)) / 4);
            xlabel = ((height() - (led + label)) / 2) + led;
          }
        }
        else
        {
          label = 0;
          xled = (height() - led) / 2;
          xlabel = 0;
          intled = 1;
        }
      }
    }
    else
    {
      if (ledOrientation == Vertical)
      {
        led = (width() * ledSize) / 12;
        if (labelsEnabled)
        {
          label = (width() * labelSize) / 12;
          if (label > led) intled = 1 + (label - led); else intled = 1;
          if (labelVerticalPosition == Left)
          {
            xlabel = ((width() - (led + label)) / 4);
            xled = ((width() - (led + label)) / 2) + label;
          }
          else
          {
            xled = ((width() - (led + label)) / 4);
            xlabel = ((width() - (led + label)) / 2) + led;
          }
        }
        else
        {
          label = 0;
          xled = (width() - led) / 2;
          xlabel = 0;
          intled = 1;
        }
      }
      else
      {
        led = ((width() / nbLed) * ledSize) / 6;
        label = ((width() / nbLed) * labelSize) / 6;
        intled = (width() - (nbLed * led)) / (nbLed + 1);

        if (labelHorizontalPosition == Bottom || !labelsEnabled)
        {
          xled = 1;
          xlabel = 2 + led;
        }
        else
        {
          xled = 2 + label;
          xlabel = 1;
        }
      }
    }

    if (label != 0) f.setPointSize(label);
    y = intled;
  }

  if (numLedEnabled && xKeyLock->isNumLockReadable())
  {
    numLed->resize(led, led);
    if (ledOrientation == Vertical) numLed->move(xled, y);
    else numLed->move(y, xled);
    numLed->show();

    if (labelsEnabled)
    {
      numLabel->resize(label, label);
      if (ledOrientation == Vertical) numLabel->move(xlabel, y + ((led - label) / 2));
      else numLabel->move(y + ((led - label) / 2), xlabel);
      numLabel->show();
      numLabel->setFont(f);
    }
    else numLabel->hide();

    y += intled + led;
  }
  else
  {
    numLed->hide();
    numLabel->hide();
  }

  if (capsLedEnabled && xKeyLock->isCapsLockReadable())
  {
    capsLed->resize(led, led);
    if (ledOrientation == Vertical) capsLed->move(xled, y);
    else capsLed->move(y, xled);
    capsLed->show();

    if (labelsEnabled)
    {
      capsLabel->resize(label, label);
      if (ledOrientation == Vertical) capsLabel->move(xlabel, y + ((led - label) / 2));
      else capsLabel->move(y + ((led - label) / 2), xlabel);
      capsLabel->show();
      capsLabel->setFont(f);
    }
    else capsLabel->hide();

    y += intled + led;
  }
  else
  {
    capsLed->hide();
    capsLabel->hide();
  }

  if (scrollLedEnabled && xKeyLock->isScrollLockReadable())
  {
    scrollLed->resize(led, led);
    if (ledOrientation == Vertical) scrollLed->move(xled, y);
    else scrollLed->move(y, xled);
    scrollLed->show();

    if (labelsEnabled)
    {
      scrollLabel->resize(label, label);
      if (ledOrientation == Vertical) scrollLabel->move(xlabel, y + ((led - label) / 2));
      else scrollLabel->move(y + ((led - label) / 2), xlabel);
      scrollLabel->show();
      scrollLabel->setFont(f);
    }
    else scrollLabel->hide();
  }
  else
  {
    scrollLed->hide();
    scrollLabel->hide();
  }
}

int KeybLED::widthForHeight(int height) const
{
  int nbLed = (numLedEnabled && xKeyLock->isNumLockReadable()) +
              (capsLedEnabled && xKeyLock->isCapsLockReadable()) +
              (scrollLedEnabled && xKeyLock->isScrollLockReadable());
  
  if (nbLed > 0)
  {
    if (ledOrientation == Vertical)
    {
      if (labelsEnabled) return (3 + (height * (ledSize + labelSize)) / (6 * nbLed));
      else return (2 + (height * ledSize) / (6 * nbLed));
    }
    else
    {
      if ((labelSize > ledSize) && labelsEnabled) return 1 + ((height * labelSize) / 12 + 1) * nbLed;
      else return (1 + ((height * ledSize) / 12 + 1) * nbLed);
    }
  }
  else return 0;
}

int KeybLED::heightForWidth(int width) const
{
  int nbLed = (numLedEnabled && xKeyLock->isNumLockReadable()) +
              (capsLedEnabled && xKeyLock->isCapsLockReadable()) +
              (scrollLedEnabled && xKeyLock->isScrollLockReadable());

  if (nbLed > 0)
  {
    if (ledOrientation == Horizontal)
    {
      if (labelsEnabled) return (3 + (width * (ledSize + labelSize)) / (6 * nbLed));
      else return (2 + (width * ledSize) / (6 * nbLed));
    }
    else
    {
      if ((labelSize > ledSize) && labelsEnabled) return 1 + ((width * labelSize) / 12 + 1) * nbLed;
      else return (1 + ((width * ledSize) / 12 + 1) * nbLed);
    }
  }
  else return 0;
}

void KeybLED::resizeEvent(QResizeEvent *event)
{
  updateGeometry();
}

void KeybLED::timerEvent()
{
  static bool init = false;

  if ((xKeyLock->getNumLock()) != numLed->state()) 
  {
    numLed->toggle();
    if (! numSoundFile.isEmpty() && soundsEnabled && init) KAudioPlayer::play(numSoundFile);
  }
  if ((xKeyLock->getCapsLock()) != capsLed->state())
  {
    capsLed->toggle();
    if (! capsSoundFile.isEmpty() && soundsEnabled && init) KAudioPlayer::play(capsSoundFile);
  }
  if ((xKeyLock->getScrollLock()) != scrollLed->state())
  {
    scrollLed->toggle();
    if (! scrollSoundFile.isEmpty() && soundsEnabled && init) KAudioPlayer::play(scrollSoundFile);
  }

  if (popupOn || !init)
  {
    QString text = "\n";

    if (xKeyLock->isNumLockReadable())
    {
      text.append(i18n("NUM.LOCK       "));
      xKeyLock->getNumLock() ? text.append(i18n("Enabled \n")) : text.append(i18n("Disabled\n"));
    }

    if (xKeyLock->isCapsLockReadable())
    {
      text.append(i18n("CAPS.LOCK      "));
      xKeyLock->getCapsLock() ? text.append(i18n("Enabled \n")) : text.append(i18n("Disabled\n"));
    }

    if (xKeyLock->isScrollLockReadable())
    {
      text.append(i18n("SCROLL.LOCK    "));
      xKeyLock->getScrollLock() ? text.append(i18n("Enabled \n")) : text.append(i18n("Disabled\n"));
    }

    popup->setText(text);
  }

  init = true;
}

// catch the mouse clicks of our child widgets
bool KeybLED::eventFilter(QObject *object, QEvent *event)
{
  if ((object == numLabel || object == capsLabel || object == scrollLabel)
     && (event->type() == QEvent::MouseButtonPress))
  {
    mousePressEvent(static_cast<QMouseEvent*>(event));
    return true;
  }

  if ((event->type() == QEvent::MouseButtonPress)
     && (object == numLed || object == capsLed || object == scrollLed)
     && (!mouseToggle || (static_cast<QMouseEvent*>(event))->button() != QMouseEvent::LeftButton))
  {
    mousePressEvent(static_cast<QMouseEvent*>(event));
    return true;
  }

  return KPanelApplet::eventFilter(object, event);
}

void KeybLED::mousePressEvent(QMouseEvent *event)
{
 if (event->button() == QMouseEvent::LeftButton && popupEnabled)
 {
   if (!popupOn) showPopup(event->globalPos());
   else hidePopup();
 }
 if (event->button() == QMouseEvent::RightButton) openContextMenu();
}


void KeybLED::slotToggleNumLed()
{
  if (mouseToggle)
  {
    xKeyLock->toggleNumLock();
    timerEvent();
  }
}

void KeybLED::slotToggleCapsLed()
{
  if (mouseToggle)
  {
    xKeyLock->toggleCapsLock();
    timerEvent();
  }
}

void KeybLED::slotToggleScrollLed()
{
  if (mouseToggle)
  {
    xKeyLock->toggleScrollLock();
    timerEvent();
  }
}

void KeybLED::showPopup(QPoint Point, int desktop)
{
  popupOn = true;
  if(popupX == 0 && popupY == 0)
  {
    popupX = Point.x() - 220;
    popupY = Point.y() - 100;
  }

  popup->move(popupX, popupY);
  if ((desktop == 0) || (desktop > KWin::numberOfDesktops())) desktop = KWin::currentDesktop();
  KWin::setOnDesktop(popup->winId(), desktop);
  popup->show();                                                  
  writeSettings();
}

void KeybLED::hidePopup()
{
  popupOn = false;
  savePopupPos();
  popup->hide();
  writeSettings();
}

void KeybLED::savePopupPos()
{
  popupX = popup->x();
  popupY = popup->y();
  writeSettings();
}
