/*
 * kino_common.cc KINO GUI Common Object
 * Copyright (C) 2001 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define MULTITHREADED

#include <iostream>
using std::cout;
using std::endl;

#include <string>

#include "kino_common.h"
#include "page_editor.h"
#include "page_capture.h"
#include "page_timeline.h"
#include "page_undefined.h"
#include "page_export.h"
#include "page_bttv.h"
#include "preferences.h"
#include "message.h"
#include "frame.h"
#include "displayer.h"

#include "preferences.h"
#include "riff.h"
#include "avi.h"
#include "filehandler.h"
#include "ieee1394io.h"
#include "progress_dialog.h"
#include "error.h"


extern "C" {
#include "interface.h"

#ifdef MULTITHREADED
#include <pthread.h>

#endif
}

/** Constructor for KinoCommon - initialises all GUI widgets.

  	\param widget top level widget for the main window
*/

KinoCommon::KinoCommon( GtkWidget *widget ) {
	cout << "> Kino Common being built" << endl;

	// Initialise class variables from GUI components
	this->widget = widget;
	this->edit_menu = lookup_widget( widget, "edit" );
	this->command_menu = lookup_widget( widget, "add_file" );
	this->view_menu = lookup_widget( widget, "view1" );
	this->notebook = GTK_NOTEBOOK( lookup_widget( widget, "main_notebook" ) );
	this->scenes = GNOME_ICON_LIST( lookup_widget( widget, "iconlist2" ) );
	this->command = GTK_ENTRY( lookup_widget( widget, "command" ) );
	this->command_label = GTK_LABEL( lookup_widget( widget, "cmd_label" ) );
	this->result_label = GTK_LABEL( lookup_widget( widget, "result_label" ) );
	this->position_label = GTK_LABEL( lookup_widget( widget, "position_label" ) );
	this->video_start_movie_button = GTK_BUTTON( lookup_widget( widget, "video_start_movie_button" ) );
	this->video_start_scene_button = GTK_BUTTON( lookup_widget( widget, "video_start_scene_button" ) );
	this->video_rewind_button = GTK_BUTTON( lookup_widget( widget, "video_rewind_button" ) );
	this->video_back_button = GTK_BUTTON( lookup_widget( widget, "video_back_button" ) );
	this->video_play_button = GTK_BUTTON( lookup_widget( widget, "video_play_button" ) );
	this->video_pause_button = GTK_BUTTON( lookup_widget( widget, "video_pause_button" ) );
	this->video_stop_button = GTK_BUTTON( lookup_widget( widget, "video_stop_button" ) );
	this->video_forward_button = GTK_BUTTON( lookup_widget( widget, "video_forward_button" ) );
	this->video_fast_forward_button = GTK_BUTTON( lookup_widget( widget, "video_fast_forward_button" ) );
	this->video_end_scene_button = GTK_BUTTON( lookup_widget( widget, "video_end_scene_button" ) );
	this->video_end_movie_button = GTK_BUTTON( lookup_widget( widget, "video_end_movie_button" ) );
	this->video_shuttle = GTK_RANGE( lookup_widget( widget, "hscale_shuttle" ) );

	// Initialise other class variables
	this->currentPage = -1;
	strcpy( this->currentDirectory, "" );
	strcpy( this->tempFileName, "" );
	strcpy( this->playlistFileName, "" );
	strcpy( this->playlistName, "" );

	// Create page objects
	this->editor = new PageEditor( this );
	this->capture = new PageCapture( this );
	this->timeline = new PageTimeline( this );
	this->undefined = new PageUndefined( this );
	this->bttv = new PageBttv( this );
	this->exportPage = new PageExport( this );

	this->g_currentFrame = -1;
	this->hasListChanged = TRUE;

	// Set to the first page
	this->setCurrentPage( PAGE_EDITOR );

	this->component_state = VIDEO_STOP;
	this->is_component_state_changing = false;
	this->showInfoFlag = (bool) this->getConfig().enableTimecode;
}

/**	Destructor for the kino common object.
 */

KinoCommon::~KinoCommon() {
	cout << "> Kino Common being destroyed" << endl;
	clean();
	delete this->editor;
	delete this->capture;
	delete this->timeline;
	delete this->undefined;
	delete this->bttv;
	delete this->exportPage;
	cout << "> Kino Common destroyed" << endl;
}

/** Carries out a New File request in the kino application. All notebook
	pages are informed of the action through their own newFile method.
*/

void KinoCommon::newFile( ) {
	cout << ">> Kino Common newFile" << endl;

	// Specifics for this class
	this->changePageRequest( 0 );
	if ( getPlayList()->GetNumFrames() > 0 )
		this->getPlayList()->Delete( 0, getPlayList()->GetNumFrames() );
	this->g_currentFrame = -1;

	// Call new File for each page
	Page *page = NULL;
	for ( int index = 0; ( page = getPage( index ) ) != NULL; index ++ )
		page->newFile();

	this->showScenes();
	this->windowChanged();
}

/**	Sends change page request to the gui.

	\param page page to change to
*/

void KinoCommon::changePageRequest( int page ) {
	gtk_notebook_set_page( notebook, page );
}

/**	Carries out a change page - this should be triggered by changePageRequest
	to ensure that the GUI stays in sync with the common structure.

	\param page page to change to
*/

void KinoCommon::setCurrentPage( int page ) {
	if ( this->currentPage != page ) {
		if ( this->currentPage != -1 )
			clean();
		this->currentPage = page;
		start();
	}
}

/** Returns the Page object associated to the specified notebook page.

	\param page index of Page object to return
	\return the Page object associated to the given notebook page or NULL if invalid.
*/

Page *KinoCommon::getPage( int page ) {
	Page *ret = NULL;

	switch( page ) {
		case PAGE_EDITOR:
			ret = getPageEditor();
			break;
		case PAGE_CAPTURE:
			ret = getPageCapture();
			break;
		case PAGE_TIMELINE:
			ret = getPageTimeline();
			break;
		case PAGE_BTTV:
			ret = getPageBttv();
			break;
		case PAGE_EXPORT:
			ret = getPageExport();
			break;
	}

	return ret;
}

/** Returns the current Page object.

	\return the current Page object (will be PageUndefined if current page is invalid).
*/

Page *KinoCommon::getCurrentPage( ) {
	Page *ret = getPage( this->currentPage );
	if ( ret == NULL )
		ret = getPageUndefined();
	return ret;
}

/** Returns the Page associated to the Editor page.

	\return the Page object requested
*/

PageEditor *KinoCommon::getPageEditor( ) {
	return editor;
}

/** Returns the Page associated to the Capture page.

	\return the Page object requested
*/

PageCapture *KinoCommon::getPageCapture( ) {
	return capture;
}

/** Returns the Page associated to the Timeline page.

	\return the Page object requested
*/

PageTimeline *KinoCommon::getPageTimeline( ) {
	return timeline;
}

/** Returns the Page associated to the Undefined page.

	\return the Page object requested
*/

PageUndefined *KinoCommon::getPageUndefined( ) {
	return undefined;
}

/** Returns the 'BTTV' page (test code for extending hardware support).

	\return the Page object requested
*/

PageBttv *KinoCommon::getPageBttv( ) {
	return bttv;
}

/** Returns the Export page.

	\return the Page object requested
*/

PageExport *KinoCommon::getPageExport( ) {
	return exportPage;
}

/** Inserts a file at the current position.
*/

void KinoCommon::insertFile( ) 
{
	char *filename = this->getFileToOpen( "Choose and AVI file or playlist to insert" );
	if ( strcmp( filename, "" ) ) 
	{
   		char	name[256];

   		save_cd(filename);

		switch ( checkFile( filename ) ) 
		{
			case AVI:
           		strcpy(name, filename);
				if ( g_currentFrame == -1 )
           			loadAVI(name, g_currentFrame + 1);
				else
           			loadAVI(name, g_currentFrame);
       			hasListChanged = TRUE;
				break;
			case PLAYLIST:
           		strcpy(name, filename);
           		loadPlayList(name, g_currentFrame);
       			hasListChanged = TRUE;
				break;
			case UNKNOWN_FORMAT:
				modal_message( "Invalid file specified." );
				break;
		}
	}

	moveToFrame( );
}

/** Appends a file after the current position.
*/

void KinoCommon::appendFile( ) 
{
	char *filename = this->getFileToOpen( "Choose and AVI file or playlist to append" );
	if ( strcmp( filename, "" ) ) 
	{
   		char	name[256];
   		save_cd(filename);

		switch ( checkFile( filename ) ) 
		{
			case AVI:
           		strcpy(name, filename);
           		loadAVI(name, g_currentFrame + 1);
        		g_currentFrame++;
        		hasListChanged = TRUE;
				break;
			case PLAYLIST:
           		strcpy(name, filename);
           		loadPlayList(name, g_currentFrame + 1);
        		g_currentFrame++;
        		hasListChanged = TRUE;
				break;
			case UNKNOWN_FORMAT:
				modal_message( "Invalid file specified." );
				break;
		}
	}

	moveToFrame( );
}

/** Opens a file. An AVI or a PLAYLIST causes a newFile to called prior to the load
	and a PLAYLIST becomes the currently editted file.
*/

void KinoCommon::loadFile( )
{
	char *filename = this->getFileToOpen( "Choose and AVI file or playlist" );

	if ( strcmp( filename, "" ) ) 
	{
    	char	name[256];

   		save_cd(filename);

		switch ( checkFile( filename ) ) 
		{
			case AVI:
				newFile( );
           		strcpy(name, filename);
           		loadAVI(name, this->getPlayList()->GetNumFrames());
       			g_currentFrame = 0;
       			hasListChanged = TRUE;
				break;
			case PLAYLIST:
				newFile( );
           		strcpy(playlistFileName, filename);
           		loadPlayList(playlistFileName, this->getPlayList()->GetNumFrames());
       			g_currentFrame = 0;
       			hasListChanged = TRUE;
				break;
			case UNKNOWN_FORMAT:
				modal_message( "Invalid file specified." );
				break;
		}
	}

	moveToFrame( );
}

/** Save the current play list with a different name to the original loaded (or
	previously saved as) play list file name. 
*/

void KinoCommon::savePlayListAs( )
{
	changePageRequest( 0 );
	char *filename = this->getFileToSave( "Save the playlist" );
	if ( strcmp( filename, "" ) ) 
	{
		save_cd( filename );
       	strcpy(playlistFileName, filename);
       	savePlayList(playlistFileName);
	}
}

/** Save the current play list to the file used to load the playlist originally. If
	no file was originally used then use savePlayListAs to obtain one.
*/

void KinoCommon::savePlayList( )
{
	if ( !strcmp( playlistFileName, "" ) )
		savePlayListAs( );
	else
		savePlayList(playlistFileName);
}

/**	Save the current play list as a movie.
*/

void KinoCommon::saveMovie( )
{
	changePageRequest( 0 );
   	char *filename = this->getFileToSave( "Save an AVI file" );
	if ( strcmp( filename, "" ) ) 
	{
		save_cd( filename );
       	saveAVI(playlistFileName);
	}
}

/**	Save the current frame as a jpg.
*/

void KinoCommon::saveFrame( )
{
	getCurrentPage()->saveFrame();
}

/** Handles the modal dialog for collecting a file to open. The file is not opened, 
	but returned by this method.

	It relies on the the callback functions defined for the open_movie_dialog. All three
	of these must call gtk_main_quit when finished and the delete and cancel should set
	the filename to an empty string.

	\todo remove glade generated code for modal dialogs
	\param title title of the window
	\return a string inidicating the selected file (an empty string denotes no selection)
*/

char *KinoCommon::getFileToOpen( char *title ) {
	GtkWidget *dialog = create_open_movie_dialog();
	changePageRequest( 0 );
	gtk_window_set_title( GTK_WINDOW (dialog), title );
	if (strlen(currentDirectory))
		gtk_file_selection_set_filename( GTK_FILE_SELECTION(dialog), currentDirectory);
	gtk_widget_show( dialog );
	gtk_main( );
	gchar *filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(dialog));
	if ( filename[ strlen( filename ) - 1 ] != '/' )
		strcpy( tempFileName, filename );
	else
		strcpy( tempFileName, "" );
	gtk_widget_destroy( dialog );
	return tempFileName;
}

/** Handles the modal dialog for collecting a file to save. The file is not saved, 
	but returned by this method.

	It relies on the the callback functions defined for the save_movie_dialog. All three
	of these must call gtk_main_quit when finished and the delete and cancel should set
	the filename to an empty string.

	\todo remove glade generated code for modal dialogs
	\param title title of the window
	\return a string inidicating the selected file (an empty string denotes no selection)
*/

char *KinoCommon::getFileToSave( char *title ) {
	GtkWidget *dialog = create_save_movie_dialog();
	gtk_window_set_title( GTK_WINDOW (dialog), title );
	if (strlen(currentDirectory))
		gtk_file_selection_set_filename( GTK_FILE_SELECTION(dialog), currentDirectory);
	gtk_widget_show( dialog );
	gtk_main( );
	gchar *filename = gtk_file_selection_get_filename(GTK_FILE_SELECTION(dialog));
	if ( filename[ strlen( filename ) - 1 ] != '/' )
		strcpy( tempFileName, filename );
	else
		strcpy( tempFileName, "" );
	gtk_widget_destroy( dialog );
	return tempFileName;
}

/**	Determines the format of the file.

	\param filename file to check
	\return AVI, PLAYLIST or UNDEFINED
*/

int KinoCommon::checkFile( char *filename ) 
{
	int ret = UNKNOWN_FORMAT;
    FILE *f;
    char buffer[20];

   	f = fopen(filename, "r");
   	if (f != NULL) 
	{
        memset(buffer, 0, 20);
        fread(buffer, 20, 1, f);
        if ((strncmp(buffer, "RIFF", 4) == 0) && (strncmp(buffer + 8, "AVI ", 4) == 0))
			ret = AVI;
        else if (strncmp(buffer, "<?xml version=\"1.0\"?>", 20) == 0)
			ret = PLAYLIST;
        fclose(f);
   	}

	return ret;
}

/** Stores the directory associated to the filename as the current working directory
	of the application (all subsequent dialogs will default to this directory).

	\param filename a fully specified directory + (optional) filename
*/

void KinoCommon::save_cd(char *filename) 
{
   	gchar    *tmp;
   	strcpy( currentDirectory, filename);
   	tmp = strrchr( currentDirectory, '/');
   	if (tmp != NULL) tmp[1] = '\0';
}

/** Loads the specified AVI before the specified frame.

	\param file file to load
	\param before frame to insert at
*/

void KinoCommon::loadAVI(char *file, int before) {

    PlayList        newList;

    try {
        newList.LoadAVI(file);
        this->getPlayList()->InsertPlayList(newList, before);
    } catch (string s) {
        cout << "Could not load AVI file " << file << ", because an exception has occurred: " << endl;
        cout << s << endl;
    }
}

/** Saves the playlist as an AVI.

	\param file file to save
*/

void KinoCommon::saveAVI(char *file) {

    Frame           frame;
    AVIHandler      *writer = NULL;
    Preferences     &prefs = Preferences::getInstance();

    CreateProgressDialog();

    try {

        writer = new AVIHandler(prefs.fileFormat);

        /* set command line parameters to the writer object */

        writer->SetMaxFrameCount(prefs.frames);
        writer->SetAutoSplit(prefs.autoSplit);
        writer->SetTimeStamp(prefs.timeStamp);
        writer->SetEveryNthFrame(prefs.every);
        writer->SetBaseName(file);

        this->getPlayList()->GetFrame(0, frame);
        writer->SetSampleFrame(frame);
        for (int i = 0; i < this->getPlayList()->GetNumFrames(); ++i) {
            this->getPlayList()->GetFrame(i, frame);
            writer->WriteFrame(frame);
            if (UpdateProgressDialog((gfloat)(i + 1) / (gfloat)this->getPlayList()->GetNumFrames(), 2.0) == true)
                break;
        }
        writer->Close();
    } catch(string s) {
        cout << "Could not save AVI file " << file << ", because an exception has occurred: " << endl;
        cout << s << endl;
    }

    delete writer;
    CloseProgressDialog();
}

/** Loads the specified play list before the specified frame.

	\param file file to load
	\param before frame to insert at
*/

void KinoCommon::loadPlayList(char *file, int before) {

    PlayList newList;

    if (newList.LoadPlayList(file) == false)
        modal_message("Could not load the playlist");
    else
        this->getPlayList()->InsertPlayList(newList, before);
}

/** Saves the playlist.

	\param file file to save
*/

void KinoCommon::savePlayList(char *file) {
    if (this->getPlayList()->SavePlayList(file) == false)
        modal_message("Could not save the playlist");
}

/** Loads all files specified in the command line style arguments.

  	\param argc number of arguments
	\param argv arguments
*/

void KinoCommon::bulkLoad(int argc, char* argv[]) {

    for (int i = 1; i < argc; ++i) {
        char *filename = argv[i];
		save_cd( filename );

     	switch ( checkFile( filename ) ) {
			case AVI:
                loadAVI(filename, this->getPlayList()->GetNumFrames());
                if (g_currentFrame == -1) g_currentFrame = 0;
				this->hasListChanged = TRUE;
				break;
			case PLAYLIST:
                loadPlayList(filename, this->getPlayList()->GetNumFrames());
                if (g_currentFrame == -1) g_currentFrame = 0;
				this->hasListChanged = TRUE;
				break;
			case UNKNOWN_FORMAT:
/* DRD> I had to comment out the following to prevent lockups with 
  invalid args. When Xv crashes my desktop, and I log back in, Kino is
  started automatically by GNOME which passes some weird --smclient arg */
//				modal_message( "Invalid file specified." );
				break;
		}
    }
}

/** Save the specified frame to the specified file.

  	\param	abs_frame_num the frame to save
	\param	file the file to save it to
*/

void KinoCommon::saveFrame(int abs_frame_num, char *file) {

    Frame           frame;
    unsigned char   pixels[720*576*4];
    this->getPlayList()->GetFrame(abs_frame_num, frame);
    bool isPAL = frame.IsPAL();
    frame.ExtractRGB(pixels);
    GdkImlibImage *im=gdk_imlib_create_image_from_data
                      (pixels, NULL, 720, isPAL ? 576 : 480);
    gdk_imlib_save_image(im, file, NULL);
}

/** Generate the Scene thumbnail icon list.
*/
void composite( unsigned char *src, unsigned char *dest, int off_x, int off_y, 
	int src_width, int src_height, int dest_width)
{
	register int i;
	register unsigned char *p_src, *p_dest;
	p_dest = dest + (((off_y * dest_width) + off_x) * 3);
	p_src = src;
	for (i = 0; i < src_height; i++ ) {
		memcpy( p_dest, p_src, src_width*3);
		p_dest += dest_width*3;
		p_src += src_width*3;
	}
}

// Temporary(?) static variable to get round stack corruption(?) issue.
static unsigned char pixels[720*576*4];

#ifdef MULTITHREADED

enum SceneStatus {
	SCENE_IDLE,
	SCENE_STARTING,
	SCENE_RUNNING,
	SCENE_RESTART
};

enum SceneStatus showScenesRunning = SCENE_IDLE;

void removeImages( vector< GdkImlibImage * >** list ) {
	if ( *list != NULL ) {
		vector< GdkImlibImage * >::iterator iter;
		for ( iter = (*list)->begin(); iter != (*list)->end(); iter++) {
			gdk_imlib_destroy_image( *iter );
		}
		(*list)->erase( (*list)->begin(), (*list)->end() );
		delete (*list);
		*list = NULL;
	}
}

void *showScenesThread( void *arg ) {
	static Frame frame;
	GdkImlibImage *image = NULL;
	GdkImlibImage *thumbnail = NULL;
	static vector< GdkImlibImage * > *icons = NULL;
	static vector< GdkImlibImage * > *backup = NULL;
	vector <int> scene;
	int frameNum = 0;
	int pos = 0;
	int icon_height = 0;
	KinoCommon *common = static_cast<KinoCommon *>(arg);

	// This line affects the quality of ALL frames - global variable somewhere down in
	// libdv perhaps? Removed it.
	//frame.decoder->quality = DV_QUALITY_COLOR | DV_QUALITY_DC;
	
	while( showScenesRunning != SCENE_IDLE ) {

		// Apparently(?) the thumbs generated cannot be removed while they're used in the
		// icon list - this doesn't appear to be the case for the timeline page, but is
		// definitely causing problems here - could be the modified images causing an issue
		// perhaps? Anyway, to avoid a blank list during the execution of this thread, take
		// a backup of the previous list and create a new one (see comments below).

		icons = new vector< GdkImlibImage *>;

		// Change from SCENE_STARTING to SCENE_RUNNING 
		showScenesRunning = SCENE_RUNNING;

		// Get the scene list (contains the frame number first frame of each scene)
		scene = common->getPageEditor()->GetScene();

		// Create the icons vector but terminate if the running state changes
		for ( unsigned int i = 0; showScenesRunning == SCENE_RUNNING && i < scene.size(); ++i ) {

			frameNum = i == 0 ? 0 : scene[ i - 1 ];

			common->getPlayList()->GetFrame(frameNum, frame);

			frame.ExtractRGB(pixels);

			// Get the image
			image = gdk_imlib_create_image_from_data( pixels, NULL, 720, frame.IsPALfromHeader() ? 576 :480 );
		
			// create the thumbnail
			// the thumbnail width is 69, so determine the correct height with aspect ratio
			// TODO: apply PAL aspect ration factor
			icon_height = ( frame.IsPALfromHeader() ? 576 :480)/ (720/69);

			// maximum height of thumbnail is 55
			if (icon_height > 55) icon_height = 55;
			thumbnail = gdk_imlib_clone_scaled_image( image, 69, icon_height );
			icons->push_back( gdk_imlib_load_image( KINO_PIXMAPSDIR "/film.xpm" ) );

			// composite the frame thumbnail inside the filmcell image
			// 15 is the left offset inside the film cell image 
			// 4 is the top offset inside the film cell image 
			// the maximum height of the thumbnail area in the film cell image is 47
			//   therefore, add additional offset on the top to center the thumbnail vertically
			// 69 is the width of the thumbnail 
			// 99  is the width of the filmcell image
			composite( thumbnail->rgb_data, (*icons)[i]->rgb_data, 15, 4 + (55 - icon_height)/2, 69, icon_height, 99);
		
			// TODO: render scene name/number into image
		
			// Clean up and inform imlib of changes
			gdk_imlib_destroy_image( image );
			gdk_imlib_destroy_image( thumbnail );
			gdk_imlib_changed_image( (*icons)[i] );
		}

		// Render the icons if we're still running, otherwise remove the [partial] list generated
		// NB: The freeze followed by a thread_leave causes the icon list to be blanked on the screen and
		// removing the current list before clearing the iconlist causes possible core dumps if a redraw
		// is required before the new list is in place - hence the backup keeps the last used list until
		// we're sure that we can safely remove them

		if ( showScenesRunning == SCENE_RUNNING ) {
			gdk_threads_enter();
			gnome_icon_list_freeze( common->getSceneStrip() );
			gnome_icon_list_clear( common->getSceneStrip() );
			for (unsigned int i = 0; showScenesRunning == SCENE_RUNNING && i < scene.size(); ++i) {
				// TODO: put the scene name in the data and let the user search for it with the / command
				pos = gnome_icon_list_append_imlib( common->getSceneStrip(), (*icons)[i], "" );
   			}
			gnome_icon_list_thaw( common->getSceneStrip() );
			gdk_threads_leave();
			// Clean up temporary backup now
			removeImages( &backup );
			// Now backup
			backup = icons;
		}
		else {
			// Remove any unused thumbs in icons
			removeImages( &icons );
		}

		// If we're still running now, it's OK to idle again.
		if ( showScenesRunning == SCENE_RUNNING ) 
			showScenesRunning = SCENE_IDLE;

	}

	return NULL;
}

void KinoCommon::showScenes( ) {
	static pthread_t scenes_thread;
	static pthread_mutex_t mutex = PTHREAD_MUTEX_INITIALIZER;

	pthread_mutex_lock( &mutex );

	// If running then, set as RESTART otherwise start the thread,
	// otherwise we're already waiting on a restart so do nothing
	if ( showScenesRunning == SCENE_RUNNING ) {
		showScenesRunning = SCENE_RESTART;
	}
	else if ( showScenesRunning == SCENE_IDLE ) {
		pthread_join( scenes_thread, NULL );
		showScenesRunning = SCENE_STARTING;
		pthread_create( &scenes_thread, NULL, showScenesThread, this );
	}

	pthread_mutex_unlock( &mutex );
}

#else

// Non-threaded implementation - Surely this isn't needed??? 
//

void KinoCommon::showScenes() {

	static Frame frame;
	GdkImlibImage *image = NULL;
	GdkImlibImage *thumbnail = NULL;
	vector< GdkImlibImage * > icons;
	vector <int> scene;
	int frameNum = 0;
	int pos = 0;
	int icon_height = 0;

	// This line affects the quality of ALL frames - global variable somewhere down in
	// libdv perhaps? Removed it.
	//frame.decoder->quality = DV_QUALITY_COLOR | DV_QUALITY_DC;
	
	// Get the scene list
	scene = this->getPageEditor()->GetScene();

	for ( unsigned int i = 0; i < scene.size(); ++i ) {
		frameNum = i == 0 ? 0 : scene[ i - 1 ];

		this->getPlayList()->GetFrame(frameNum, frame);
		frame.ExtractRGB(pixels);

		// Get the image
		image = gdk_imlib_create_image_from_data( pixels, NULL, 720, frame.IsPALfromHeader() ? 576 :480 );
	
		// create the thumbnail
		// the thumbnail width is 59, so determine the correct height with aspect ratio
		// TODO: apply PAL aspect ration factor
		icon_height = ( frame.IsPALfromHeader() ? 576 :480)/ (720/59);

		// maximum height of thumbnail is 47
		if (icon_height > 47) icon_height = 47;
		thumbnail = gdk_imlib_clone_scaled_image( image, 59, icon_height );
		icons.push_back( gdk_imlib_load_image( KINO_PIXMAPSDIR "/film.xpm" ) );

		// composite the frame thumbnail inside the filmcell image
		// 20 is the left offset inside the film cell image 
		// 16 is the top offset inside the film cell image 
		// the maximum height of the thumbnail area in the film cell image is 47
		//   therefore, add additional offset on the top to center the thumbnail vertically
		// 59 is the width of the thumbnail
		// 99  is the width of the filmcell image 
		composite( thumbnail->rgb_data, icons[i]->rgb_data, 20, 16 + (47 - icon_height)/2, 59, icon_height, 99);
		
		// TODO: render scene name/number into image
	
		// Clean up and inform imlib of changes
		gdk_imlib_changed_image( icons[i] );
		gdk_imlib_destroy_image( image );
		gdk_imlib_destroy_image( thumbnail );
	}

	gnome_icon_list_freeze( this->getSceneStrip() );
	gnome_icon_list_clear( this->getSceneStrip() );
	for (unsigned int i = 0; i < scene.size(); ++i) {
		pos = gnome_icon_list_append_imlib( this->getSceneStrip(), icons[i], "" );
		// TODO: put the scene name in the data and let the user search for it with the / command
   	}
	gnome_icon_list_thaw( this->getSceneStrip() );
	gdk_threads_leave();

	// Clean up temporary images before leaving the thread regarless of running state
	vector< GdkImlibImage * >::iterator iter;
	for ( iter = icons.begin(); iter != icons.end(); iter++) {
		gdk_imlib_destroy_image( *iter );
	}
	icons.erase( icons.begin(), icons.end() );
}

#endif 

/** Relay the current command and any message associated to it.

  	\param cmd command
	\param msg message;
*/

void KinoCommon::keyboardFeedback(char *cmd, char *msg) {
	char s[256];
	
	strcpy(s, cmd);
	strcat(s, "   ");
	strcat(s, msg);
	
	gtk_label_set_text( command_label, s );
}

/** Move to the current frame. Corrects the frame specified to be within the
	range of the current play list and informs the currently selected page of the
	change.

	\param frame requested frame
	\return the frame actually moved to
*/

int KinoCommon::moveToFrame( ) {
	return moveToFrame( g_currentFrame );
}

/** Move to the frame specified. Corrects the frame specified to be within the
	range of the current play list and informs the currently selected page of the
	change.

	\param frame requested frame
	\return the frame actually moved to
*/

int KinoCommon::moveToFrame( int frame ) {
	if ( frame >= 0 && frame < getPlayList()->GetNumFrames() )
		g_currentFrame = frame;
	else if ( frame >= getPlayList()->GetNumFrames() )
		g_currentFrame = getPlayList()->GetNumFrames() - 1;
	else if ( frame < 0 && getPlayList()->GetNumFrames() > 0 )
		g_currentFrame = 0;
	else
		g_currentFrame = -1;

	if ( this->hasListChanged ) {
		getCurrentPage()->movedToFrame( g_currentFrame );
		showScenes();
	}
	else {
		getCurrentPage()->movedToFrame( g_currentFrame );
	}

	return g_currentFrame;
}

/** Move the frame relative to the number specified. 

	\param frames requested frame offset
	\return the frame actually moved to
*/

int KinoCommon::moveByFrames( int frames ) {
	if (frames == 0) return g_currentFrame;
	int frame = g_currentFrame + frames;
	return moveToFrame( frame );
}

/** Relay the current frame info to the GUI.

	\param i current frame
	\param frame the frame object
*/

void KinoCommon::showFrameInfo( int i ) {
	if ( showInfoFlag )
		getCurrentPage()->showFrameInfo( i );
}

void KinoCommon::toggleShowInfo() {
	showInfoFlag = !showInfoFlag;
	if ( showInfoFlag ) {
		showFrameInfo( g_currentFrame );
	}
	else {
		gtk_label_set_text( position_label, "-\n-\n-" );
	}
}

/** Trigger the start action of the current page. This method is also responsible
	for determining the state of the main page buttons and widgets.
*/

void KinoCommon::start() {
	getCurrentPage()->start();
	activateWidgets();

	toggleComponents( getComponentState(), false);
	component_enum pattern = (component_enum) ( getCurrentPage()->activate() ^ getCurrentPage()->deactivate() );
	if ( pattern & VIDEO_STOP) toggleComponents( VIDEO_STOP, true);
	commitComponentState();
}

/** Trigger the clean action of the current page.
*/

void KinoCommon::clean() {
	getCurrentPage()->clean();
}

/** Trigger the keyboard action of the current page.
*/

gboolean KinoCommon::processKeyboard( GdkEventKey *event ) {
	return getCurrentPage()->processKeyboard( event );
}

/** Trigger the command line action of the current page.
*/

gboolean KinoCommon::processCommandLine( char *cmd ) {
	return getCurrentPage()->processCommandLine( cmd);
}

/** Trigger a menu command action of the current page. These can be
	either keyboard or command line style strings.
*/

gboolean KinoCommon::processMenuCommand( char *cmd ) {
	return getCurrentPage()->processMenuCommand( cmd );
}

/** Trigger the select scene action of the current page.
*/

void KinoCommon::selectScene( int scene ) {
	getCurrentPage()->selectScene( scene );
}

/** Trigger the start of movie action of the current page.
*/

void KinoCommon::videoStartOfMovie() {
	if (! is_component_state_changing ) {

		getCurrentPage()->videoStartOfMovie();

		commitComponentState();
	}
}

/** Trigger the previous scene action of the current page.
*/

void KinoCommon::videoPreviousScene() {
	if (! is_component_state_changing ) {
		
		getCurrentPage()->videoPreviousScene();

		commitComponentState();
	}
}

/** Trigger the start of scene action of the current page.
*/

void KinoCommon::videoStartOfScene() {
	if (! is_component_state_changing ) {
		
		getCurrentPage()->videoStartOfScene();

		commitComponentState();
	}
}

/** Trigger the rewind action of the current page.
*/

void KinoCommon::videoRewind() {
	if (! is_component_state_changing ) {
		
		getCurrentPage()->videoRewind();

		commitComponentState();
	}
}

/** Trigger the back action of the current page.
*/

void KinoCommon::videoBack() {
	if (! is_component_state_changing ) {
		
		getCurrentPage()->videoBack();

		commitComponentState();
	}
}

/** Trigger the play action of the current page.
*/

void KinoCommon::videoPlay() {
	if (! is_component_state_changing ) {
		
		getCurrentPage()->videoPlay();

		commitComponentState();
	}
}

/** Trigger the forward action of the current page.
*/

void KinoCommon::videoForward() {
	if (! is_component_state_changing ) {
		
		getCurrentPage()->videoForward();

		commitComponentState();
	}
}

/** Trigger the fast forward action of the current page.
*/

void KinoCommon::videoFastForward() {
	if (! is_component_state_changing ) {
		
		getCurrentPage()->videoFastForward();

		commitComponentState();
	}
}

/** Trigger the next scene action of the current page.
*/

void KinoCommon::videoNextScene() {
	if (! is_component_state_changing ) {
		
		getCurrentPage()->videoNextScene();

		commitComponentState();
	}
}

/** Trigger the end of scene action of the current page.
*/

void KinoCommon::videoEndOfScene() {
	if (! is_component_state_changing ) {
		
		getCurrentPage()->videoEndOfScene();

		commitComponentState();
	}
}

/** Trigger the end movie action of the current page.
*/

void KinoCommon::videoEndOfMovie() {
	if (! is_component_state_changing ) {
		
		getCurrentPage()->videoEndOfMovie();

		commitComponentState();
	}
}

/** Trigger the pause action of the current page.
*/

void KinoCommon::videoPause() {
	if (! is_component_state_changing ) {
		
		getCurrentPage()->videoPause();

		commitComponentState();
	}
}

/** Trigger the stop action of the current page.
*/

void KinoCommon::videoStop() {
	if (! is_component_state_changing ) {
		
		getCurrentPage()->videoStop();

		commitComponentState();
	}
}

/** Bi-directional variable-speed playback 

	\param angle A number from -7 (fastest reverse) to 7 (fastest forward)
*/

void KinoCommon::videoShuttle( int angle ) {
	if (! is_component_state_changing ) {
		toggleComponents( getComponentState(), false);

		GtkAdjustment *adjust = gtk_range_get_adjustment( this->video_shuttle );
		gtk_adjustment_set_value( adjust, angle );
		getCurrentPage()->videoShuttle( angle );
		
		commitComponentState();
	}
}

void KinoCommon::windowChanged() {
	getCurrentPage()->windowChanged();
}

void KinoCommon::windowMoved() {
	getCurrentPage()->windowMoved();
}

void KinoCommon::visibilityChanged( gboolean visible ) {
	getCurrentPage()->visibilityChanged( visible );
}

/** Activate or deactivate the widgets at the request of the previous page.
    This method is always called immediately after the current pages start method
	and can be called at the discretion of the page if required.
*/

void KinoCommon::activateWidgets() {
	component_enum pattern = (component_enum)( getCurrentPage()->activate() ^ getCurrentPage()->deactivate() );
	gtk_widget_set_sensitive( GTK_WIDGET( edit_menu ), pattern & EDIT_MENU );
	gtk_widget_set_sensitive( lookup_widget(widget, "button_cut"), pattern & EDIT_MENU );
	gtk_widget_set_sensitive( lookup_widget(widget, "button_copy"), pattern & EDIT_MENU );
	gtk_widget_set_sensitive( lookup_widget(widget, "button_paste"), pattern & EDIT_MENU );
	gtk_widget_set_sensitive( GTK_WIDGET( command_menu ), pattern & COMMAND_MENU );
	gtk_widget_set_sensitive( lookup_widget(widget, "button_insert"), pattern & COMMAND_MENU );
	gtk_widget_set_sensitive( lookup_widget(widget, "button_append"), pattern & COMMAND_MENU );
	gtk_widget_set_sensitive( lookup_widget(widget, "button_split"), pattern & COMMAND_MENU );
	gtk_widget_set_sensitive( GTK_WIDGET( scenes ), pattern & SCENE_LIST );
	gtk_widget_set_sensitive( GTK_WIDGET( video_start_movie_button ), pattern & VIDEO_START_OF_MOVIE );
	gtk_widget_set_sensitive( GTK_WIDGET( video_start_scene_button ), pattern & VIDEO_START_OF_SCENE );
	gtk_widget_set_sensitive( GTK_WIDGET( video_rewind_button ), pattern & VIDEO_REWIND );
	gtk_widget_set_sensitive( GTK_WIDGET( video_back_button ), pattern & VIDEO_BACK );
	gtk_widget_set_sensitive( GTK_WIDGET( video_play_button ), pattern & VIDEO_PLAY );
	gtk_widget_set_sensitive( GTK_WIDGET( video_pause_button ), pattern & VIDEO_PAUSE );
	gtk_widget_set_sensitive( GTK_WIDGET( video_stop_button ), pattern & VIDEO_STOP );
	gtk_widget_set_sensitive( GTK_WIDGET( video_forward_button ), pattern & VIDEO_FORWARD );
	gtk_widget_set_sensitive( GTK_WIDGET( video_fast_forward_button ), pattern & VIDEO_FAST_FORWARD );
	gtk_widget_set_sensitive( GTK_WIDGET( video_end_scene_button ), pattern & VIDEO_NEXT_SCENE );
	gtk_widget_set_sensitive( GTK_WIDGET( video_end_movie_button ), pattern & VIDEO_END_OF_MOVIE );
	gtk_widget_set_sensitive( GTK_WIDGET( video_shuttle ), pattern & VIDEO_SHUTTLE );
	gtk_widget_set_sensitive( GTK_WIDGET( command ), pattern & VIDEO_COMMAND );
}


/** Set the state of toggle buttons.

	The state of the buttons are retained in memory, and
	this method will set a flag that the state has changed (dirty),
	but has not yet been committed. The button commands will
	not respond while the state is dirty. Use commitComponentState()
	to commit the changes and let the buttons issue their commands once again.

    \param pattern A set of component_enums to toggle
    \param state If true then lower the button, else raise the button
*/

void KinoCommon::toggleComponents(component_enum pattern, bool state) {

	this->is_component_state_changing = true;

	if (state)
		this->component_state |= pattern;
	else
		this->component_state ^= pattern;
	
	if (pattern & VIDEO_START_OF_MOVIE)
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( video_start_movie_button ), state );
	if (pattern & VIDEO_START_OF_SCENE)
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( video_start_scene_button ), state );
	if (pattern & VIDEO_REWIND)
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( video_rewind_button ), state );
	if (pattern & VIDEO_BACK)
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( video_back_button ), state );
	if (pattern & VIDEO_PLAY )
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( video_play_button ), state );
	if (pattern & VIDEO_PAUSE)
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( video_pause_button ), state );
	if (pattern & VIDEO_STOP)
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( video_stop_button ), state );
	if (pattern & VIDEO_FORWARD)
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( video_forward_button ), state );
	if (pattern & VIDEO_FAST_FORWARD)
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( video_fast_forward_button ), state );
	if (pattern & VIDEO_NEXT_SCENE)
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( video_end_scene_button ), state );
	if (pattern & VIDEO_END_OF_MOVIE)
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON( video_end_movie_button ), state );
	
}

/** Get the current component state data
*/
component_enum KinoCommon::getComponentState() {
	return (component_enum) this->component_state;
}

/** Set the component state data and commit it. 

    You can set component state at the same time too.
    See the toggleComponents() method.
	
    \param pattern A set of component_enums to set true
*/
void KinoCommon::commitComponentState( component_enum pattern )
{
	this->component_state |= pattern;
	this->is_component_state_changing = false;
}


/** Resize the video preview area to 50 percent of source image size.
*/
void KinoCommon::view50percent()
{
	this->getCurrentPage()->view50percent();
}

/** Resize the video preview area to 100 percent of source image size.
*/
void KinoCommon::view100percent()
{
	this->getCurrentPage()->view100percent();
}


/** Load the splash image into the video preview area.

    \param widget the GtkDrawingArea into which to render the image.
*/
void KinoCommon::loadSplash( GtkDrawingArea *drawable ) {
	unsigned char *background;
	int width = ((GtkWidget*)drawable)->allocation.width;
	int height = ((GtkWidget*)drawable)->allocation.height;

	background = (unsigned char *) malloc( width * height * 3 );
	fail_null( background );
	memset( background, 0, width * height * 3 );
	GdkImlibImage *bgImage = gdk_imlib_create_image_from_data( background, NULL, width, height );
	fail_null( bgImage );
	GdkImlibImage *splash = gdk_imlib_load_image( KINO_PIXMAPSDIR "/kino.jpeg" );
	fail_null( splash );
	AspectRatioCalculator calc( width, height, splash->rgb_width, splash->rgb_height, splash->rgb_width, splash->rgb_height);
	GdkImlibImage *image= gdk_imlib_clone_scaled_image( splash, calc.width, calc.height );
	fail_null( image );

	composite( image->rgb_data, bgImage->rgb_data,
		calc.x, calc.y,
		image->rgb_width, image->rgb_height, width );

	gdk_imlib_changed_image( bgImage );

	gdk_imlib_apply_image( bgImage, ( (GtkWidget *)drawable )->window );

	gdk_imlib_destroy_image( bgImage );
	gdk_imlib_destroy_image( splash );
	gdk_imlib_destroy_image( image );
	
	free( background );
}


/*
 * Callbacks
 */

extern "C" {

	gboolean
	on_open_movie_dialog_delete_event (GtkWidget *widget,
                                   GdkEvent *event,
                                   gpointer user_data)
	{
    	GtkWidget *dialog = gtk_widget_get_toplevel(widget);
    	gtk_file_selection_set_filename(GTK_FILE_SELECTION(dialog), "");
		gtk_main_quit();
    	return FALSE;
	}


	void
	on_open_movie_dialog_ok_button_clicked (GtkButton *button,
                                        	gpointer user_data)
	{
		gtk_main_quit();
	}


	void
	on_open_movie_dialog_cancel_button_clicked
	(GtkButton *button,
 	gpointer user_data)
	{
    	GtkWidget *dialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
    	gtk_file_selection_set_filename(GTK_FILE_SELECTION(dialog), "");
		gtk_main_quit();
	}
	
	
	gboolean
	on_save_movie_dialog_delete_event (GtkWidget *widget,
                                   	GdkEvent *event,
                                   	gpointer user_data)
	{
    	GtkWidget *dialog = gtk_widget_get_toplevel(widget);
    	gtk_file_selection_set_filename(GTK_FILE_SELECTION(dialog), "");
		gtk_main_quit();
    	return FALSE;
	}
	
	
	void
	on_save_movie_dialog_ok_button_clicked (GtkButton *button,
                                        	gpointer user_data)
	{
		gtk_main_quit();
	}
	
	
	void
	on_save_movie_dialog_cancel_button_clicked
	(GtkButton *button,
 	gpointer user_data)
	{
    	GtkWidget *dialog = gtk_widget_get_toplevel(GTK_WIDGET( button ));
    	gtk_file_selection_set_filename(GTK_FILE_SELECTION(dialog), "");
		gtk_main_quit();
	}
	
	
	gboolean
	on_save_frame_dialog_delete_event      (GtkWidget       *widget,
                                        	GdkEvent        *event,
                                        	gpointer         user_data)
	{
    	GtkWidget *dialog = gtk_widget_get_toplevel(widget);
    	gtk_file_selection_set_filename(GTK_FILE_SELECTION(dialog), "");
		gtk_main_quit();
    	return FALSE;
	}
	
	
	void
	on_save_frame_dialog_ok_button_clicked (GtkButton       *button,
                                        gpointer         user_data)
	{
		gtk_main_quit();
	}


	void
	on_save_frame_dialog_cancel_button_clicked
	(GtkButton       *button,
 	gpointer         user_data)
	{
    	GtkWidget *dialog = gtk_widget_get_toplevel(GTK_WIDGET( button ));
    	gtk_file_selection_set_filename(GTK_FILE_SELECTION(dialog), "");
		gtk_main_quit();
	}
}
