/* alpha.h by John Leuner
   990906

This file describes the function calls needed to send and receive data from the Virtual Disk Server (ALPHA).
   
The public calls that Alpha exports are:

void ALPHA_initialize();
int ALPHA_write_non_blocking(struct page_entry* stpePage);
int ALPHA_read_blocking(long long address, void* buf, int buf_len);

The initialize call starts up the threads, intializes the queues (empty) and creates some sockets for talking to the Virtual Disk Server.
 
The write call is asynchronous, it simply sends the page off to the Virtual Disk Server and records an entry on its "send queue" to be able to keep track of the acknowledgement from the Server.

The read call is synchronous. It sends a request for the page to the Server and then polls an internal "receive queue", waiting for the result to appear in that queue. If it timeouts, it will resend the request and continue polling.

The private functions used internally:

Thread functions:

There are 3 threads running in Alpha:

void listen_for_alpha_read_replies(char** arg);
void listen_for_alpha_write_replies(char** arg);
void ALPHA_send_timeout(char** arg);

The first one listens for read replies from the Virtual Disk Server and puts those replies on the "receive queue". The second one listens for write replies removes the relevant item from the "send queue".

Notice the difference between the asynch and synch behaviour. The former is responsible for putting things on its queue, the latter takes things off its queue.

The last thread is there to make sure that writes don't get lost. Every so often it checks the "send queue", if there are any writes that are slightly old, it resends the write requests to the Virtual Disk Server.

Queue functions:

int ALPHA_put_item_on_read_queue(int sequence_number, char* buf, int buf_len);
int ALPHA_remove_item_from_read_queue(int sequence_number, char* buf, int* buf_len);

The former just puts a read reply onto the queue, along with the data that came with it. The latter is used to get that reply and delete it from the queue.


Constants used for communicating with the Virtual Disk Server
 */

/*Constants*/
#define ALPHA_WRITE 241        //Signifies a write request
#define ALPHA_READ 240         //Signifies a read request

#define ALPHA_NACK 245         //Negative Acknowledgement for a read failure
#define ALPHA_ACK 244          //Positive Acknowledgement of a write
#define ALPHA_DATA 243         //Positive reply, with data, for a read

//The port that we listen on for read replies from the VDS, we listen for writes on the same port + 1
#define ALPHA_listen_port 26787


//The port that the Virtual Disk Server is running on
//This should be configurable in an rc file
#define ALPHA_PORT 23559       

//The size of the header for a write request
//6 Integers and a long
#define WRITE_HEADER_SIZE (6 * 4 + 8)  

//The size of the header for a read request
//6 Integers and a long
#define READ_HEADER_SIZE (6 * 4 + 8)

//The size of the header for a write acknowledgement
//5 Integers 
#define ACK_HEADER_SIZE (5 * 4)

//The size of the header for a data reply (to a read)
//6 Integers 
#define DATA_HEADER_SIZE (6 * 4)

/* The number of times we loop before resending a request */
#define ALPHA_READ_TIMEOUT 2
#define ALPHA_SEND_TIMEOUT 2

/* Data structures for the queues */

/* Structure for the read queue */
struct listentry {
  int sequence_number;  //Each request to ALPHA has a unique sequence number, the reply will have the same sequence
  char* data; //The data that came with the reply
  int data_length; //data length
  struct listentry* next;
};

/* Structure for the send queue*/

struct send_queue_entry {
  int sequence_number; //as above
  struct page_entry* stpePage; //a reference to the "page" structure, see eta.h
  int timeout; //How many times has this page timed out?
  struct send_queue_entry* next;
};



/* We need the page_entry structure from eta.h */
#include "eta.h"


/* Functions for interacting with alpha */

void ALPHA_initialize();
int ALPHA_write_non_blocking(struct page_entry* stpePage);
int ALPHA_read_blocking(long long address, void* buf, int buf_len);

/* Private functions for our threads */

void listen_for_alpha_read_replies(char** arg);
void listen_for_alpha_write_replies(char** arg);
void ALPHA_send_timeout(char** arg);

/* Private queue functions */
int ALPHA_put_item_on_read_queue(int sequence_number, char* buf, int buf_len);
int ALPHA_remove_item_from_read_queue(int sequence_number, char* buf, int* buf_len);

// The send queue is managed inside the other methods, there are no put/remove queue methods






