/*
 * Copyright (C) 2001, John Leuner.
 *
 * This file is part of the kissme/teaseme project, which in turn is part of the JOS project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2,
 * or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>
#include "wrappers.h"
#include "stdtypes.h"
#include "thread.h"
#include "jni.h"
#include "locks.h"

#include "interp_methods.h"
#include "newobject.h"

/* This file contains a new lock implementation for kissme/teaseme
 
It is based on Roger Bacon's thin locks and some ideas from IBM's work on their JDK as well as a paper by Etienne Gagnon on SableVM.
 
Status 15/01
	- atomic enter done for unlocked and same thread cases
	- atomic enter not done for contention case
	- exit for single lock and recursive cases done
	- exit for foreign owner not done
 
 
When a thread tries to acquire a lock an object and it sees that the lock is already owned by another thread (and is thin), it should 
 
1) find the owning thread
2) acquire the mutex for that thread
3) set the contention bit for that thread
3.1) rechecks the lock
 5) if set, adds the tuple <this thread, this object, cond var> to the owning threads list
 5.1) then sleep on the cond var
 6) else check if the lock is thin
  7) if so go back to 1)
  8) else acquire the fat lock
 
When a thread is about to release a thin lock, it should check its contention bit. If it is set:
 
1) It should aquire its own contention mutex
1.1) Inflate the lock to a fat lock
2) For each tuple in the list, if the object matches the current object
 3) Signal that cond var in the tuple
 4) else if it's a different object, inflate that lock and signal the thread
4) Clear the waiting list
5) Clear contention bit
6) Release its contention lock
 
24/01/2001 - Creating a separate structure that holds fat locks
	     This means that the lock word will now hold 1 bit indicating whether it is a fat or thin lock
	   0 indicates a fat lock or no lock
	   1 indicates a thin lock
	   Then the last 8 bits are a recursion count in the case of a thin lock
	   The next 23 bits are a thread identifier in the thin lock case
 
	   If it is a fat lock, the next 31 bits are a pointer to the fat monitor.
	   The fat monitor's address must be word aligned
*/



#ifdef KISSME_LINUX_USER
#include <stdio.h>
#include <pthread.h>
#include <stdlib.h>
#include <assert.h>
#include <time.h>
#endif

#include <errno.h>

#ifdef SABLE_LOCKS

#include "threads_native.h"
#include "interp.h"

#ifndef TEASEME
int pthread_mutexattr_setkind_np (pthread_mutexattr_t* kind, int tkind);
#endif

/* Structure of the lock word
 *
 * Thin lock case, 
 * T is 1 (1 bit)
 * I is the thread identifier (23 bits)
 * R is the recursion count (8 bits)
 *
 * bit 31                         bit 0
 * |--------------------------------|
 * |TIIIIIIIIIIIIIIIIIIIIIIIRRRRRRRR|
 * |--------------------------------|
 *
 * Fat lock case
 *
 * T is 0 (1 bit)
 * P is the pointer to the fat lock struct >> 1 (31 bits)
 *
 * bit 31                        bit 0
 * |--------------------------------|
 * |TPPPPPPPPPPPPPPPPPPPPPPPPPPPPPPP|
 * |--------------------------------|
 *
 * If the whole word is 0, there is no lock
 *
 */

void THREAD_SynchroniseEnter (tOBREF pstObject	/* @parm Object to lock */ )
{
 int before;
    int temp_word;

    //set high bit to 1 (thin lock), set the next 23 bits to the thread identifier, and set the recursion count to 1

    temp_word = (0x80000000 | (((sys_current_thread() << 8) & 0x01ffff00))) | (0x1);

#ifdef DEBUG_LOCKS
    eprintf ("Before lock it is %x and temp_word is %x\n",
             ODEREF (pstObject)->lock_word, temp_word);
#endif
    before = ODEREF (pstObject)->lock_word;

    if (before == 0)		//object is unlocked
      {
	#ifdef MIPSEL
//	__asm__ __volatile__ (".set noat ; add $0, %0, $1 ; \n":"=r" (temp_word));
	#else
            //set eax to 0
            //compare ecx to eax, if they are the same, then copy ecx (temp_word) to lock_word
            __asm__ __volatile__ ("movl $0x0, %%eax ; \n"
                                  "movl %1, %%ecx ; \n"
                      " cmpxchg %%ecx,%0 ; \n":"=r" (ODEREF(pstObject)->lock_word):"r"
                                          (temp_word),
                                          "0" (ODEREF (pstObject)->lock_word):"ax", "cx");
	#endif
#ifdef DEBUG_LOCKS
            eprintf ("Thread %x locked object %p (%s), Lock word is now %x\n",  pthread_self(), pstObject, ODEREF(pstObject)->pstType->uidName, ODEREF(pstObject)->lock_word);
#endif
        }
    else
{
            //check if it is a fat or thin lock
            if((ODEREF (pstObject)->lock_word) < 0)
                {
                    //this is a thin lock
                    if ((ODEREF (pstObject)->lock_word & 0x01ffff00) == ((sys_current_thread() << 8) & 0x01ffff00))
                        {
                            //we own the lock
                            //so we just increment the count
                            if ((before & 0xff) == 0xff)
                                {
                                    eprintf
                                    ("-------------------------- overflow ----------------------- \n");

                                    temp_word = before;
                                }
                            else
                                {
                                    temp_word = before + 1;
                                }

	#ifdef MIPSEL
//	__asm__ __volatile__ (".set noat; add $0, %0, $1 ; \n":"=r" (temp_word));
	#else
                            __asm__ __volatile__ ("movl %3, %%eax ; \n"
                                                  "movl %1, %%ecx ; \n"
                      " cmpxchg %%ecx,%0 ; \n":"=r" (ODEREF(pstObject)->lock_word):"r"
                                                          (temp_word),
                                                          "0" (ODEREF (pstObject)->lock_word),
                                                          "r" (before):"ax", "cx");
	#endif
                            //again we must check if we succeeded
                            if (temp_word == ODEREF (pstObject)->lock_word)
                        {
#ifdef DEBUG_LOCKS
                                    eprintf
                                    ("Succeeded in doing inc atomic lock, now %x, obj %p, type %s\n",
                                     ODEREF (pstObject)->lock_word, pstObject,
                                     ODEREF (pstObject)->pstType->uidName);
#endif
                                }
                            else
                                {
                                    eprintf ("Tried to do incremental lock but failed %x\n",
                                             (int)  ODEREF (pstObject)->lock_word);
                                    assert (4 == 0);
                                }
                        }
                    else			//we aren't the owner
                        {
                            int loop = 5;
                            int oldContentionState;
                            int id = (ODEREF (pstObject)->lock_word & 0x01ffff00) >> 8;
                            tThreadNode *threadNode = THREAD_FindThreadByID (id);
                            while (threadNode == NULL && (loop-- > 0))
                                {
                                    eprintf
                                    ("Couldn't find thread node structure for thread with id %x\n",
                                     id);
                                    threadNode = THREAD_FindThreadByID (id);
                                }
#ifdef DEBUG_LOCKS
                            eprintf("Locking contention lock (in lock) %p\n", &threadNode->mutContentionLock);
#endif
                            sys_mutex_lock (threadNode->mutContentionLock);
                            oldContentionState = threadNode->i32ContentionBit;
                            threadNode->i32ContentionBit = 1;

                            if (threadNode->pstWaitList == NULL)
                                {
                                    threadNode->pstWaitList = LLIST_Create_List ();
                                }
                            //now make a tuple and add it to the wait list
                            {
                                int32 *waitStruct =
                                    (int32 *) sys_malloc (2 * sizeof (int32) +
                                                          sizeof (tCondVariable*));
                                waitStruct[0] = sys_current_thread();	
				waitStruct[1] = (int32) pstObject;
				waitStruct[2] = (int32) sys_condition_create();

                                LLIST_AddAtHead (threadNode->pstWaitList, waitStruct);
                                //and now sleep on the cond var
                                sys_condition_wait ((tCondVariable*) (waitStruct[2]), threadNode->mutContentionLock);

#ifdef DEBUG_LOCKS
                                eprintf ("was signalled (thread %x) (object %p, lock word %x)\n", sys_current_thread(), pstObject, (int) ODEREF(pstObject)->lock_word);
#endif

                                //The pthread man page says:  Before returning
                                //       to the calling thread, pthread_cond_wait re-acquires mutex
                                //       (as per pthread_lock_mutex).

                                sys_mutex_unlock (threadNode->mutContentionLock);
                            }

                            //      eprintf("Attempted to acquire lock owned by another thread, kissme/teaseme does not handle this yet, stopping ...\n");
                            //assert( 1 == 9);



                            if (ODEREF (pstObject)->lock_word < 0)	//if lock is thin
                                {
#ifdef DEBUG_LOCKS
                                    eprintf ("lock is still thin for object %p, %x\n", pstObject, ODEREF(pstObject)->lock_word);
#endif
                                    THREAD_SynchroniseEnter (pstObject);
                                    return;
                                }
                            else
                                {
                                    tFatlock* lock;
                                    int tmpword = (ODEREF(pstObject)->lock_word);
                                    lock =  (tFatlock*) (tmpword << 1);
                                    if(tmpword == 0 || tmpword < 0)
                                        {
                                            THREAD_SynchroniseEnter (pstObject);
                                            return;
                                        }
                                    else
                                        sys_mutex_lock(lock->lock ); //acquire the mutex (maybe recursively)
                                    return;
                                }
                        }
                }
            else
                {
                    //this is a fat lock
                    tFatlock* lock;
                    int tmpword = (ODEREF(pstObject)->lock_word);

#ifdef DEBUG_LOCKS
                    eprintf("is a fat lock %x %x\n", before, before << 1);
#endif
                    lock =  (tFatlock*) (tmpword << 1);
                    if(tmpword == 0 || tmpword < 0)
                        {
                            THREAD_SynchroniseEnter (pstObject);
                            return;
                        }
                    sys_mutex_lock(lock->lock ); //acquire the mutex (maybe recursively)r
                    return ;
                }
        }

    //check if we succeeded
    if (temp_word == ODEREF (pstObject)->lock_word)
        {
#ifdef DEBUG_LOCKS
            eprintf ("Succeeded in doing atomic lock %x\n",
                     ODEREF (pstObject)->lock_word);
#endif
        }
    else
        {
#ifdef DEBUG_LOCKS
            eprintf ("Failed in doing atomic lock it is %x and temp_word is %x\n",
                     ODEREF (pstObject)->lock_word, temp_word);
#endif
            //this means either we, or someone else, owns the lock
	   eprintf("The lock word is %x, expected %x\n", (int) ODEREF(pstObject)->lock_word, (int) temp_word);
            assert (2 == 3);
        }

    return;
}

/* This is called from SynchroniseExit when the current thread has its contention bit set
*
* (which means there are other threads which want this (or another) lock that this thread is holding)
*
*/

int THREAD_HandleContendedThread(tOBREF pstObject, tThreadNode* threadNode)
{
    tListIterator *iter;
#ifdef DEBUG_LOCKS
    eprintf("Contention bit set %p, lock word %x\n", pstObject, ODEREF(pstObject)->lock_word);
#endif
    if(threadNode->pstWaitList)
        {
            iter = LLIST_FindFirst (threadNode->pstWaitList);
            while (iter != NULL)
                {
                    int32 *waitStruct = (int32 *) iter->entry->payload;
                    if (waitStruct[1] == (int) pstObject)
                        {
#ifdef DEBUG_LOCKS
                            eprintf("Signalling %p (our thread %x %i)\n", &waitStruct[2], sys_current_thread(), sys_current_thread());
#endif
                            //just signal the thread
			    sys_condition_broadcast((tCondVariable*) waitStruct[2]); //wake up all threads(
                        }
                    else
                        {
                            tOBREF pstLockedObject = (tOBREF) waitStruct[1];
                            //not only signal the thread, but inflate the lock to a fat lock
                            if (ODEREF (pstLockedObject)->lock_word < 0)
                                {
                                    int j = 0;
                                    tFatlock* lock = (tFatlock*) sys_malloc( sizeof(tFatlock));
                                    int rec =
                                     ODEREF (pstLockedObject)->lock_word & 0xff;
                                    lock->lock = sys_mutex_create(1);

                                    for (j = 0; j < (rec - 1); j++)  //make sure the recursion count stays the same for the lock
                                        {
                                            sys_mutex_lock (lock->lock);
                                        }
                                    ODEREF(pstObject)->lock_word = ((int32) lock) >> 1;
                                }
			    sys_condition_broadcast((tCondVariable*) waitStruct[2]); //wake up all threads

                        }
                    iter = LLIST_FindNext (threadNode->pstWaitList, iter);
                } //while iter != NULL
            //		  sys_free (iter->entry);
            LLIST_DestroyList (threadNode->pstWaitList);
            threadNode->pstWaitList = NULL;
        }
    threadNode->i32ContentionBit = 0;
    ODEREF(pstObject)->lock_word = 0;
    return 0;
}

void THREAD_SynchroniseExit (tOBREF pstObject	/* @parm Object to release */  )
{
    if (ODEREF (pstObject)->lock_word == 0)
        {
	  eprintf("Object %p, type %s is not locked in SynchroniseExit!\n", pstObject, ODEREF(pstObject)->pstType->uidName);
            assert (1 == 2);		//this object should be locked!
        }

    if(ODEREF(pstObject)->lock_word > 0)
        {
            int before = ODEREF(pstObject)->lock_word;
            tFatlock* lock = (tFatlock*) (before << 1);
#ifdef DEBUG_LOCKS
            eprintf("is a fat lock in unlock\n");
#endif
            sys_mutex_unlock(lock->lock );
        }
    else
        {
            //we are the owner
            if ((ODEREF (pstObject)->lock_word & 0x01ffff00) ==
                    ((sys_current_thread () << 8) & 0x01ffff00))
	      {
		if ((ODEREF (pstObject)->lock_word & 0xff) == 1)	//this must go back to unlocked
		  {
		    tThreadNode *threadNode = THREAD_FindThread ();
		    assert(threadNode);
		    sys_mutex_lock(threadNode->mutContentionLock);
		    if (threadNode->i32ContentionBit)
		      {
			THREAD_HandleContendedThread(pstObject, threadNode);
		      }
		    else 
		      {
			ODEREF (pstObject)->lock_word = 0; 			
		      }
		    sys_mutex_unlock(threadNode->mutContentionLock);
		  }
		else
		  {
		    int prev_temp_word;
		    int temp_word;
		    prev_temp_word = (ODEREF (pstObject)->lock_word);
		    temp_word = (ODEREF (pstObject)->lock_word) - 1;
		    ODEREF (pstObject)->lock_word = temp_word; //since we own the lock, this doesn't have to be atomic
		   }
		}
	       else
		{
		 //we are not the owner
		 eprintf("We do not own the lock on this object\n");
		 assert( 8 == 9);
		}
	} //lock word > 0
}

int THREAD_FreeLocks()
{
    tThreadNode *threadNode = THREAD_FindThread ();
    sys_mutex_lock (threadNode->mutContentionLock);
    {
        tListIterator *iter;
        if(threadNode->pstWaitList)
            {
                iter = LLIST_FindFirst (threadNode->pstWaitList);
                while (iter != NULL)
                    {
                        int32 *waitStruct = (int32 *) iter->entry->payload;
                        //just signal the thread
                        sys_condition_signal ((tCondVariable*) waitStruct[2]);
                        iter = LLIST_FindNext (threadNode->pstWaitList, iter);
                    }
                LLIST_DestroyList (threadNode->pstWaitList);
            }
    }
    threadNode->i32ContentionBit = 0;
    threadNode->pstWaitList = NULL;
    sys_mutex_unlock(threadNode->mutContentionLock);
    return 0;
}

/*
 * @doc FUNC
 * @func
 * This procedure causes the current thread to wait on the specified object
 * either until it is notified or until its waiting time expires. If the
 * specified waiting time is 0 then it waits forever or until notified.
 *
 * The calling thread has to have obtained a lock (monitorenter) on the object before calling wait.
 *
 *
 *
 *
 */


int THREAD_WaitOnObject (JNIEnv* env, 
			 tOBREF pstObject,	/* @parm Object on which to wait */
			 jlong millis
                        )
{
#ifdef SABLE_LOCKS
      int our_id = sys_current_thread();
      tTimeSpec timeout;

      int lock_word = ODEREF (pstObject)->lock_word;
      int own_lock = 0;
#ifdef DEBUG_LOCKS
      eprintf("WaitOnObject %p\n", pstObject);
#endif
      if (((lock_word >> 8) & 0x7fff) == our_id)
            {
	      own_lock = 1;
	    }
      else if(lock_word > 0)
	{
	  tFatlock* lock = (tFatlock*) (lock_word << 1);

	  if( lock->owner == our_id)
	    own_lock = 1;
	}

      if(own_lock != 0)
	{
	  int recursion_count = 0;
	  int i = 0;
	  tThreadNode *threadNode = THREAD_FindThread ();
	  tCondVariable* cond = ODEREF(pstObject)->condVariable;
                  //this thread owns the lock on this object (as it should)

		  sys_mutex_lock(threadNode->mutContentionLock);

//we must release the java lock on this object, recording the recursion count
	recursion_count = lock_word & 0xff;
	for(i = 0; i < recursion_count;i++)
	  {
	    THREAD_SynchroniseExit( pstObject );
	  }

	timeout.tv_sec = 0; //XXX limits us
	timeout.tv_nsec = millis; 
	if(millis == 0)
	      sys_condition_wait(cond, threadNode->mutContentionLock);
	    else
	      sys_condition_timedwait(cond, threadNode->mutContentionLock,  &timeout);

	 //now reacquire the lock
	for(i = 0; i < recursion_count;i++)
	  {
	    THREAD_SynchroniseEnter( pstObject );
	  }
            sys_mutex_unlock(threadNode->mutContentionLock);
            return 0;
        }
    else
        {
            //we don't own this lock, so throw an exception
            tOBREF pstExOb = INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalMonitorStateException", "thread not owner");
            return (int) pstExOb;
        }
#else

#endif
#ifdef DEBUG_LOCKS
    eprintf("Returning NULL from WaitOnObject\n");
#endif
    return (int) NULL;
}


int THREAD_NotifyRandom (JNIEnv* env, tOBREF pstObject	/* @parm Object from whose wait list we pick a thread */
                        )
{
      return THREAD_Notify (env,pstObject, 0);	//Notify single thread
}



int THREAD_NotifyAll (JNIEnv* env, tOBREF pstObject	/* @parm Object whose wait list must be cleared */
                     )
{
      return THREAD_Notify (env, pstObject, 1);	//Notify all threads
}

int THREAD_Notify (JNIEnv* env, tOBREF pstObject, int notify_all)
{
  int own_lock = 0;
  tCondVariable* cond;
  int our_id = sys_current_thread();

#ifdef SABLE_LOCKS
    int lock_word = ODEREF (pstObject)->lock_word;
    if(lock_word < 0)
      {
	if (((lock_word >> 8) & 0x7fff) == our_id)
	  {
            //this thread owns the lock on this object (as it should)
		own_lock = 1;
	  }
      }
    else if(lock_word > 0)
      {
	tFatlock* lock = (tFatlock*) (lock_word << 1);
	if( lock->owner == our_id)
	  own_lock = 1;
      }

    if(own_lock == 0)
      {
    //no one owns this lock or someone else does
    tOBREF pstExOb = INTERP_ExceptionObjectFromNameAndMessage(env, "java/lang/IllegalMonitorStateException", "thread not owner");
#ifdef DEBUG_LOCKS
	eprintf ("Calling notify on an object not locked by this thread\n");
#endif
	return (int) pstExOb;	
      }


    cond = ODEREF(pstObject)->condVariable;
    sys_condition_signal ( cond );
#endif
    return 0;
}


#endif
