// KreateCD - CD recording software for the K desktop environment
//
// 1999-2000 by Alexander Feigl <Alexander.Feigl@gmx.de>
//
// This file is subject to the terms and conditions of the GNU General
// Public License.  See the file COPYING in the main directory of the
// KreateCD distribution for more details.

//D'N'D Stuff by Joseph WENNINGER

#include "FileTree.h"
#include "IsoFile.h"

#include <qobject.h>
#include <qlistview.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <kapp.h>
#include <klocale.h>
#include <kiconloader.h>

#include <kglobal.h>
#include <kglobalsettings.h>
#include <kurldrag.h>
#include <kmessagebox.h>
#include <kurl.h>

#include "FileTree.moc"

const char *isoDragType="kreatecd/isotreeDND";

FileTreeItem::FileTreeItem(ISOFile *isofile,FileTreeItem *parent)
  :QListViewItem(parent) {
  parentItem=parent;
  setupISOFile(isofile);
}

FileTreeItem::FileTreeItem(ISOFile *isofile,FileTree *parent):
  QListViewItem(parent) {
  parentItem=0;
  setupISOFile(isofile);
}


void FileTreeItem::setupISOFile(ISOFile *isofile) {
  QPixmap pmap;
  fileObject=0;
  dirObject=0;
  dummyItem=0;

  fileList=new FileTreeItems();
  isoFile=isofile;
  isDirExpanded=false;
  isDir=false;

  if (isoFile->type()==ISOFile::ISO_RealDir) {
    fileObject=new QFileInfo(isoFile->reference()->data());
    setupDirObject();
  }

  switch (isoFile->type()) {
  case ISOFile::ISO_ISODir:
    pmap=BarIcon("kcd_folder_red");
    isDir=true;
    break;
  case ISOFile::ISO_RealDir:
    pmap=BarIcon("kcd_folder_blue");
    isDir=true;
    break;
  case ISOFile::ISO_RealFile:
    pmap=BarIcon("kcd_unknown_blue");
    break;
  }

  setText(0,isoFile->name()->data());
  setPixmap(0,pmap);

  if (isDir) {
    dummyItem=new FileTreeItem(this);
  }
}



FileTreeItem::FileTreeItem(FileTreeItem *parent):QListViewItem(parent) {
  QPixmap pmap;
  fileObject=0;
  dirObject=0;
  dummyItem=0;

  fileList=new FileTreeItems();
  isoFile=0;
  isDirExpanded=false;
  isDir=false;
  parentItem=parent;

  pmap=BarIcon("kcd_folder");
  setText(0,".");
  setPixmap(0,pmap);
}



FileTreeItem::FileTreeItem(QFileInfo *fileinfo,FileTree *parent)
  :QListViewItem(parent) {
  parentItem=0;
  setupQFileInfo(fileinfo);
}

FileTreeItem::FileTreeItem(QFileInfo *fileinfo,FileTreeItem *parent)
  :QListViewItem(parent) {
  parentItem=parent;
  setupQFileInfo(fileinfo);
}


void FileTreeItem::setupQFileInfo(QFileInfo *fileinfo) {
  QPixmap pmap;

  fileObject=new QFileInfo(*fileinfo);
  fileList=new FileTreeItems();
  isoFile=0;
  dummyItem=0;
  isDirExpanded=false;
  isDir=false;

  setupDirObject();

  if (fileObject->isDir()) {
    if (fileObject->isReadable()) {
      isDir=true;
      pmap=BarIcon("kcd_folder");
    } else {
      pmap=BarIcon("kcd_lockedfolder");
    }
  } else {
    if (fileObject->isReadable()) {
      pmap=BarIcon("kcd_unknown");
    } else {
      pmap=BarIcon("kcd_locked");
    }
  }

  if (fileObject->fileName().isEmpty()) {
    this->setText(0,fileObject->absFilePath());
  } else {
    this->setText(0,fileObject->fileName());
  }
  this->setPixmap(0,pmap);

  if (isDir) {
    dummyItem=new FileTreeItem(this);
  }
}

FileTreeItem::FileTreeItem(QString *virtualname,FileTree *parent)
  :QListViewItem(parent) {
  QPixmap pmap;

  parentItem=0;
  fileObject=0;
  dirObject=0;
  fileList=new FileTreeItems();
  isoFile=0;
  dummyItem=0;
  isDirExpanded=false;
  isDir=false;

  pmap=BarIcon("kcd_cd");

  this->setText(0,*virtualname);
  this->setPixmap(0,pmap);

  if (isDir) {
    dummyItem=new FileTreeItem(this);
  }
}


FileTreeItem::~FileTreeItem(void) {
  if (fileObject!=0) delete(fileObject);
  if (dirObject!=0) delete (dirObject);
  delete(fileList);
}

void FileTreeItem::setupDirObject(void) {
  if ( (fileObject->isDir()) && (fileObject->isReadable()) ) {
    QString filename;
    dirObject=new QDir(fileObject->absFilePath());
  } else {
    dirObject=0;
  }
}


void FileTreeItem::setOpen(bool doopen) {
  if ( (doopen) && (parentItem!=0) ) {
    parentItem->setOpen(doopen);
  }
  doopen?expandTree():collapseTree();
  QListViewItem::setOpen(doopen);
}

void FileTreeItem::expandTree(void) {
  if (!isDir) return;
  if (isDirExpanded==true) return;
  if (isoFile!=0) {
    if (isoFile->type()==ISOFile::ISO_ISODir) {
      ISOFile *walker=0;
      while ( (walker=isoFile->getChildren(walker))!=0) {
        addISOObject(walker,false,true);
      }
      dropDummy();
      isDirExpanded=true;
      return;
    }
  }
  if (dirObject==0) return;
  newFiles(dirObject->entryInfoList());
  dropDummy();
  isDirExpanded=true;
}

void FileTreeItem::dropDummy(void) {
  if (dummyItem) {
    delete dummyItem;
    dummyItem=0;
  }
}

void FileTreeItem::collapseTree(void) {
  FileTreeItem *item;

  if (!isDir) return;
  if (!fileList->isEmpty()) {
    collapseSubTree();
  }

  while ( (item=fileList->first())!=0) {
    fileList->remove(item);
    delete item;
  }
  if (!dummyItem) {
    dummyItem=new FileTreeItem(this);
  }
  isDirExpanded=false;
}

void FileTreeItem::expandSubTree(void) {
  FileTreeItem *item;
  item=fileList->first();
  while (item!=0) {
    item->expandTree();
    if (item->isOpen()) item->expandSubTree();
    item=fileList->next();
  }
}

void FileTreeItem::collapseSubTree(void) {
  FileTreeItem *item;
  item=fileList->first();
  while (item!=0)
    {
      item->collapseTree();
      item=fileList->next();
    }
}



void FileTreeItem::newFiles(const QFileInfoList *infos) {
  QFileInfoList *infolist;
  QFileInfo *info;
  FileTreeItem *item;

  infolist=new QFileInfoList(*infos);
  info=infolist->first();
  while (info!=0) {
    if ( (strcmp("..",info->fileName())==0) ||
         (strcmp(".",info->fileName())==0) ) {
      info=infolist->next();
      continue;
    }
    item=new FileTreeItem(info,this);
    addItemSorted(item);
    info=infolist->next();
  }
}


bool FileTreeItem::fileName(QString *filename) {
  if (fileObject==0) return (false);
  *filename=fileObject->absFilePath();
  return(true);
}

bool FileTreeItem::addISOObject(ISOFile *isoobj,bool addit,bool forceitem) {
  FileTreeItem *item=0;
  if (isoFile==0) return(false);
  if ( (addit) && (isoFile->findChildren(isoobj->name())!=0)) return(false);
  if (isoFile->type()!=ISOFile::ISO_ISODir) return(false);
  if ( (this->isOpen()) || forceitem) {
    item=new FileTreeItem(isoobj,this);
    addItemSorted(item);
  }
  if (addit) isoFile->addObject(isoobj);
  if ( ((this->isOpen()) || forceitem) && (item->isOpen())) item->expandTree();
  return(true);
}

void FileTreeItem::addItemSorted(FileTreeItem *item) {
  FileTreeItem *child,*ochild;
  const char *text1;
  int idx=0;
  bool preflag=false;

  child=fileList->first();
  if (child==0) {
    fileList->append(item);
    return;
  }
  text1=item->text(0);
  while (child!=0) {
    ochild=child;
    if (strcmp(text1,child->text(0))<0) {
      preflag=true;
      break;
    }
    child=fileList->next();
    ++idx;
  }
  fileList->insert(idx,item);
}


bool FileTreeItem::deleteObject(void) {
  if (isoFile!=0) {
    if (isoFile->type()==ISOFile::ISO_ISODir) {
      while (!this->fileList->isEmpty()) {
        FileTreeItem *temp;
        temp=this->fileList->first();
        if (!temp->deleteObject()) return(false);
        delete(temp);
      }
    }
  }

  if (parentItem!=0) {
    if (fileObject!=0) delete(fileObject);
    fileObject=0;
    if (dirObject!=0) delete(dirObject);
    dirObject=0;
    if (isoFile!=0) delete(isoFile);
    isoFile=0;
    parentItem->fileList->removeRef(this);
  } else {
    if (isVirtual()) {
      fileList->removeRef(this);
      return(true);
    }
    return(false);
  }
  return(true);
}


bool FileTreeItem::isVirtual(void) {
  if ( (isoFile==0) && (fileObject==0) ) return(true);
  return(false);
}

bool FileTreeItem::acceptDrop(void) {
  if (isoFile==0) return false;

  if (isoFile->type()==ISOFile::ISO_ISODir) return true;
  return false;
}

QString FileTreeItem::getPath() {
  //if (isoFile->type()==ISOFile::ISO_RealDir) qDebug("HALLO");
  //if (isoFile->type()==ISOFile::ISO_RealDir) return (QString(dirObject->absPath()));
  return (QString(fileObject->absFilePath()));
  return (QString(""));
}

FileTreeItem *FileTreeItem::parent(void) {
  return(parentItem);    
}

// main tree widget

FileTree::FileTree(QWidget *parent,const char *name,bool Source_Only)
  :QListView(parent,name) {
  this->addColumn(i18n("Filename"));
  currentItem=0;
  defaultItem=0,
  m_bDrag=false;
  sourceonly=Source_Only;
  if (!Source_Only) viewport()->setAcceptDrops(true);
  connect(this,SIGNAL(selectionChanged(QListViewItem *)),this,SLOT(changeSelect(QListViewItem *)));
  connect(this,SIGNAL(doubleClicked(QListViewItem *)),this,SLOT(doubleClick(QListViewItem *)));
}

FileTree::~FileTree(void) {
}

void FileTree::changeSelect(QListViewItem *sitem) {
  currentItem=(FileTreeItem *) sitem;
}

void FileTree::addDir(QFileInfo *finfo) {
  FileTreeItem *item;

  item=new FileTreeItem(finfo,this);
  item->setOpen(true);
}

void FileTree::addDir(ISOFile *isofile) {
  FileTreeItem *item;
  item=new FileTreeItem(isofile,this);
  item->setOpen(true);
  setSelected(item,true);
  defaultItem=item;
}

void FileTree::addVirtual(QString *virtualname) {
  FileTreeItem *item;
  item=new FileTreeItem(virtualname,this);
}


bool FileTree::getSelected(QString *fname) {
  if (currentItem==0) return(false);

  return(currentItem->fileName(fname));
}

bool FileTree::deleteObject(void) {
  FileTreeItem *ci,*parent;
  if (currentItem==0) return(false);
  ci=currentItem;
  parent=ci->parent();
  if (!ci->deleteObject()) {
    return(false);
  }
  currentItem=0;
  if (defaultItem==ci) {
    defaultItem=0;
  }
  delete(ci);
  if (parent!=0) {
    currentItem=parent;
    setSelected(parent,true);
  } else {
    if (defaultItem!=0) {
      currentItem=defaultItem;
      setSelected(defaultItem,true);
    }
  }
  return(true);
}

bool FileTree::isVirtual(void) {
  if (currentItem==0) return(false);
  if (!currentItem->isVirtual()) return(false);
  return(true);
}


bool FileTree::addRealObject(QString *fname) {
  QString complete;
  QString file;

  QFileInfo *kfi;
  ISOFile::ISOType itype;
  ISOFile *ifil;

  if (currentItem==0) return(false);
  kfi=new QFileInfo(fname->data());
  if (kfi==0) return(false);

  if (!kfi->isReadable()) {
    delete(kfi);
    return(false);
  }

  if (kfi->isDir()) {
    itype=ISOFile::ISO_RealDir;
  } else {
    itype=ISOFile::ISO_RealFile;
  }


  complete=fname->data();
  file=kfi->fileName();

  ifil=new ISOFile(itype,&file,&complete);

  currentItem->setOpen(true);
  if (!currentItem->addISOObject(ifil)) {
    delete(ifil);
    delete(kfi);
    return(false);
  }
  delete(kfi);
  return(true);
}

bool FileTree::createISODir(const char *dirname) {
  ISOFile *isofil;
  QString qs;

  if (currentItem==0) return(false);
  qs=dirname;

  isofil=new ISOFile(ISOFile::ISO_ISODir,&qs);
  currentItem->setOpen(true);
  if (!currentItem->addISOObject(isofil)) {
    delete(isofil);
    return(false);
  }
  return(true);
}

void FileTree::resizeEvent(QResizeEvent *rsiz) {
  QListView::resizeEvent(rsiz);
  setColumnWidth(0,width());
}

void FileTree::doubleClick(QListViewItem *nlist) {
  FileTreeItem *fti;
  QString fname;
  fti=(FileTreeItem *) nlist;
  fti->fileName(&fname);
  emit(addDouble(&fname));
}


void FileTree::contentsMousePressEvent ( QMouseEvent * e ) {
  if (sourceonly) {
    QPoint p(contentsToViewport(e->pos()));
    FileTreeItem *i = (FileTreeItem*)itemAt(p);

    if (e->button() == LeftButton && i) {
      qDebug("Clicked onto Item");
      m_dragPos=e->pos();
      m_bDrag=true;
    }
  }
  QListView::contentsMousePressEvent (e);
}

void FileTree::contentsMouseReleaseEvent ( QMouseEvent *e) {
  m_bDrag=false;
  QListView::contentsMouseReleaseEvent(e);
}

void FileTree::contentsMouseMoveEvent ( QMouseEvent * e ) {

  if (!m_bDrag || (e->pos()-m_dragPos).manhattanLength()
      <=KGlobalSettings::dndEventDelay()) return;

  m_bDrag=false;
  FileTreeItem *item=(FileTreeItem*)
    itemAt(contentsToViewport(m_dragPos));
  if (!item) {return;}
  qDebug("Start dragging");
  KURL url(item->getPath());
  KURL::List list;
  list.append(url);
  QUriDrag *drag=KURLDrag::newDrag(list,viewport(),0);
  const QPixmap *pxm=item->pixmap(0);
    if (pxm && drag->pixmap().isNull())
      {
        QPoint hs(pxm->width()/2,pxm->height()/2);
	drag->setPixmap(*pxm,hs);
      }
  drag->drag();
};


void FileTree::contentsDragEnterEvent(QDragEnterEvent* event) {
  event->accept(QUriDrag::canDecode(event));
  for(int i=0; event->format(i); i++)
    qDebug(event->format(i));
}

void FileTree::contentsDragMoveEvent(QDragMoveEvent* e) {
  if (!QUriDrag::canDecode(e)) {e->accept(false);return;}
  FileTreeItem *item=(FileTreeItem*)itemAt(contentsToViewport(e->pos()));
  if (!item) {
    item=(FileTreeItem*)firstChild();
  }
  setSelected(item,true);

  if (!item) {
    e->accept(false);
    return;
  }
  
  bool tmp=item->acceptDrop();

  if (!tmp) {
    if (((QListViewItem*)item)->parent())
      {item=(FileTreeItem*)(((QListViewItem*)item)->parent());
      tmp=item->acceptDrop();
      setSelected(item,true);}
  }

  e->accept(tmp);
}

void FileTree::contentsDropEvent(QDropEvent *event) {
  if (QUriDrag::canDecode(event)) dropURL(event);
}

void FileTree::dropURL(QDropEvent *event) {
  KURL::List urls;
  if (KURLDrag::decode(event,urls) && urls.count() >0) {
    KURL::List::ConstIterator it(urls.begin());
    for(;it!=urls.end();  ++it) {
      if ((*it).isLocalFile())
	{ qDebug("is Local");
	qDebug((*it).path(0));
	addRealObject(new QString((*it).path(0)));
	emit (added());
	} else {
	  KMessageBox::sorry(this,"You can't add the non local object: <B>"+
			     (*it).prettyURL()+"</B>","Not supported yet");
	}
    }
  }
}
FileTreeItemDrag::FileTreeItemDrag(FileTreeItem * /*it*/, QWidget* dragsource)
  :QDragObject(dragsource) {
  //dragit=it;
}


void FileTree::reset()
{
  clear();
}
