/* -*-C-*-

$Id: ifd.c,v 1.19 2002/03/14 17:34:15 cph Exp $

Copyright (c) 2000-2002 Massachusetts Institute of Technology

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
02111-1307, USA.  */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <syslog.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <net/if.h>

#include "server.h"

/* Number of seconds between polls. */
#ifndef POLL_INTERVAL
#define POLL_INTERVAL 1
#endif

#ifndef PID_FILE
#define PID_FILE "/var/run/ifd.pid"
#endif

#ifndef SIOCGMIIPHY
#define SIOCGMIIPHY 0x8947	/* Get the PHY in use. */
#define SIOCGMIIREG 0x8948 	/* Read a PHY register. */
#endif

/* Pre-2.4 kernels used these numbers instead.  */
#define OLD_SIOCGMIIPHY SIOCDEVPRIVATE
#define OLD_SIOCGMIIREG (SIOCDEVPRIVATE + 1)

#define STATUS_UP		0x01
#define STATUS_RUNNING		0x02
#define STATUS_CONNECTED	0x04
#define STATUS_CONNVALID	0x08
#define STATUS_LIMIT		0x10

struct ifc_record
{
  const char * name;
  unsigned int status;
  unsigned int mark : 1;
};

static unsigned int records_length;
static unsigned int n_records;
static struct ifc_record * records;
static const char * program_to_run;
static unsigned int n_watched_ifcs;
static const char ** watched_ifcs;

static void process_record (struct ifc_record *, unsigned int);
static void print_status (char *, unsigned int);
static int watched_ifc_p (const char *);

static void preload_watched_ifcs (void);
static void clear_record_marks (void);
static struct ifc_record * intern_record (const char *);
static void process_unmarked_records (void);

static struct ifreq * get_interface_list (int, unsigned int *);
static struct ifreq * proc_interface_list (unsigned int *);
static void discard_line (FILE *);
static int proc_interface_name (FILE *, struct ifreq *);
static struct ifreq * conf_interface_list (int, unsigned int *);
static int get_link_status (int, struct ifreq *, int, unsigned int *);

static void * xmalloc (unsigned long);
static void * xrealloc (void *, unsigned long);
static char * copy_string (const char *);
static void log_perror (const char *);
static int execute (const char *, const char *);

int
main (int argc, const char ** argv)
{
  int fd;
  n_records = 0;
  records_length = 16;
  records = (xmalloc ((sizeof (struct ifc_record)) * records_length));

  /* Process command-line arguments. */
  if (argc < 2)
    {
      fprintf (stderr, "Usage: %s <program> <interface> ...\n", (argv[0]));
      exit (1);
    }
  program_to_run = (argv[1]);
  n_watched_ifcs = (argc - 2);
  watched_ifcs = (argv + 2);

  if ((strcmp (program_to_run, "--debug")) == 0)
    {
      program_to_run = 0;
      openlog ("ifd", LOG_PID, LOG_DAEMON);
    }
  else
    {
      /* Check that program can be executed. */
      if ((access (program_to_run, X_OK)) < 0)
	{
	  log_perror (program_to_run);
	  exit (1);
	}

      /* Run in background as a daemon. */
      run_in_background (0, PID_FILE); 
      openlog ("ifd", LOG_PID, LOG_DAEMON);
      syslog (LOG_INFO, "starting");
    }

  /* Open a basic socket. */
  fd = (socket (AF_INET, SOCK_DGRAM, 0));
  if (fd < 0)
    {
      log_perror ("socket");
      exit (1);
    }

  /* The main loop.  Scan the set of interfaces, calling
     process_record on each one (except loopbacks).  Then delay by the
     polling interval and repeat. */
  preload_watched_ifcs ();
  while (1)
    {
      unsigned int n_ifcs;
      struct ifreq * ifcs;
      struct ifreq * scan;
      struct ifreq * end;

      ifcs = (get_interface_list (fd, (&n_ifcs)));
      if (ifcs == 0)
	goto scan_done;
      end = (ifcs + n_ifcs);
      clear_record_marks ();
      for (scan = ifcs; (scan < end); scan += 1)
	{
	  if ((ioctl (fd, SIOCGIFFLAGS, scan)) < 0)
	    {
	      log_perror ("SIOCGIFFLAGS");
	      continue;
	    }
	  if (((scan -> ifr_flags) & IFF_LOOPBACK) == 0)
	    {
	      struct ifc_record * record = (intern_record (scan -> ifr_name));
	      unsigned int status;
	      if ((get_link_status (fd, scan, (watched_ifc_p (record -> name)),
				    (&status)))
		  < 0)
		continue;
	      process_record (record, status);
	    }
	}
      free (ifcs);
      process_unmarked_records ();
    scan_done:
      sleep (1);
    }
}

static void
process_record (struct ifc_record * record, unsigned int status)
{
  unsigned int old_status = (record -> status);
  (record -> status) = status;
  (record -> mark) = 1;
  if ((status != old_status) && (program_to_run != 0))
    {
      char cmd [10240];		/* 10240 is max command line length. */
      strcpy (cmd, program_to_run);
      strcat (cmd, " ");
      strcat (cmd, (record -> name));
      strcat (cmd, " ");
      strcat (cmd,
	      ((watched_ifc_p (record -> name)) ? "watched" : "unwatched"));
      strcat (cmd, " ");
      print_status (cmd, old_status);
      strcat (cmd, " ");
      print_status (cmd, status);
      execute ("network-change script", cmd);
    }
}

static void
print_status (char * cmd, unsigned int status)
{
  if (status == STATUS_LIMIT)
    strcat (cmd, "unknown");
  else
    {
      strcat (cmd, (((status & STATUS_UP) != 0) ? "up" : "down"));
      strcat (cmd, ",");
      strcat (cmd, (((status & STATUS_RUNNING) != 0) ? "running" : "stopped"));
      strcat (cmd, ",");
      strcat (cmd,
	      (((status & STATUS_CONNVALID) == 0)
	       ? "unknown"
	       : ((status & STATUS_CONNECTED) != 0)
	       ? "connected"
	       : "disconnected"));
    }
}

static int
watched_ifc_p (const char * name)
{
  const char ** scan = watched_ifcs;
  const char ** end = (scan + n_watched_ifcs);
  while (scan < end)
    if ((strcmp ((*scan++), name)) == 0)
      return (1);
  return (0);
}

static void
preload_watched_ifcs (void)
{
  const char ** scan = watched_ifcs;
  const char ** end = (scan + n_watched_ifcs);
  while (scan < end)
    intern_record (*scan++);
}

static void
clear_record_marks (void)
{
  struct ifc_record * scan = records;
  struct ifc_record * end = (scan + n_records);
  while (scan < end)
    ((scan++) -> mark) = 0;
}

static struct ifc_record *
intern_record (const char * name)
{
  struct ifc_record * scan = records;
  struct ifc_record * end = (scan + n_records);
  while (scan < end)
    {
      if ((strcmp ((scan -> name), name)) == 0)
	return (scan);
      scan += 1;
    }
  if (n_records == records_length)
    {
      records_length *= 2;
      records
	= (xrealloc (records,
		     ((sizeof (struct ifc_record)) * records_length)));
    }
  scan = (records + n_records);
  (scan -> name) = (copy_string (name));
  (scan -> status) = STATUS_LIMIT;
  (scan -> mark) = 1;
  n_records += 1;
  return (scan);
}

static void
process_unmarked_records (void)
{
  struct ifc_record * scan = records;
  struct ifc_record * end = (scan + n_records);
  while (scan < end)
    {
      if ((scan -> mark) == 0)
	process_record (scan, STATUS_LIMIT);
      scan += 1;
    }
}

/* Use /proc/net/dev if available, since that will give us a list of
   all of the interfaces.  Otherwise use SIOCGIFCONF, which will at
   least give us the interfaces that are up.  */

static struct ifreq *
get_interface_list (int fd, unsigned int * n_ifcs_return)
{
  struct ifreq * ifcs = (proc_interface_list (n_ifcs_return));
  return
    ((ifcs == 0)
     ? (conf_interface_list (fd, n_ifcs_return))
     : ifcs);
}

static struct ifreq *
proc_interface_list (unsigned int * n_ifcs_return)
{
  FILE * s = (fopen ("/proc/net/dev", "r"));
  if (s == 0)
    {
      log_perror ("Unable to open /proc/net/dev");
      return (0);
    }
  discard_line (s);
  discard_line (s);
  {
    unsigned int i = 0;
    unsigned int n = 4;
    struct ifreq * ifcs = (xmalloc ((sizeof (struct ifreq)) * n));
    struct ifreq ifc;
    while (proc_interface_name (s, (&ifc)))
      {
	if (i == n)
	  {
	    n *= 2;
	    ifcs = (xrealloc (ifcs, ((sizeof (struct ifreq)) * n)));
	  }
	(ifcs[i++]) = ifc;
      }
    fclose (s);
    (*n_ifcs_return) = i;
    return (ifcs);
  }
}

static void
discard_line (FILE * s)
{
  while (1)
    {
      int c = (getc (s));
      if ((c == EOF) || (c == '\n'))
	break;
    }
}

#define PIN_GETC(c, s)							\
{									\
  c = (getc (s));							\
  if (c == EOF)								\
    return (0);								\
  if (c == '\0')							\
    {									\
      discard_line (s);							\
      goto restart;							\
    }									\
}

#define PIN_ACCUM(c)							\
{									\
  if (i < n)								\
    (name[i++]) = (c);							\
  else									\
    {									\
      discard_line (s);							\
      goto restart;							\
    }									\
}

static int
proc_interface_name (FILE * s, struct ifreq * ifc)
{
  char * name = (ifc -> ifr_name);
  unsigned int n = (sizeof (ifc -> ifr_name));
  unsigned int i;
  int c;

 restart:
  do
    {
      PIN_GETC (c, s);
    }
  while (isspace (c));
  i = 0;
  while (1)
    {
      PIN_ACCUM (c);
      PIN_GETC (c, s);
      if (isspace (c))
	{
	  if (c != '\n')
	    discard_line (s);
	  break;
	}
      if (c == ':')
	{
	  if (i < n)
	    /* Might be part of an alias; check.  */
	    {
	      unsigned int saved_i = i;
	      while (1)
		{
		  PIN_ACCUM (c);
		  PIN_GETC (c, s);
		  if (!isdigit (c))
		    {
		      if (c != ':')
			/* Not an alias; first colon was name terminator.  */
			i = saved_i;
		      if (c != '\n')
			discard_line (s);
		      break;
		    }
		}
	    }
	  break;
	}
    }
  if (i < n)
    (name[i]) = '\0';
  return (1);
}

static struct ifreq *
conf_interface_list (int fd, unsigned int * n_ifcs_return)
{
  struct ifconf ifc;
  unsigned int n_ifcs = 4;
  unsigned int n_bytes = ((sizeof (struct ifreq)) * n_ifcs);

  (ifc . ifc_len) = n_bytes;
  (ifc . ifc_req) = (xmalloc (n_bytes));
  while (1)
    {
      if ((ioctl (fd, SIOCGIFCONF, (&ifc))) < 0)
	{
	  log_perror ("SIOCGIFCONF");
	  free (ifc . ifc_req);
	  return (0);
	}
      if ((ifc . ifc_len) < n_bytes)
	break;
      n_ifcs *= 2;
      n_bytes = ((sizeof (struct ifreq)) * n_ifcs);
      (ifc . ifc_req) = (xrealloc ((ifc . ifc_req), n_bytes));
    }

  (*n_ifcs_return) = ((ifc . ifc_len) / (sizeof (struct ifreq)));
  return (ifc . ifc_req);
}

static int
get_link_status (int fd, struct ifreq * ifr, int watched_p,
		 unsigned int * rresult)
{
  unsigned int result = 0;
  int siocgmiireg = SIOCGMIIREG;

  if (((ifr -> ifr_flags) & IFF_UP) != 0)
    result |= STATUS_UP;
  if (((ifr -> ifr_flags) & IFF_RUNNING) != 0)
    result |= STATUS_RUNNING;

  /* Get the vitals from the interface. */
  if ((ioctl (fd, SIOCGMIIPHY, ifr)) < 0)
    {
      if (errno == ENODEV)
	{
	  syslog (LOG_DEBUG, "%s is unavailable\n", (ifr -> ifr_name));
	  return (-1);
	}
      /* Try old ioctl too, but only for watched interfaces, because
	 it isn't always used for this purpose on all interfaces.  */
      if (!watched_p)
	goto done;
      if ((ioctl (fd, OLD_SIOCGMIIPHY, ifr)) < 0)
	{
	  if (errno == ENODEV)
	    {
	      syslog (LOG_DEBUG, "%s is unavailable\n", (ifr -> ifr_name));
	      return (-1);
	    }
	  syslog (LOG_DEBUG, "SIOCGMIIPHY on %s failed: %s\n",
		  (ifr -> ifr_name), (strerror (errno)));
	  goto done;
	}
      siocgmiireg = OLD_SIOCGMIIREG;
    }

  /* Check the link status. */
  {
    unsigned short * data = ((unsigned short *) (& (ifr -> ifr_data)));
    (data[1]) = 1;
    if ((ioctl (fd, siocgmiireg, ifr)) < 0)
      {
	if (errno == ENODEV)
	  {
	    syslog (LOG_DEBUG, "%s is unavailable\n", (ifr -> ifr_name));
	    return (-1);
	  }
	syslog (LOG_DEBUG, "SIOCGMIIREG on %s failed: %s\n",
		(ifr -> ifr_name), (strerror (errno)));
	goto done;
      }
    result |= STATUS_CONNVALID;
    if (((data[3]) & 0x0016) == 0x0004)
      result |= STATUS_CONNECTED;
  }

 done:
  (*rresult) = result;
  return (0);
}

static void *
xmalloc (unsigned long n_bytes)
{
  void * p = (malloc (n_bytes));
  if (p == 0)
    {
      syslog (LOG_DEBUG, "Unable to allocate %lu bytes.\n", n_bytes);
      exit (1);
    }
  return (p);
}

static void *
xrealloc (void * p1, unsigned long n_bytes)
{
  void * p2 = (realloc (p1, n_bytes));
  if (p2 == 0)
    {
      syslog (LOG_DEBUG, "Unable to reallocate %lu bytes.\n", n_bytes);
      exit (1);
    }
  return (p2);
}

static char *
copy_string (const char * string)
{
  char * result = (xmalloc ((strlen (string)) + 1));
  strcpy (result, string);
  return (result);
}

static void
log_perror (const char * string)
{
  syslog (LOG_DEBUG, "%s: %s", string, (strerror (errno)));
}

static int
execute (const char * msg, const char * cmd)
{
    int ret;
    FILE * f;
    char line [256];
    const char * suffix = " 2>&1";
    unsigned int limit = ((sizeof (line)) - ((strlen (suffix)) + 1));
    
    syslog (LOG_INFO, "executing: '%s'", cmd);
    strncpy (line, cmd, limit);
    (line[limit]) = '\0';
    strcat (line, suffix);
    f = (popen (line, "r"));
    while (fgets (line, 255, f))
      {
	(line [(strlen (line)) - 1]) = '\0';
	syslog (LOG_INFO, "+ %s", line);
      }
    ret = (pclose (f));
    if (WIFEXITED (ret))
      {
	if (WEXITSTATUS (ret))
	  syslog (LOG_INFO, "%s exited with status %d",
		  msg, (WEXITSTATUS (ret)));
	return (WEXITSTATUS (ret));
      }
    else
      {
	syslog (LOG_INFO, "%s exited on signal %d",
		msg, (WTERMSIG (ret)));
	return (-1);
      }
}
