/*
 * $Id: cmd_config.c,v 1.1 2004/12/21 23:26:18 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 *
 * Copyright (C) 2003 - 2004 Silicon Graphics, Inc. All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */
#include <lcrash.h>
#include <kl_config.h>
#include <kl_libhwconfig.h>

/*
 * print_hwcmp_info()
 */
void 
print_hwcmp_info(hwcomponent_t *hwcp, FILE *ofp)
{
	fprintf(ofp, "%5d  ", hwcp->seq);

	if (hwcp->name) {
		fprintf(ofp, "%14s  ", hwcp->name);
	} else {
		fprintf(ofp, "            NA  ");
	}

	if (hwcp->location) {
		fprintf(ofp, "%12s  ", hwcp->location);
	} else {
		fprintf(ofp, "          NA  ");
	}

	if (hwcp->serial_number) {
		fprintf(ofp, "%18s  ", hwcp->serial_number);
	} else {
		fprintf(ofp, "                NA  ");
	}

	if (hwcp->part_number) {
		fprintf(ofp, "%12s  ", hwcp->part_number);
	} else {
		fprintf(ofp, "          NA  ");
	}

	if (hwcp->revision) {
		fprintf(ofp, "%8s\n", hwcp->revision);
	} else {
		fprintf(ofp, "      NA\n");
	}
}

/*
 * print_hwconfig()
 */
void
print_hwconfig(hwconfig_t *hcp, FILE *ofp)
{
	hwcomponent_t *hwcp = hcp->hwcp_root;

	if (hwcp) {
		fprintf(ofp, "  SEQ            NAME      LOCATION          "
			"SERIAL_NUM   "
			"PART_NUMBER  REVISION\n");
		fprintf(ofp, "=============================================="
			"=================================\n");
		do {
			print_hwcmp_info(hwcp, ofp);
		} while ((hwcp = next_hwcmp(hwcp)));
		fprintf(ofp, "=============================================="
			"=================================\n");
	} else {
		fprintf(ofp, "Hardware config data not available!\n");
	}
	return;
}

/*
 * config_cmd() -- Run the 'config' command.
 */
int
config_cmd(command_t *cmd)
{
	hwconfig_t *hcp;
	
	if (!(hcp = kl_get_hwconfig(0))) {
		return(1);
	}
	print_hwconfig(hcp, cmd->ofp);
	kl_free_hwconfig(hcp);
	return(0);
}

#define _CONFIG_USAGE "[-f] [-w outfile] [element_list]"

/*
 * config_usage() -- Print the usage string for the 'config' command.
 */
void
config_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _CONFIG_USAGE);
}

/*
 * config_help() -- Print the help information for the 'config' command.
 */
void
config_help(command_t *cmd)
{
	CMD_HELP(cmd, _CONFIG_USAGE,
		"Display hardware configuration information.");
}

/*
 * config_parse() -- Parse the command line arguments for 'config'.
 */
int
config_parse(command_t *cmd)
{
	if (set_cmd_flags(cmd, (C_FULL|C_WRITE|C_NEXT|C_ALL), 0)) {
		return(1);
	}
	return(0);
}

/*
 * config_complete() -- Complete arguments of 'config' command.
 */
char *
config_complete(command_t *cmd)
{
	char *ret;

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	fprintf(cmd->ofp, "\n");
	config_usage(cmd);
	return(DRAW_NEW_ENTIRE_LINE);
}
