/*
 * $Id: cmd_help.c,v 1.1 2004/12/21 23:26:18 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>
#include <lc_license.h>

#define C_CFLAG (1 << C_LFLG_SHFT)
#define C_WFLAG (2 << C_LFLG_SHFT)

/*
 * help_list() -- Print out the list of possible commands that have 'help'.
 *                This prints out duplicates as well (no little alias trick).
 */
int
help_list(command_t *cmd)
{
	int i, j, index, cur_count = 0;
	static int count = 0;
	static char **cmd_list = (char **)0;
	cmd_rec_t *cmd_rec;

	if ((cmd_rec = first_cmd_rec())) {
		do {
			cur_count++;
		} while ((cmd_rec = next_cmd_rec(cmd_rec)));
	}

	if (!count || (count < cur_count)) {
		if (cmd_list) {
			kl_free_block(cmd_list);
		}
		count = cur_count;
		cmd_list = kl_alloc_block(count * sizeof(char *), K_PERM); 
		cmd_rec = first_cmd_rec();
		for (i = 0; i < count; i++)  {
			cmd_list[i] = cmd_rec->CMD_NAME;
			cmd_rec = next_cmd_rec(cmd_rec);
		}
	}

	index = count / 4 + ( count % 4 ? 1 : 0);
	for (i = 0; i < index; i++) {
		fprintf(cmd->ofp, "%-17s", cmd_list[i]);
		if ((j = index + i) < count) {
			fprintf(cmd->ofp, "%-17s", cmd_list[j]);
		}
		if ((j = index * 2 + i) < count) {
			fprintf(cmd->ofp, "%-17s", cmd_list[j]);
		}
		if ((j = index * 3 + i) < count) {
			fprintf(cmd->ofp, "%-17s", cmd_list[j]);
		}
		fprintf(cmd->ofp, "\n");
	}
	return(0);
}

/*
 * help_all_list() -- Print out help on every command for 'help all'.
 */
void
help_all_list(command_t *cmd)
{
	cmd_rec_t *cmd_rec;

	cmd_rec = first_cmd_rec();
	while (cmd_rec) {
		if (!cmd_rec->real_cmd) {
			sprintf(cmd->command, "%s", cmd_rec->CMD_NAME);
			(*cmd_rec->cmdhelp)(cmd);
		}
		cmd_rec = next_cmd_rec(cmd_rec);
	}
}

/*
 * help_found() -- Find the index of the help item we are looking for.
 */
cmd_rec_t *
help_found(char *helpcmd)
{
	cmd_rec_t *cmd_rec;

	if ((cmd_rec = find_cmd_rec(helpcmd))) {
		if (cmd_rec->real_cmd) {
			return(cmd_rec->real_cmd);
		}
		else {
			return(cmd_rec);
		}
	}
	return((cmd_rec_t *)NULL);
}

/*
 * help_cmd() -- Run the 'help' command.
 */
int
help_cmd(command_t *cmd)
{
	int j;
	cmd_rec_t *cmd_rec; 

	/* As GPL requires, we print out copying and warranty info here.
	 */
	if (cmd->flags & (C_CFLAG | C_WFLAG)){
		if (cmd->flags & C_CFLAG){
			fprintf(cmd->ofp, LCRASH_COPYING);
		}
		if (cmd->flags & C_WFLAG){
			fprintf(cmd->ofp, LCRASH_WARRANTY);
		}
		return(0);
	}

	/* First off, grab the command passed in.
	 */
	if (cmd->nargs < 1) {
		/* See if we didn't specify anything to get help on.
		 * If we didn't, just dump out all commands.
		 */
		help_list(cmd);
	} else {
		/* Print all help information out if 'all' is the first 
		 * argument.
		 */
		if (!strcmp(cmd->args[0], "all")) {
			help_all_list(cmd);
		}
		else {
			for (j = 0; j < cmd->nargs; j++) {
				if (!(cmd_rec = help_found(cmd->args[j]))) {
					fprintf(cmd->efp, "\nNo help exists on "
						"\"%s\".\n", cmd->args[j]);
				}
				else {
					sprintf(cmd->command, "%s", 
						cmd_rec->CMD_NAME);
					(*cmd_rec->cmdhelp)(cmd);
				}
			}
		}
	}
	return(0);
}

#define _HELP_USAGE \
"\n        [-C]" \
"\n        [-W]" \
"\n        [all | <command-list>]"\
"\n        [-w outfile]"

#define _HELP_HELP \
"Display help for lcrash and lcrash commands.\n"\
"\nOPTIONS:"\
"\n -C    Show copying conditions."\
"\n -W    Show warranty details."\
"\n all   Show description and syntax for all lcrash commands."\
"\n <command-list>"\
"\n       Show description and syntax for specified lcrash commands."

/*
 * help_usage() -- Print the usage string for the 'help' command.
 */
void
help_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _HELP_USAGE);
}

/*
 * help_help() -- Print the help information for the 'help' command.
 */
void
help_help(command_t *cmd)
{
	CMD_HELP(cmd, _HELP_USAGE, _HELP_HELP);
}

/*
 * help_parse() -- Parse the command line arguments for 'help'.
 */
int
help_parse(command_t *cmd)
{
	option_t *op;

	if (set_cmd_flags(cmd, (C_CFLAG|C_WFLAG|C_WRITE), "CW")) {
		return(1);
	}
	op = cmd->options;

	while (op) {
		switch(op->op_char) {
			case 'C':
				cmd->flags |= C_CFLAG;
				break;
			case 'W':
				cmd->flags |= C_WFLAG;
				break;
		}
		op = op->op_next;
	}
	return(0);
}

/*
 * help_complete() -- Complete arguments of 'help' command.
 */
char *
help_complete(command_t *cmd)
{
	char *ret;

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	fprintf(cmd->ofp, "\n");
	help_usage(cmd);
	return(DRAW_NEW_ENTIRE_LINE);
}
