/*
 * $Id: cmd_module.c,v 1.1 2004/12/21 23:26:18 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002, 2004 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>
#include <string.h>

/* Forward declaration
 */
void module_usage(command_t *);

/* 
 * module_cmd() -- Command for ...
 */
int
module_cmd(command_t *cmd)
{
	void * modptr = NULL;
	kaddr_t modaddr = 0;
	char * modname = NULL;

	if(cmd->nargs==0){
		if(KL_LINUX_RELEASE >= LINUX_2_6_0){
			syment_t *sym_module_list = NULL;
			if(!(sym_module_list = kl_lkup_symname("modules"))){
				KL_ERROR = KLE_NO_MODULE_LIST;
				return(1);
			}
			modaddr = sym_module_list->s_addr;
			if(walk_structs("module", "list", -1, modaddr, 
					cmd->flags | C_LISTHEAD_N, cmd->ofp)){
                                fprintf(KL_ERRORFP, "Error while walking through "
                                        "module structures of dump.\n");
                                return(1);
                        }	
                } else {
			if(kl_get_module(modname,&modaddr, &modptr)){
				if(KL_ERROR == KLE_NO_MODULE_LIST) {
					fprintf(KL_ERRORFP, "ERROR: ");
					kl_print_error();
					kl_reset_error();
				} else {
					fprintf(KL_ERRORFP, "Error while reading module"
						" structure from dump.\n");
				}
				return(1);
			}
			kl_free_block(modptr);
			if(walk_structs("module", "next", -1, modaddr, cmd->flags, cmd->ofp)){
				fprintf(KL_ERRORFP, "Error while walking through "
					"module structures of dump.\n");
				return(1);
			}
		}
	} else if (cmd->nargs ==1) {
		int rc;
		
		if(KL_LINUX_RELEASE >= LINUX_2_6_0)
			rc = kl_get_module_2_6(cmd->args[0], &modaddr, &modptr);
		else
			rc = kl_get_module(cmd->args[0], &modaddr, &modptr);
		if(rc) {
			if(KL_ERROR == KLE_NO_MODULE_LIST) {
				fprintf(KL_ERRORFP, "ERROR: ");
				kl_print_error();
				kl_reset_error();
			} else {
				fprintf(KL_ERRORFP, "Could not load module "
					"structure from dump for module: %s\n", 
					cmd->args[0]);
			}
			return(1);
		}
		module_banner(cmd->ofp, cmd->flags | BANNER | SMAJOR);
		print_module(modaddr, modptr, cmd->flags, cmd->ofp);
		module_banner(cmd->ofp, cmd->flags |  SMAJOR);
		kl_free_block(modptr);
	} else {
		fprintf(KL_ERRORFP, "Invalid commandline.\n");
		module_usage(cmd);
		return(1);
	}
	return(0);
}

#define _MODULE_USAGE \
"\n        [modulename]"\
"\n        [-f]"\
"\n        [-k]"\
"\n        [-p]"\
"\n        [-i iteration_threshold]"\
"\n        [-w outfile]"

#define _MODULE_HELP \
"Display list of loaded modules or exported module symbols.\n"\
"\nOPTIONS:"\
"\n modulename"\
"\n       Select a module structure of linked list \"module_list\" of the"\
"\n       kernel for which information should be displayed."\
"\n       Without a 'modulename' information for all module structures"\
"\n       of the linked list \"module_list\" of the kernel is given."\
"\n -f    Instead of the default output display a list of exported module"\
"\n       symbols for all or the selected module structure(s)."\
"\n       Equals \"cat /proc/ksyms\" in a running Linux system."\
"\n -k    List all kallsyms symbol information."\
"\n -p    Same as '-f' option except that plain output is generated,"\
"\n       e.g. no headers are printed."\
"\n -i iteration_threshold"\
"\n       By default certain loops are interrupted after 10'000 iterations"\
"\n       to avoid endless loops while following invalid pointers. Using"\
"\n       this option you can change the threshold for the current command."\
"\n       A value '0' means infinite iteration threshold, i.e. no"\
"\n       interruption of the loop is caused by reaching any threshold.\n"\
"\nThe default output shows: address of module structure, its size,"\
"\nusecount, name, and names of modules that depend on the module."\
"\nThis is similar to output of \"cat /proc/modules\" in a running Linux"\
"\nsystem.\n"\
"\nThe \"struct module kernel_module\" can be selected by using "\
"\n\"kernel_module\" as 'modulename'."

/*
 * module_usage() -- Print the usage string for the 'module' command.
 */
void
module_usage(command_t *cmd)
{
	CMD_USAGE(cmd, _MODULE_USAGE);
}


/*
 * module_help() -- Print the help information for the 'module' command.
 */
void
module_help(command_t *cmd)
{
	CMD_HELP(cmd, _MODULE_USAGE, _MODULE_HELP);
}

/*
 * module_parse() -- Parse the command line arguments for 'module'.
 */
int
module_parse(command_t *cmd)
{
	option_t *op;

	if (set_cmd_flags(cmd, (C_FULL|C_WRITE), "i:kp")) {
		return(1);
	}
	op = cmd->options;

	while (op) {
		switch(op->op_char) {
			case 'i':
				cmd->flags |= C_ITER;
				kl_get_value(op->op_arg, 0, 0, &iter_threshold);
				if(KL_ERROR){
					fprintf(KL_ERRORFP, "Invalid value for "
						"iter_threshold.\n");
					return(1);
				}
				break;
			case 'k':
				cmd->flags |= C_KALLSYMS;
				break;
			case 'p':
				cmd->flags |= C_PLAIN | C_FULL;
				break;
		}
		op = op->op_next;
	}
	return(0);
}

/*
 * module_complete() -- Complete arguments of 'module' command.
 */
char *
module_complete(command_t *cmd)
{
	char *ret;

	/* complete standard options (for example, -w option) arguments
	 */
	if ((ret = complete_standard_options(cmd)) != NOT_COMPLETED) {
		return(ret);
	}
	fprintf(cmd->ofp, "\n");
	module_usage(cmd);
	return(DRAW_NEW_ENTIRE_LINE);
}
