/*
 * $Id: lc_command.h,v 1.2 2005/02/25 22:04:04 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#ifndef __LC_COMMAND_H
#define __LC_COMMAND_H

#include <rl.h>

#define MAX_ARGS	128
#define MAX_CMDLINE 	256

/*
 * Internal structure to store options and arguments of current
 * executed command. 
 */
typedef struct option_s {
	struct option_s	       *op_next;
	char			op_char;
	char		       *op_arg;
} option_t;

/*
 * Internal structure to keep information about current executed command.
 */
typedef struct command_s {
	int			flags;
	char 			cmdstr[MAX_CMDLINE];
	char 		       *command;
	char		       *cmdline;
	option_t	       *options; 
	int 			nargs;
	char   	       	       *args[MAX_ARGS];
	char 		       *pipe_cmd;
	FILE   	       	       *ofp;
	FILE		       *efp; /* should be removed, only one error stream for lcrash */
} command_t;

/*
 * Function pointers used to manage commands.
 */
typedef int(*cmdfunc_t) (command_t *);
typedef int(*cmdparse_t) (command_t *);
typedef void(*cmdhelp_t) (command_t *);
typedef void(*cmdusage_t) (command_t *);
typedef char *(*cmdcomplete_t) (command_t *);

/*
 * Store function set belonging to one command.
 */
typedef struct _command {
	char           *cmd;  	  /* command name */
	char   	       *real_cmd; /* real command name if cmd is alias */
	cmdfunc_t	cmdfunc;  /* base function */
	cmdparse_t 	cmdparse; /* argument parsing function  */
	cmdhelp_t 	cmdhelp;  /* help function */
	cmdusage_t	cmdusage; /* usage string function */
	cmdcomplete_t	cmdcomplete; /* completion function */
} _command_t;

/* set of commands common to all dump architectures */
extern _command_t cmdset[];

/* 
 * Internal data structure that contains vital information about a 
 * command. Unlike with _command_t, cmd_rec_s structs are inserted
 * into a binary search tree, using command name as the search key.
 * The command tree contains all commands, including those that are
 * part of lcrash by default and those that are registered at runtime.
 */
typedef struct cmd_rec_s {
        btnode_t                bt;        /* Must be first */
        struct cmd_rec_s       *real_cmd;
        cmdfunc_t               cmdfunc;   /* base function */
        cmdparse_t              cmdparse;  /* argument parsing function  */
        cmdhelp_t               cmdhelp;   /* help function */
        cmdusage_t              cmdusage;  /* usage string function */
        cmdcomplete_t           cmdcomplete; /* completion function */
} cmd_rec_t;

/* macro for easier access to command name */
#define CMD_NAME bt.bt_key

/* pointer to tree where commands are kept */
extern cmd_rec_t *cmd_tree;

/* some commonly used flags for display purposes */
#define BANNER		   1
#define SMAJOR		   2
#define SMINOR		   4

/* 
 * Standard LCRASH command line flags. There is no requirement that
 * they be used, but it is possible to use them in conjunction with
 * command specific flags (as long as the flag bits do not conflict.
 */
#define C_FALSE         0x00000001   /* Command takes no arguments */
#define C_TRUE          0x00000002   /* Command requires arguments */
#define C_ALL           0x00000004   /* All elements */
#define C_PERM		0x00000008   /* Allocate perminant blocks */
#define C_TEMP			 0   /* For completeness */
#define C_FULL          0x00000010   /* Full output */
#define C_LIST          0x00000020   /* List items */
#define C_NEXT          0x00000040   /* Follow links */
#define C_WRITE         0x00000080   /* Write output to file */
#define C_NO_OPCHECK    0x00000100   /* Don't reject bad cmd line options */
#define C_ITER          0x00000200   /* set iteration threshold */

/* 
 * Shift value so that commands can define their own local command line 
 * option flags that don't conflict with the standard ones defined above.
 */
#define C_LFLG_SHFT	  12

/* 
 * Common "Local" flags for use by dump_memory(), eval.c, module.c, etc.
 */
#define C_DECIMAL  	(0x0001 << C_LFLG_SHFT) /* dump memory in decimal */
#define C_OCTAL    	(0x0002 << C_LFLG_SHFT) /* dump memory in octal */ 
#define C_HEX      	(0x0004 << C_LFLG_SHFT) /* dump memory in hex */
#define C_BINARY      	(0x0008 << C_LFLG_SHFT) /* dump memory in binary */
#define C_INSTR      	(0x0010 << C_LFLG_SHFT) /* dump instruction */
#define C_DWORD    	(0x0020 << C_LFLG_SHFT) /* Eight byte values */
#define C_WORD     	(0x0040 << C_LFLG_SHFT) /* Four byte values */
#define C_HWORD    	(0x0080 << C_LFLG_SHFT) /* two byte values */
#define C_BYTE     	(0x0100 << C_LFLG_SHFT) /* byte values */ 

/**
 ** Command specific "Local" flags used in conjunction with common falgs.
 **/

/* 
 * "Local" flags for use by eval module
 */
#define C_SIZEOF        (0x1000 << C_LFLG_SHFT) /* eval flag */
#define C_WHATIS        (0x2000 << C_LFLG_SHFT) /* eval flag */
#define C_NOVARS        (0x4000 << C_LFLG_SHFT) /* eval flag */
#define C_WOFFSET       (0x8000 << C_LFLG_SHFT)  /* with offset field */

/* 
 * "Local" flags for use by task command
 */
#define C_RUNABLE       (0x1000 << C_LFLG_SHFT) /* runable task flag */
#define C_UNRUNABLE     (0x2000 << C_LFLG_SHFT) /* unrunable task flag */
#define C_STOPPED       (0x4000 << C_LFLG_SHFT) /* stopped task flag */

/* 
 * "Local" flags for use by module command
 */
#define C_KALLSYMS      (0x1000 << C_LFLG_SHFT) /* print kallsysms info */
#define C_PLAIN         (0x2000 << C_LFLG_SHFT) /* plain output */

/* 
 * "Local" flags for use by dump and/or search commands
 */
#define C_PADDR         (0x1000 << C_LFLG_SHFT) /* address is physical */
#define C_STRING        (0x2000 << C_LFLG_SHFT) /* string value */
#define C_CONCAT        (0x4000 << C_LFLG_SHFT) /* no space between values */
#define C_MASK          (0x8000 << C_LFLG_SHFT) /* mask for search (w/args)   */

/* 
 * "Local" flags for use by walk command
 */
#define C_STRUCT	(0x01 << C_LFLG_SHFT)  /* dump C-like struct */
#define C_LISTHEAD_N    (0x04 << C_LFLG_SHFT)  /* walk using list_head.next */
#define C_LISTHEAD_P    (0x08 << C_LFLG_SHFT)  /* walk using list_head. prev */
#define C_LISTHEAD      (C_LISTHEAD_N | C_LISTHEAD_P) /* use struct list_head */
#define C_PDSTRUCT      (0x10 << C_LFLG_SHFT)  /* handle "predefined" struct */
#define C_TABLE         (0x20 << C_LFLG_SHFT)  /* output of "list_head-walk" */
                                               /*    in short tabular format */
/* "Local" flags for use by savedump()
 */
 
#define C_LDUMP_COMPR_NONE     (0x01 << C_LFLG_SHFT)  /* no compression */
#define C_LDUMP_COMPR_RLE      (0x02 << C_LFLG_SHFT)  /* Runlength encoding */
#define C_LDUMP_COMPR_GZIP     (0x04 << C_LFLG_SHFT)  /* Gzip compression  */
#define C_LDUMP_X_BCACHE       (0x10 << C_LFLG_SHFT)  /* exclude buffer cache*/
#define C_LDUMP_X_USER_PAGES   (0x20 << C_LFLG_SHFT)  /* exclude user pages */
#define C_LDUMP_X_FREE_PAGES   (0x40 << C_LFLG_SHFT)  /* exclude free pages */
#define C_LDUMP_X_PCACHE       (0x80 << C_LFLG_SHFT)  /* exclude page cache */
#define C_LDUMP_VERBOSE        (0x100<< C_LFLG_SHFT)  /* verbose output */
#define C_LDUMP_INFO_FILE      (0x200<< C_LFLG_SHFT)  /* info file */

/* 
 * Plural flag definitions.
 */
#define PLURAL(str, cnt, ofp) \
	fprintf(ofp, "%d %s%s found\n", cnt, str, (cnt != 1) ? "s" : "");

/* 
 * macros used in help ans usage functions of commands
 */
 
#define CMD_USAGE(cmd, s) \
    fprintf(cmd->ofp, "USAGE: %s %s\n", cmd->command, s);

#define CMD_HELP(cmd, u, s) \
do{ \
    char *t = helpformat(s); \
    fprintf(cmd->ofp, "COMMAND: %s %s\n\n%s\n", cmd->command, u, t); \
    kl_free_block(t); \
} while(0)


/*
 * function prototypes
 */

int register_cmd(
	char *			/* command name */,
	char *			/* real command name if this is an alias */,
	int			/* command argument flag */,
	cmdfunc_t 		/* base function */,
	cmdparse_t		/* argument parsing function  */,
	cmdhelp_t		/* help function */,
	cmdusage_t 		/* usage string function */);

int process_commands(void);

/* register a command set to lcrash */
int register_cmds(_command_t *);

/* unregister a command from lcrash */
void unregister_cmd(char *);

/* process commands */
int process_cmds(void);

/* find a command record */
cmd_rec_t *find_cmd_rec(char *);

/* return first command record */
cmd_rec_t *first_cmd_rec(void);

/* return last command record */
cmd_rec_t *last_cmd_rec(void);

/* return next command record */
cmd_rec_t *next_cmd_rec(cmd_rec_t *);

/* return previous command record */
cmd_rec_t *prev_cmd_rec(cmd_rec_t *);

/* set flags for a command */
int set_cmd_flags(command_t *, int, char *);

/* specify help format */
char *helpformat(char *);

char *complete_cmds(char *, int);
char *complete_subcmd_name(char *);
char *complete_standard_options(command_t *);
char *complete_symbol_name(char *, int);
char *complete_file_name(char *, int);

#define NOT_COMPLETED	(char *)-2

#endif /* __LC_COMMAND_H */
