/*
 * $Id: report.c,v 1.2 2005/02/23 19:31:16 tjm Exp $
 *
 * This file is part of lcrash, an analysis tool for Linux memory dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <lcrash.h>

/*
 * do_report() -- generate an lcrash report.
 */
int
do_report(int flags, FILE *ofp)
{
	int is_live_dump = 0;
	time_t curtime;
	kl_dump_header_t dh;
	time_t t;

#if defined(DUMP_ARCH_S390) || defined(DUMP_ARCH_S390X)
	/* dumps generated by s390/x stand alone dump tools, those just write
	 * kl_dump_header_s390_t and no kl_dump_header_t, handle it separately
	 */
	if(MIP->core_type == s390_core) {
		return do_report_s390(flags, ofp);
	}
#endif

	/* first, make sure this isn't a live system 
	 */
	if (!CORE_IS_DUMP) {
		fprintf(ofp, "Error: a report can only be generated for "
			"dump files!\n");
		return(1);
	}

	/* get the dump header 
	 */
	if(kl_get_dump_header(&dh)){
		fprintf(ofp, "Error: Cannot read() dump header "
			"from dump file!\n");
		return(1);
	}

	if (dh.magic_number == KL_DUMP_MAGIC_LIVE) {
		is_live_dump = 1; 
	} else if (dh.magic_number != KL_DUMP_MAGIC_NUMBER) {
		fprintf(ofp, "Error: DUMP_MAGIC_NUMBER in dump header "
			"is invalid!\n");
			return(1);
	}

	fprintf(ofp,
		"=======================\n"
		"LCRASH CORE FILE REPORT\n"
		"=======================\n\n");

	curtime = time((time_t *)0);
	fprintf(ofp, "GENERATED ON:\n    %s\n\n", ctime(&curtime));

	/* dh.time is of type struct timeval -> arch dependent */
	t = (time_t)dh.time.tv_sec;
	if (is_live_dump) {
		fprintf(ofp, "LIVE DUMP CREATED:\n    %s\n\n", ctime(&t));
	} else {
		fprintf(ofp, "TIME OF CRASH:\n    %s\n\n", ctime(&t));
		if (dh.panic_string[0] != 0) {
			fprintf(ofp, "PANIC STRING:\n    %s\n\n",
				dh.panic_string);
		} else {
			fprintf(ofp, "PANIC STRING:\n    (none)\n\n");
		}
	}

	fprintf(ofp, "MAP:\n    %s\n\n", KL_MAP_FILE);
	fprintf(ofp, "DUMP:\n    %s\n\n", KL_DUMP_FILE);
	fprintf(ofp, "KERNTYPES:\n    %s\n\n", KL_KERNTYPES_FILE);

	fprintf(ofp,
		"================\n"
		"COREFILE SUMMARY\n"
		"================\n\n");

	/*
	 * Indicate the general nature of the event leading up to the
	 * dump. For dumps initiated by the kernel, there is currently
	 * only one type of crash dump -- the one that occurs due to a 
	 * software kernel problem.  Later, we need to add code to deal 
	 * with NMIs, hardware failures, etc. It's possible that the
	 * dump was generated by an application from live system memory. 
	 */
	if (is_live_dump) {
		fprintf(ofp, 
			"    This is a dump from a live, running system.\n\n");
	} else {
		fprintf(ofp, 
			"    The system died due to a software failure.\n\n");
	}

	fprintf(ofp,
		"===================\n"
		"UTSNAME INFORMATION\n"
		"===================\n\n");

	fprintf(ofp, "   sysname : %s\n", dh.utsname_sysname);
	fprintf(ofp, "  nodename : %s\n", dh.utsname_nodename);
	fprintf(ofp, "   release : %s\n", dh.utsname_release);
	fprintf(ofp, "   version : %s\n", dh.utsname_version);
	fprintf(ofp, "   machine : %s\n", dh.utsname_machine);
	fprintf(ofp, "domainname : %s\n", dh.utsname_domainname);
	fprintf(ofp, "\n");

	fprintf(ofp,
		"===============\n"
		"LOG BUFFER DUMP\n"
		"===============\n\n");

	/* print out the system log
	 */
	print_log_buf(ofp);
	fprintf(ofp, "\n\n");

	fprintf(ofp,
		"====================\n"
		"CURRENT SYSTEM TASKS\n"
		"====================\n\n");

	print_active_tasks(flags, ofp);
	fprintf(ofp, "\n");

	fprintf(ofp,
		"===========================\n"
		"STACK TRACE OF FAILING TASK\n"
		"===========================\n\n");

 	if(DUMPTASK_TRACE((kaddr_t)dh.current_task, flags, ofp)){
 		return(1);
 	}

	return(0);
}
