/*
 * $Id: kl_kern.c,v 1.3 2005/03/02 21:38:01 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <klib.h>

/* These are global variables, so declare them here. They may not get
 * set for all architectures, however...
 */
kaddr_t VMALLOC_START = 0;
kaddr_t VMALLOC_END = 0;

/*
 * Name: kl_valid_physmem()
 * Func: Returns 1 if a physical address represents valid physical
 *       memory, otherwise 0 is returned. In the event that the addr
 *       passed in is invalid, a 0 is returned and KL_ERROR is set
 *       to indicate which error occurred.
 */
int
kl_valid_physmem(kaddr_t addr, int size)
{
	/* For now, always return TRUE. If there is an architectures
	 * that require a valid test of a physical memory address,
	 * then implement an architecture specific version of this
	 * function in the kl_kern_<arch>.c file.
	 *
	 * TODO: Check to see if the end physical address is greater
	 * than the value of num_physpages.
	 */
	return(1);
}

/*
 * Name: kl_valid_physaddr()
 * Func: Returns 1 if a physical address represents valid physical
 *       memory, otherwise 0 is returned. In the event that the addr
 *       passed in is invalid, a 0 is returned and KL_ERROR is set
 *       to indicate which error occurred.
 */
int
kl_valid_physaddr(kaddr_t addr)
{
	return(KL_VALID_PHYSMEM(addr, 1));
}

/*
 * Name: kl_next_valid_physaddr()
 * Func: Returns the next valid physical address located beyond addr.
 *       Note that the physical address returned will be word aligned.
 *       If the address passed in is invalid (kl_virtop() fails), or
 *       there is no valid next physical address, then high_mem is
 *       returned and KL_ERROR is set to indicate the cause of the
 *       error.
 */
kaddr_t
kl_next_valid_physaddr(kaddr_t addr)
{
	kaddr_t paddr;

	if (addr == KL_HIGH_MEMORY) {
		KL_ERROR = KLE_INVALID_PADDR;
		return(KL_HIGH_MEMORY);
	} else if (KL_VIRTOP(addr, NULL, &paddr)) {
		return(KL_HIGH_MEMORY);
	}

	/* We know that paddr is valid. Now compute the next valid
	 * (word aligned) physical address and return it.
	 */
	paddr = ((paddr + KL_NBPW) & ~(KL_NBPW));
	return(paddr);
}

/* 
 * While dumping memory the task structures might change. If implemented,
 * this function redirects requests to read memory belonging to a task
 * structure and its kernel stack.
 * Snapshots of this memory area might be saved in the dump asm header.
 */
kaddr_t
kl_fix_vaddr(kaddr_t vaddr, size_t dummy2)
{
	return vaddr;
}

/*
 * Name: kl_init_virtop()
 * Func: initialize virtual to physical address translation
 *       This function must at least initialize high_memory and init_mm.
 */
int
kl_init_virtop(void)
{
	syment_t *sp;
	
	if(!(sp = kl_lkup_symname("high_memory"))){
		/* XXX set error code */
		return(1);
	} else {
		/* Read (void *) high_memory from dump.
		   We need to make sure that the address is not virtually
		   mapped kernel data. We have to do this by hand since
		   we haven't got virtop mechanism working yet...
		*/
		KL_HIGH_MEMORY = KL_READ_PTR(sp->s_addr -
							 KL_PAGE_OFFSET);
		if (KL_ERROR) {
			KL_HIGH_MEMORY = (kaddr_t) -1;
			/* XXX set error code */
			return(1);
		}
	}

	/* if high_memory not set, we assume all addresses above PAGE_OFFSET
	 * to be physical ones (see macro KL_KADDR_IS_PHYSICAL in kl_mem.h)
	 */
	if(!KL_HIGH_MEMORY){
		KL_HIGH_MEMORY = (kaddr_t) -1;
	}

	/* Get the address of init_mm and convert it to a physical address
	 * so that we can make direct calls to kl_readmem(). We make a call
	 * to kl_vtop() since we have not finished setting up for calls to
	 * kl_virtop().
	 */
        if (!(sp = kl_lkup_symname("init_mm"))) {
		/* XXX set error code */
                return(1);
	} else {
		KL_INIT_MM = sp->s_addr - KL_PAGE_OFFSET;
	}

	if ((KL_ARCH == KL_ARCH_S390) || (KL_ARCH == KL_ARCH_S390X)){
		sp = kl_lkup_symname("max_low_pfn");
		if(!sp){
			sp = kl_lkup_symname("num_physpages");
		}
        } else {
		sp = kl_lkup_symname("num_physpages");
	}
	if(!sp){
		/* XXX set error code */
		return(1);
	}
	/* unsigned long num_physpages */
	NUM_PHYSPAGES = KL_READ_PTR(sp->s_addr - KL_PAGE_OFFSET);
	if (KL_ERROR) {
		/* XXX set error code */
		return(1);
	}

	/* Get the start address of the kernel page table
	 */
	if (!(sp = kl_lkup_symname("mem_map"))) {
		/*
		 * 'mem_map' symbol will not be defined for DISCONTIG memory.
		 * lcrash supports DISCONTIG memory from 2.6 onwards.
		 */
		if (!(sp  = kl_lkup_symname("pgdat_list"))) {
			/* XXX set error code */
			return(1);
		} else {
			MEM_MAP = 0;
			KL_PGDAT_LIST= KL_VREAD_PTR(sp->s_addr);
			if (KL_ERROR) {
				/* XXX set error code */
				return(1);
			}
		}
	} else {
		/* mem_map_t * mem_map */
		MEM_MAP = KL_READ_PTR(sp->s_addr - KL_PAGE_OFFSET);
		if (KL_ERROR) {
			/* XXX set error code */
			return(1);
		}
	}

	/* Get the cr4 settings. This will tell us if the system is PAE enabled
	 *
	 */
	if(KL_ARCH == KL_ARCH_I386){
		sp = kl_lkup_symname("mmu_cr4_features");
		if(sp){
			KL_KERNEL_FLAGS = KL_READ_PTR(sp->s_addr - KL_PAGE_OFFSET);
			if (KL_ERROR) {
				/* XXX set error code */
				return(1);
			}
		} else {
			return(1);
		}
	}

	return(0);
}
