/*
 * $Id: kl_nmlist.c,v 1.3 2005/02/23 01:09:12 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2005 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 */

#include <libelf.h>
#include <klib.h>
#include <kl_debug.h>

/* The nmlist[] array contains information about various namelists
 * open for access. The namelist referenced by nmlist[0] is considered
 * the primary namelist. It is used in the setup of the global type 
 * information. The rest of the namelists can be used for accessing 
 * additional type information  At that, only those types which are 
 * not defined in the primary namelist get added to the type tree. 
 */
nmlist_t nmlist[MAXNMLIST];
int numnmlist = 0;
int curnmlist;

extern int st_setup_typeinfo(void);
extern int st_open_namelist(char *filename, int flags);
extern int debug_format;

void check_namelist_format(char *);
struct namelist_format_opns *namelist_opns;

struct namelist_format_opns dwarf2_opns = {

        open_namelist:  dw_open_namelist, 
	setup_typeinfo: dw_setup_typeinfo
};

struct namelist_format_opns stabs_opns = {

        open_namelist:  st_open_namelist,
        setup_typeinfo: st_setup_typeinfo
};

/* 
 * kl_set_curnmlist()
 */
void
kl_set_curnmlist(int index)
{
	/* Make sure that there is at least one nmlist record. If there
	 * isn't, then that's not too big a problem, as only some 
	 * functionalty depends upon kernel type information being 
	 * available.
	 */
	if ((index >= 0) && (index < numnmlist)) {
		curnmlist = index;
	}
}

/*
 * kl_open_namelist()
 */
int
kl_open_namelist(char *namelist, int flags)
{
	int saved_nmlist;

	if (numnmlist == 0) {
		/* This is the first one being opened. make sure we 
		 * zero out the table memory.
		 */
		memset(nmlist, 0, sizeof(nmlist));
	} else if (numnmlist == (MAXNMLIST - 1)) {
		/* There isn't room for this namelist in the table
	 	 */
		return(1);
	}

	/* Set up the current nmlist
	 */
	saved_nmlist = curnmlist;
	curnmlist = numnmlist;
	numnmlist++;
	nmlist[curnmlist].index = curnmlist;
	nmlist[curnmlist].namelist = (char *)kl_str_to_block(namelist, K_PERM);
	nmlist[curnmlist].stringtab = kl_init_string_table(K_PERM);
	
	/* Check the debug format and initialize the function pointers */
	check_namelist_format(namelist);

	/* Open up the namelist 
	 */
	if ((*(namelist_opns->open_namelist)) (namelist, flags)) {
		numnmlist--;
		nmlist[curnmlist].index = 0;
		if (nmlist[curnmlist].namelist) {
			kl_free_block(nmlist[curnmlist].namelist);
			nmlist[curnmlist].namelist = 0;
		}
		if (nmlist[curnmlist].stringtab) {
			kl_free_string_table(nmlist[curnmlist].stringtab);
			nmlist[curnmlist].stringtab = (string_table_t *)NULL;
		}
		curnmlist = saved_nmlist;
		return(1);
	} 

	(*(namelist_opns->setup_typeinfo))();

	if (numnmlist == 1) {
		kl_init_struct_sizes();
	}

	return(0);
}

void check_namelist_format(char *filename) {

	int fd;
	char *name;
	Elf *elf_fd = NULL;
	Elf_Scn *ptr_section = NULL;
        
        Elf32_Ehdr *ptr32_eheader = NULL;
        Elf64_Ehdr *ptr64_eheader = NULL;
        
	Elf32_Shdr *ptr32_sheader = NULL;
        Elf64_Shdr *ptr64_sheader = NULL;

        Elf_Cmd command = ELF_C_READ;
	
        fd = open(filename,O_RDONLY);
        if(fd < 0) {
                fprintf(stderr,"check_namelist_format: open on namelist file failed\n");
                exit(1);
        }

        (void)elf_version(EV_NONE);
        if(elf_version(EV_CURRENT) == EV_NONE) {
                fprintf(stderr,"check_namelist_format: libelf outdated\n");
                exit(1);
        }

        command = ELF_C_READ;

	elf_fd = elf_begin(fd,command,(Elf*)0);
	if(!elf_fd) {
		fprintf(stderr,"check_namelist_format: elf_begin failed\n");
		exit(1);
        }

	/* Get the Elf header. Try elf32 first and if that doesn't work,
	 * try elf64 (if that doesn't work, return an error). 
	 */
	if(!(ptr32_eheader = elf32_getehdr(elf_fd))) {
		if(!(ptr64_eheader = elf64_getehdr(elf_fd))) {
			fprintf(stderr,"check_namelist_format: "
				"error while reading elf header\n");
			exit(1);                              
		}                                             
	}                                                     
							      
	/* Check to see if type information is in Dwarf2 or stabs format    
	 * by checking each section for its name. .debug_info is specific   
	 * to Dwarf2 debug format and .stab is specific to stab debug       
	 * format.                                            
	 */                                                   
	while((ptr_section = elf_nextscn(elf_fd, ptr_section)))  {
							      
		if (ptr32_eheader) {                          
			if(!(ptr32_sheader = elf32_getshdr(ptr_section))) { 
				fprintf(stderr,"check_namelist_format: "    
					"error while reading elf section "      
					"headers\n");         
				exit(1);                      
			}                                     
			name = elf_strptr(elf_fd, ptr32_eheader->e_shstrndx,
				(size_t)(ptr32_sheader->sh_name));
		} else { /* ptr64_eheader */                  
			if(!(ptr64_sheader = elf64_getshdr(ptr_section))) { 
				fprintf(stderr,"check_namelist_format: "    
					"error while reading section "      
					"headers\n");         
				exit(1);                      
			}                                     
			name = elf_strptr(elf_fd, ptr64_eheader->e_shstrndx,
				(size_t)(ptr64_sheader->sh_name));
		}                                             
							      
		if(strcmp(name,".debug_info") == 0) {         
			namelist_opns= &dwarf2_opns;          
			debug_format = DBG_DWARF2;            
			return;
		} else if(strcmp(name,".stab") == 0) {        
			namelist_opns= &stabs_opns;           
			debug_format = DBG_STABS;             
			return;                               
		}                                             
	} /* end of while loop */
}
