/*
 * $Id: kl_page_x86_64.c,v 1.1 2004/12/21 23:26:20 tjm Exp $
 *
 * This file is part of libklib.
 * A library which provides access to Linux system kernel dumps.
 *
 * Created by Silicon Graphics, Inc.
 * Contributions by IBM, NEC, and others
 *
 * Copyright (C) 1999 - 2002 Silicon Graphics, Inc. All rights reserved.
 * Copyright (C) 2001, 2002 IBM Deutschland Entwicklung GmbH, IBM Corporation
 * Copyright 2000 Junichi Nomura, NEC Solutions <j-nomura@ce.jp.nec.com>
 *
 * This code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or
 * (at your option) any later version. See the file COPYING for more
 * information.
 *
 * Added support for X86_64 architecture Mar 2004
 *      Prashanth Tamraparni (prasht@in.ibm.com)
 *      Sachin Sant (sachinp@in.ibm.com)
 */

#include <klib.h>

/*
 * declarations of static functions
 */
static kaddr_t _kl_pml4_offset_x86_64(
	kaddr_t        ); /* kernel virtual address of page directory */
static kaddr_t _kl_pgd_offset_x86_64(
	kaddr_t         /* kernel virtual address of page directory */,
	kaddr_t         /* kernel virtual address */);

static kaddr_t _kl_pmd_offset_x86_64(
	kaddr_t         /* kernel virtual address of page middle directory */,
	kaddr_t         /* kernel virtual address */);

static kaddr_t _kl_pte_offset_x86_64(
	kaddr_t         /* kernel virtual address of page table */,
	kaddr_t         /* kernel virtual address */);

/*
 * function definitions
 */

/* page table traversal functions */
static kaddr_t
_kl_pml4_offset_x86_64(kaddr_t vaddr)
{
	syment_t *sp;
	kaddr_t pml4_off, phy_level4, pgd_base = 0;

	if (!(sp = kl_lkup_symname("init_level4_pgt"))) {
        	/* XXX set error code */
        	return(0);
	} else {
		pml4_off = sp->s_addr + (((vaddr >> KL_PML4_SHIFT_X86_64_PAE) & 
					(KL_PTRS_PER_PML4_X86_64_PAE - 1)) * 8) ;
		
		phy_level4 = KL_DIRECT_MAPPING_X86_64(pml4_off);
		
		pgd_base = KL_READ_UINT64(phy_level4);
		return pgd_base;	
	}
}

static kaddr_t
_kl_pgd_offset_x86_64(kaddr_t pgd_base, kaddr_t vaddr)
{
	kaddr_t pgd_off, pmd_base = 0, phy_level3;

	pgd_base &= KL_PHYSICAL_MASK_X86_64;
	
	if(KL_KERNEL_FLAGS & KL_IS_PAE_X86_64){
		
		pgd_off = (((vaddr >> KL_PGDIR_SHIFT_X86_64_PAE) &
			   (KL_PTRS_PER_PGD_X86_64_PAE - 1)) * 8);

		pmd_base = KL_READ_UINT64(pgd_base + pgd_off);
	} 
	return pmd_base;
}

static kaddr_t
_kl_pmd_offset_x86_64(kaddr_t pmd_base, kaddr_t vaddr)
{
	kaddr_t pmd_off, pte_base=0, phy_level2;

	pmd_base &= KL_PHYSICAL_MASK_X86_64;

	if(KL_KERNEL_FLAGS & KL_IS_PAE_X86_64){
	
		pmd_off = (((vaddr >> KL_PMD_SHIFT_X86_64_PAE) &
			   (KL_PTRS_PER_PMD_X86_64_PAE - 1)) * 8);
		
		pte_base = KL_READ_UINT64(pmd_base + pmd_off);
		
	}
	return pte_base;
}

static kaddr_t
_kl_pte_offset_x86_64(kaddr_t pte_base, kaddr_t vaddr)
{
	kaddr_t pte_off, pte_val=0, phy_level1;

	pte_base &= KL_PHYSICAL_MASK_X86_64;

	if(KL_KERNEL_FLAGS & KL_IS_PAE_X86_64){

		pte_off = (((vaddr >> KL_PAGE_SHIFT_X86_64) &
			   (KL_PTRS_PER_PTE_X86_64_PAE - 1)) * 8);

		pte_val = KL_READ_UINT64(pte_base + pte_off);
	
	} 
	return pte_val;
}

/* lookup virtual address in page tables */
kaddr_t kl_mmap_virtop_x86_64(kaddr_t vaddr, void *mmp)
{
	kaddr_t pgd_base, pmd_base;
	kaddr_t pte_base, pte_val, paddr=0;
	kaddr_t pml4_base;

	pgd_base = _kl_pml4_offset_x86_64(vaddr);
	if (KL_ERROR) {
		KL_ERROR = KLE_INVALID_MAPPING;
		return(0);
	}
	/* get the pgd entry */
	pmd_base = _kl_pgd_offset_x86_64(pgd_base,vaddr);
	if (KL_ERROR) {
		KL_ERROR = KLE_INVALID_MAPPING;
		return(0);
	}

	if(KL_KERNEL_FLAGS & KL_IS_PAE_X86_64){
		/* get the pmd entry */
		pte_base = _kl_pmd_offset_x86_64(pmd_base, vaddr);
		if (KL_ERROR) {
			KL_ERROR = KLE_INVALID_MAPPING;
			return(0);
		}
		
		/* 2MByte Page translation support in long mode */
		if(pte_base & KL_PAGE_PML4E_X86_64) {
        		if (kl_pte_none_x86_64(pte_base)){
                		KL_ERROR = KLE_INVALID_MAPPING;
                		return(0);
        		}
        		if (!kl_pte_present_x86_64(pte_base)){
                		KL_ERROR = KLE_PAGE_NOT_PRESENT;
                		return(0);
        		}

        		paddr = (pte_base & KL_PHYSICAL_MASK_X86_64) |
                		(vaddr & KL_PSE_OFFSET_MASK_X86_64_PAE);
		} else {
			if (kl_pmd_bad_x86_64(pte_base) || 
			    kl_pmd_none_x86_64(pte_base)) {
				KL_ERROR = KLE_INVALID_MAPPING;
				return(0);
			}
			/* get the pte */
			
			pte_val = _kl_pte_offset_x86_64(pte_base,vaddr);
			if (KL_ERROR){
				return 0;
			}
			if (kl_pte_none_x86_64(pte_val)){
				KL_ERROR = KLE_INVALID_MAPPING;
				return 0;
			}
			if (!kl_pte_present_x86_64(pte_val)){
				KL_ERROR = KLE_PAGE_NOT_PRESENT;
				return (0);
			}

			paddr = (pte_val & KL_PHYSICAL_MASK_X86_64) | 
				(vaddr & (~(KL_PAGE_MASK_X86_64)));
		}

	}

	return(paddr);
}
