/*
 *   Copyright (C) 2002 by Phil Karn KA9Q
 *   Copyright (C) 2003 by Jonathan Naylor G4KLX/HB9DRD
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "Noise.h"
#include "Gaussian.h"

#include "common/Exception.h"
#include "common/SoundFile.h"

#include <cmath>
#include <limits.h>
using namespace std;

int main(int argc, char *argv[])
{
	if (argc < 4) {
		::fprintf(stderr, "Usage: Noise <snr> <infile> <outfile>\n");
		return 1;
	}

	double           snr = ::pow(10.0, ::atof(argv[1]) / 20.0);
	wxString inFileName  = wxString(argv[2]);
	wxString outFileName = wxString(argv[3]);

	try {
		CNoise noise(snr, inFileName, outFileName);
		noise.run();
	}
	catch (CException& ex) {
		::fprintf(stderr, "Noise: error: %s\n", ex.getMessage().c_str());
		return 1;
	}
	catch (...) {
		::fprintf(stderr, "Noise: an error has occured\n");
		return 1;
	}

	return 0;
}

CNoise::CNoise(double snr, const wxString& inFileName, const wxString& outFileName) :
m_snr(snr),
m_inFileName(inFileName),
m_outFileName(outFileName)
{
}

CNoise::~CNoise()
{
}

void CNoise::run()
{
	CSoundFile in;
	in.openRead(m_inFileName, 11025, 8);

	CSoundFile out;
	out.openWrite(m_outFileName, 11025, 8);

	CGaussian gaussian;

	double samples[4096];

	for (;;) {
		int len = 4096;
		bool ret = in.read(samples, len);

		if (!ret) break;

		double energy = 0.0;
		for (int i = 0; i < len; i++)
			energy += samples[i] * samples[i];

		/* Compute signal, noise amplitudes */
		double signalAmpl = ::sqrt(energy / len);
		double noiseAmpl  = signalAmpl / m_snr;

		/* Set gain to make signal + noise amplitude 12 dB below clipping */
		double gain = 1.0 / (4.0 * (signalAmpl + noiseAmpl));

		for (int i = 0; i < len; i++){
			double s = gain * gaussian.rand(samples[i], noiseAmpl);
			samples[i] = (s > 1.0) ? 1.0 : (s < -1.0 ? -1.0 : s);
		}

		out.write(samples, len, 1.0);
	}

	in.close();
	out.close();
}
