package Lire::Firewall::IpfilterDlfConverter;

use strict;

use Lire::DlfConverter;
use Lire::Firewall qw/firewall_number2names/;
use Lire::Syslog;
use Carp;

use base qw/Lire::DlfConverter/;

my %action2cisco = (
    'b' => 'denied',
    'a' => 'permitted'
);


sub new {
    my $proto = shift;
    bless {}, (ref $proto || $proto);
}

sub name { 'ipfilter' }
sub title { 'ipf firewall log' }
sub description { '<para>ipf firewall log</para>' }
sub schemas { qw/firewall/ }

sub handle_log_lines { 1 }

sub init_dlf_converter {
    my ($self, $process) = @_;

    $self->{'parser'} = new Lire::Syslog;
}

sub process_log_line {
    my ($self, $process, $line) = @_;

    #
    # skip invalid log entries
    #
    return $process->ignore_log_line($line)
        unless $line =~ /ipmon/;

    eval {
        my $log = $self->{'parser'}->parse($line);
        local $_ = $log->{'content'};
        #
        # skip invalid log entries
        #
        return $process->ignore_log_line($line)
            unless defined $log->{'process'} && $log->{'process'} eq 'ipmon';

        my @line = split;

        my %dlf = (
            'time' => $log->{'timestamp'},
        );

        if(defined $line[1] and $line[1] =~ /^(\d+)x$/) {
            # print as many dlf's as the number indicates
            $dlf{'count'} = $1;
            shift @line;
        } else {
            $dlf{'count'} = 1;
        }

        defined $line[0] and $dlf{'iptime'} = $line[0];
        defined $line[1] and $dlf{'rcv_intf'} = $line[1];
        defined $line[2] and $dlf{'rule'} = $line[2];
        defined $line[3] and $dlf{'action'} = $line[3]; # b or p

        if(defined $line[4]) {
            my ($ip, $port) = split /,/, $line[4];
            $dlf{'from_port'} =  $port;
            ($dlf{'from_host'}, $dlf{'from_ip'}) =
                $ip =~ /^(?:(.*)\[)?([.0-9]+)\]?$/;
        }

        if(defined $line[5]) {
            $line[5] eq '->' or
                return $process->error($line,
                    "skipping line '$_': invalid format: fifth field " .
                    "should be '->', not '". $line[5] ."'");
        }

        if(defined $line[6]) {
            my ($ip, $port) = split /,/, $line[6];
            $dlf{'to_port'} =  $port;
            ($dlf{'to_host'}, $dlf{'to_ip'}) =
                $ip =~ /^(?:(.*)\[)?([.0-9]+)\]?$/;
        }

        if(defined $line[7]) {
            $line[7] eq 'PR' or
                return $process->error($line,
                    "skipping line '$_': invalid format: seventh field " .
                    "should be 'PR', not '". $line[7] ."'");
        }

        defined $line[8] and $dlf{'protocol'} = $line[8]; # one of tcp, udp, icmp
        if(defined $line[9]) {
            $line[9] eq 'len' or
                return $process->error($line,
                    "skipping line '$_': invalid format: ninth field " .
                    "should be 'len', not '". $line[9] ."'");
        }

        if(defined $line[10] and defined $line[11]) {
            # sum lenght of header and lenght of packet

            # Jul  6 08:13:15 rolle ipmon[2658]: 08:13:15.373154 ep1 @0:7 b
            #    100.187.115.1 -> 100.0.0.1 PR igmp len 24 (32) IN
            $line[11] =~ s/^[^\d]*(\d+)[^\d]*$/$1/;

            $dlf{'length'} = $line[10] + $line[11];
            $dlf{'length'} *= $dlf{'count'};
        }

        if(defined $line[12]) {
            if($line[12] =~ /^-\w+$/) {
                $dlf{'flags'} = $line[12];
                defined $line[13] and $dlf{'direction'} = $line[13];
            } elsif ($line[12] eq 'icmp') {
                splice(@line, 0, 13);

                if($line[-1] eq 'OUT' or $line[-1] eq 'IN') {
                    $dlf{'direction'} = $line[-1];
                    pop @line;
                }

                $dlf{'flags'} = join('_', @line);
            } else {
                $dlf{'direction'} = $line[12];
            }
        }

        $dlf{'action'} = $action2cisco{$dlf{'action'}}
            if exists $action2cisco{$dlf{'action'}};

        #
        # convert numbers to names and create dlf-record
        #
        firewall_number2names(\%dlf);
        $process->write_dlf('firewall', \%dlf);
    };

    if($@) {
        $process->error($line, $@);
    }
}

sub finish_conversion {
    delete $_[0]->{'parser'};
}

1; # nag nag.

__END__

=pod

=head1 NAME

Lire::Firewall::IpfilterDlfConverter - convert ipf (ipmon) logs to firewall DLF

=head1 DESCRIPTION

B<Lire::Firewall::IpfilterDlfConverter> converts Ipfilter logs into
firewall DLF format.
Input for this converter is the standard ipf syslog log file as produced
by ipmon. IP Filter is shipped with FreeBSD, OpenBSD (up to 2.9)
and some other OS's.

=head1 EXAMPLE

A ipfilter logfile which looks like

 Oct 30 07:42:29 rolle ipmon[16747]: 07:42:28.585962              ie0 @0:9
  b 192.168.48.1,45085 -> 192.168.48.2,22 PR tcp len 20 64 -S OUT
 Oct 30 07:40:24 rolle ipmon[16747]: 07:40:23.631307              ep1 @0:6
  b 192.168.26.5,113 -> 192.168.26.1,3717 PR tcp len 20 40 -AR OUT
 Oct 30 07:42:29 rolle ipmon[16747]: 07:42:28.585962              ie0 @0:9
  b 192.168.48.1,45085 -> 192.168.48.2,22 PR tcp len 20 64 -S OUT
 Oct 30 07:44:11 rolle ipmon[16747]: 07:44:10.605416 2x              ep1 @0:15
  b 192.168.26.1,138 -> 192.168.26.255,138 PR udp len 20 257  IN
 Oct 30 07:44:34 rolle ipmon[16747]: 07:44:33.891869              ie0 @0:10
  b 192.168.48.1,23406 -> 192.168.48.2,22 PR tcp len 20 64 -S OUT
 Oct 30 07:49:13 rolle ipmon[16747]: 07:49:12.554420              ep1 @0:15
  b 210.132.100.117 -> 192.168.26.5 PR icmp len 20 56 icmp 3/3 for
  192.168.26.5,61915 - 210.132.100.117,53 PR udp len 20 23040 IN
 Oct 30 07:50:23 rolle ipmon[16747]: 07:50:22.908107              ep1 @0:15
  b 210.132.100.117 -> 192.168.26.5 PR icmp len 20 56 icmp 3/3 for
  192.168.26.5,4480 - 210.132.100.117,53 PR udp len 20 19712 IN
 Oct 30 07:56:11 rolle ipmon[16747]: 07:56:11.113029 2x              ep1 @0:15
  b 192.168.26.1,138 -> 192.168.26.255,138 PR udp len 20 257  IN

(that's: .... 'PR' protocol 'len' length_of_ip_headers_saved packetlength
direction) will get converted to something like

 994398737 denied igmp 100.187.115.1 - ep1 LIRE_NOTAVAIL 224.0.0.2 - 56
 994398861 denied igmp 100.187.115.1 - ep1 LIRE_NOTAVAIL 224.0.0.1 - 56
 994398862 denied igmp 100.187.115.1 - ep1 LIRE_NOTAVAIL 224.0.0.2 - 56
 994406849 denied udp 192.168.26.4 137 ie0 LIRE_NOTAVAIL 192.168.26.255 137 116
 994406850 denied udp 192.168.26.4 137 ie0 LIRE_NOTAVAIL 192.168.26.255 137 116
 994406866 denied udp 192.168.26.4 137 ie0 LIRE_NOTAVAIL 192.168.26.255 137 98

=head1 SEE ALSO

ipl(4) for description of log structure.

The ipmon.c source (e.g. on
http://www.openbsd.org/cgi-bin/cvsweb/~checkout~/src/usr.sbin/ipmon/Attic/ipmon.c?rev=1.27&content-type=text/plain&hideattic=0
for the specification) of the log syntax.

The IP Filter webpage on http://coombs.anu.edu.au/~avalon/ip-filter.html

=head1 AUTHOR

Joost van Baal <joostvb@logreport.org>,
Wessel Dankers <wsl@logreport.org>

=head1 VERSION

$Id: IpfilterDlfConverter.pm,v 1.6 2006/07/23 13:16:35 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2001-2003 Stichting LogReport Foundation LogReport@LogReport.org

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html.

=cut

# Local Variables:
# mode: cperl
# End:
