package Lire::AsciiDlf::Timegroup;

use strict;

use vars qw( $VERSION @ISA $year_sec $month_sec %MONTHS );

use Lire::Timegroup;
use Lire::DataTypes qw( :time );

use POSIX qw( strftime );
use Time::Local;
use Time::Timezone;

use Carp;

BEGIN {
    ($VERSION)	= '$Revision: 1.15 $' =~ m!Revision: ([.\d]+)!;
    @ISA = qw( Lire::Timegroup );

    # Since duration2sec uses approximation for those period,
    # get the values of those approximation
    $year_sec	= duration2sec( "1y" );
    $month_sec	= duration2sec( "1M" );

    # We don't want to use the %B strftime because of possible
    # charsets conflicts with locale settings
    my $i = 0;
    %MONTHS = map { $i++ => $_ } qw( January February March
				     April   May      June
				     July    August   September
				     October November December );
}

sub init_report {
    my $self = shift;

    $self->{field_idx} =
      $self->{report_spec}->schema->field( $self->field )->pos();

    my $period = $self->period;
    if ( $period =~ /^\$/ ) {
	$period = substr $period, 1;
	$period = $self->{report_spec}->param( $period )->value;
    }

    $self->{period_sec} = duration2sec( $period );

    # The way to determine the index in the timeslices is different
    # for values above 1M and 1y
    # We determine if the period was of X month types if it is
    # an even multiple of the 1M approximation
    if ( $self->{period_sec} >= $month_sec &&
	 !($self->{period_sec} % $month_sec) ) {
	$self->{use_month} = 1;
    } elsif ( $self->{period_sec} >= $year_sec &&
	      !($self->{period_sec} % $year_sec)) {
	$self->{use_year} = 1;
    } else {
	$self->{use_sec} = 1;
    }
    $self->{started}    = 0;
    $self->{timeslices} = [];

    # FIXME: Backward compatibily hack. This should be done
    # in the stylesheets
    if ( $self->{use_year} ) {
	$self->{time_fmt} = '%Y';
    } elsif ( $self->{period_sec} >= 86400 ) { # 1d
	$self->{time_fmt} = '%Y-%m-%d';
    } elsif ( $self->{period_sec} >= 60 ) {
	$self->{time_fmt} = '           %H:%M';
    } else {
	$self->{time_fmt} = '           %H:%M:%S';
    }

    foreach my $op ( @{$self->{ops}}) {
	$op->init_report( @_ );
    }

    $self;
}

sub update_report {
    my ( $self, $dlf ) = @_;

    my $time = $dlf->[$self->{field_idx}];

    unless ( $self->{started} ) {
	# Calculate the start of the period
	if ( $self->{use_year} ) {
	    $self->{year_start} = (localtime $time)[5];
	} elsif ( $self->{use_month} ) {
	    my ($month,$year) = (localtime $time)[4,5];
	    $self->{month_start}  = $month;
	    $self->{year_start}   = $year;
	} else {
	    # The start of the period is offset by the timezone offset.
	    # We need only do this once, because UTC doesn't have a standard
	    # vs. daylight saving issue. This means that all subsequent period
	    # will have the proper localtime value.
	    $self->{tz_offset} = tz_local_offset( $time );
	    $self->{start} = int( $time / $self->{period_sec})
	      * $self->{period_sec};

	    # FIXME: Can somebody explain to me why is this working.
	    # This was found heuristically and I'm not sure I
	    # understand why this works. -- FJL
	    $self->{start} -= $self->{tz_offset}
	      if abs($self->{tz_offset}) < $self->{period_sec};
	}
	$self->{started} = 1;
    }

    my $idx;
    if ( $self->{use_sec} ) {
	$idx = int( ($time - $self->{start}) / $self->{period_sec});
    } elsif ( $self->{use_month} ) {
	my ($month,$year) = (localtime $time)[4,5];

	if ( $year > $self->{year_start}) {
	    $idx = ($year - $self->{year_start}) * 12 + $month;
	} else {
	    $idx = $month - $self->{month_start};
	}
    } else {
	my $year = (localtime $time)[5];
	$idx = $year - $self->{year_start};
    }

    # Negative time index, means that the data wasn't sorted by the time
    # field. We could shift the time slice array to fix this, but we could
    # switch from or to DST with this, so don't do it.
    die "found negative time slice index ($idx). DLF file isn't sorted by the time field\n"
      if $idx < 0;

    my $data = $self->{timeslices}[$idx];
    unless ( defined $data ) {
	my $start;
	if ( $self->{use_sec} ) {
	    $start = $idx * $self->{period_sec} + $self->{start};
	} elsif ( $self->{use_month} ) {
	    $start = timelocal( 0, 0, 0, 1, (localtime $time)[4,5] );
	} else {
	    $start = timelocal( 0, 0, 0, 1, 0, (localtime $time)[5] );
	}
	$data = [ $start ];

	my $i = 1;
	foreach my $op ( @{$self->ops} ) {
	    $data->[$i++] = $op->init_group_data();
	}

	$self->{timeslices}[$idx] = $data;
    }


    my $i = 1;
    foreach my $op ( @{$self->ops} ) {
	$op->update_group_data( $dlf, $data->[$i++] );
    }

    $self;
}

sub end_report {
    my ( $self ) = @_;

    # Finalize each timeslice
    # Either create empty one or call end_group_data on them
    my $last_idx = $#{$self->{timeslices}};
    my $i = 0;
    while ( $i <= $last_idx) {
	if ( $self->{timeslices}[$i]) {
	    my $data = $self->{timeslices}[$i];
	    my $j = 1;
	    foreach my $op ( @{$self->ops} ) {
		$op->end_group_data( $data->[$j++] );
	    }
	} else {
	    # Create empty set
	    my $start;
	    if ( $self->{use_sec} ) {
		$start = $i * $self->{period_sec} + $self->{start};
	    } elsif ($self->{use_month} ) {
		my $year_off  = int( ($self->{month_start} + $i) / 12);
		if ( $year_off ) {
		    my $month = $i - ($year_off * 11);
		    $start = timelocal( 0, 0, 0, 1, $month,
					$self->{year_start} + $year_off );
		} else {
		    $start = timelocal( 0, 0, 0, 1, $self->{month_start} + $i,
					$self->{year_start} );
		}
	    } else {
		$start = timelocal( 0, 0, 0, 1, 0, $self->{year_start} + $i );
	    }
	    my $data = [ $start ];

	    my $j = 1;
	    foreach my $op ( @{$self->ops} ) {
		$data->[$j] = $op->init_group_data();
		$op->end_group_data( $data->[$j++] );
	    }

	    $self->{timeslices}[$i] = $data;
	}
	$i++;
    }

    $self;
}

sub write_report {
    my ( $self, $fh, $prefix ) = @_;
    $fh	    ||= \*STDOUT;
    $prefix ||= 0;

    my $pfx = ' ' x $prefix;

    my $last_day = 0;
    foreach my $tslice ( @{$self->{timeslices}} ) {
	print $fh $pfx, qq{<lire:entry>\n};

	# FIXME: Special backward-compatibility hack.
	# This should be done in the stylesheet.
	print $fh $pfx, " <lire:name>";
	if ( $self->{use_month}) {
	    my $month = (localtime $tslice->[0] )[4];
	    print $MONTHS{$month};
	} else {
	    my $fmt = $self->{time_fmt};
	    if ( $self->{period_sec} < 86400 ) { # 1d
		# Make sure that date change are indicated
		if ( $tslice->[0] >= $last_day + 86400 ) {
		    $fmt = '%Y-%m-%d %H:%M';
		    $last_day = $tslice->[0] - ($tslice->[0] % 86400)
		      - $self->{tz_offset};
		}
	    }
	    print $fh strftime( $fmt, localtime $tslice->[0] );
	}
	print $fh "</lire:name>\n";

	my $i = 1;
	foreach my $op ( @{$self->ops} ) {
	    $op->write_group_data( $fh, $prefix + 1, $tslice->[$i++] );
	}
	print $fh $pfx, qq{</lire:entry>\n};
    }

    $self;
}


# keep perl happy
1;

__END__

=pod

=head1 NAME

Lire::AsciiDlf::Timegroup -

=head1 SYNOPSIS


=head1 DESCRIPTION

=head1 VERSION

$Id: Timegroup.pm,v 1.15 2001/12/13 21:08:17 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=cut
