package Lire::Syslog;

use strict;

use Lire::Time qw/ syslog2cal clf2cal /;
use Lire::Program qw( :msg );
use vars qw/ $LTIME /;

BEGIN {
    $LTIME = [ localtime ];
}

#
# Using one parser object returning record in hash is nearly 50% faster
# then using one object by syslog record.
#
# Testing on 22410 syslog lines
# Benchmark: timing 10 iterations of 1 line = 1 object, parser object...
# 1 line = 1 object: 24 wallclock secs (24.23 usr +  0.01 sys = 24.24 CPU) @  0.41/s (n=10)
# parser object: 17 wallclock secs (16.52 usr +  0.00 sys = 16.52 CPU) @  0.61/s (n=10)
#                  s/iter 1 line = 1 object     parser object
# 1 line = 1 object   2.42                --              -32%
# parser object       1.65               47%                --
#
sub new {
    my $self = shift;
    my $class = ref($self) || $self;
    bless $self = {
		   syslog_style	=> 1,
		   date_re	=> undef,
		  }, $class;

    return $self;
}

sub parse {
    my ($self, $line) = @_;

    # Determine timestamp style on first line
    unless ( defined $self->{date_re} ) {
	if ( $line =~ m,^\[\d\d/[a-zA-Z]{3}/\d{4}, ) {
	    lr_info( __PACKAGE__, " log is from Netscape syslog");
	    $self->{date_re} = qr/\[.*?\]/;
	    $self->{syslog_style} = 0;
	} else {
	    lr_info( __PACKAGE__, " log is from classic syslog");
	    $self->{date_re}    = qr/\S+ ? \S+ \S+/;
	}
    }

    my ( $date, $hostname, $process, $pid, $id, $facility, $level, $content ) =
      ( $line =~
	m/^ ($self->{date_re})\s+
            (\S+)\s+           # $hostname = hostname
            ([^:\[]+)          # $process = process name + PID
            (?:\[(\d+)\])?:\s+ # $pid = pid, could be undef
            (?:\[ID\s	       # Special Solaris 8 identifier
	       (\d+)\s	       # ID ?
	       ([a-z0-9]+)\.([a-z]+) # Facility.loglevel
	      \]\s+)?
            (.*)               # $content = "the rest" (to be split later)
        $/x
      )
	or die "invalid syslog line: $line\n";

    my $timestamp;
    if ( $self->{syslog_style} ) {
	my ( $month, $day, $time ) = split /\s+/, $date;
	$timestamp = syslog2cal( $month, $day, $time, $LTIME );
    } else {
	$timestamp = clf2cal( $date );
    }

    return {
	    timestamp	=> $timestamp,
	    hostname	=> $hostname,
	    process	=> $process,
	    content	=> $content,
	    pid		=> $pid,
	    identifier	=> $id,
	    facility	=> $facility,
	    level	=> $level,
	   };
}

1;

__END__

=pod

=head1 NAME

Lire::Syslog - syslog style lines parser

=head1 SYNOPSIS

use Lire::Syslog;

my $parser = new Lire::Syslog;

my $rec = $parser->parse( $line );

=head1 DESCRIPTION

This module defines objects able to parse syslog lines.
Syslog lines have a leading part which looks something like:

 Mar 30 09:42:41 rolle ipmon[9498]:

It can also parse log files generated by Netscape log daemon which looks 
like this:

 [01/Nov/2001:08:27:44 +0100] rolle ipmon[2864]: 

It also support the extra fields added by Solaris 8 log daemon:

 Mar 30 09:42:41 rolle ipmon[9498]: [ID 801593 mail.info]

Keys such as 'timestamp', 'hostname', 'process', 'pid', 'identifier',
'faciliy', 'level' and 'content' are defined.

=head1 USAGE

 package Lire::Foo;
 require Lire::Syslog;

 @ISA = qw/ Lire::Syslog /;

 sub parse {
    my $self = shift;
    my $line = shift;

    # this runs parse from Lire::Syslog, setting keys like 'day', 'process'
    # and 'hostname'
    my $rec = $self->SUPER::parse($line);

    $rec->{'foo'} = dosomethingwith( $rec->{content} );

    return $rec
 }

Now, one can run in a script

 my $parser = new Lire::Foo();

 while ( <> ) {
    chomp;
    my $log = $parser->parse( $line );
 }

which sets $log->{'day'}, ... $log->{'process'} and $log->{'foo'}.


=head1 SEE ALSO

Lire::Email(3)

=head1 VERSION

$Id: Syslog.pm,v 1.20 2002/01/24 16:42:26 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2000-2001 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.


=head1 AUTHOR

Joost van Baal, based on an idea by Joost Kooij

=cut

