package Lire::WWW::URL;

use strict;

# use vars qw/$REVISION $VERSION/;

sub new {
     my $class = shift;

     my $self = {};
     bless $self, $class;

     if (@_) {
        $self->parse(@_);
     }
     return $self;
}

sub parse {
    my $self = shift;
    my ( $url ) = @_;

    # e.g.
    # http://ds.internic.net/instructions/overview.html#WARNING

    my ( $scheme, $authority, $path, $query, $fragment) = (
      $url =~ m/^
        (?:
          ([^:\/?\#]+):     # scheme
        )?
        (?:
          \/\/([^\/?\#]*)   # authority
        )?
        ([^?\#]*)           # path
        (?:
          \?([^\#]*)        # query
        )?
        (?:
          \#(.*)            # fragment
        )?$/x
    ) or die "invalid url '$url'";

    # fullblown
    #
    #  authority     = server | reg_name
    #  server        = [ [ userinfo "@" ] hostport ]
    #  hostport      = host [ ":" port ]
    #  host          = hostname | IPv4address
    #
    # is not yet supported
    defined $authority and my ( $host, $port ) = (
      $authority =~ m/^([-\.a-zA-Z0-9]+)(?::(\d+))?$/
    );

    defined $scheme   and $self->{'scheme'}   = $scheme;
    defined $host     and $self->{'host'}     = $host;
    defined $port     and $self->{'port'}     = $port;
    defined $path     and $self->{'path'}     = $path;
    defined $query    and $self->{'query'}    = $query;
    defined $fragment and $self->{'fragment'} = $fragment;

    return 1;
}

1;

__END__

=pod

=head1 NAME

Lire::WWW::URL - analyze urls

=head1 SYNOPSIS

 use Lire::WWW::URL;

 my $url = new Lire::WWW::URL;

 $url->parse( 'http://www.example.com:8080/index.html' );

 my $url = Lire::WWW::URL->new( 'ftp://localhost/' );

 my $port = $url->{'port'} || '-';

=head1 DESCRIPTION

Lire::WWW::URL objects have 'scheme', 'host', 'port', 'path', 'query' and
'fragment' fields, if defined.

=head1 DEVELOPERS

Do not add extra functionality to this module!  If more functionality is
needed, be sure to consider dropping this module, and make Lire requiring the
URI CPAN module.  (Available for Debian as liburi-perl.)  Be aware URI depends
on MIME::Base64.

=head1 BUGS

This module is somewhat http biased.  Support for ftp and other schemes is
sloppy.

=head1 SEE ALSO

RFC 1738, RFC 2396, URI(3pm): a CPAN module by Gisle Aas, Graham Barr
e.a.

=head1 VERSION

$Id: URL.pm,v 1.2 2002/02/03 22:35:41 vanbaal Exp $

=head1 COPYRIGHT

Copyright (C) 2001  Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=head1 AUTHORS

Joost van Baal <joostvb@logreport.org>

=cut

