/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quar@vitea.pl> */

#include "include/support.h"
#include "include/piece.h"
#include "include/environment.h"
#include "include/wizard.h"
#include <string>

#define BACKWARD_COMPATIBILITY 1

static const GtkTargetEntry target_table[] = {
  { (gchar *) "application/lwizard", GTK_TARGET_SAME_APP, 0 }
};

static const guint n_targets = sizeof(target_table) / sizeof(target_table[0]);

static const gboolean dndpermissions_table[5][5] = {
/* FROM / TO 	   ICONS, COMMANDS, WORLD, PROGRAM, PLAYGROUND */
/* ICONS      */ { FALSE,    FALSE,  TRUE,    TRUE,      FALSE },
/* COMMANDS   */ { FALSE,    FALSE, FALSE,    TRUE,      FALSE },
/* WORLD      */ {  TRUE,     TRUE,  TRUE,   FALSE,      FALSE },
/* PROGRAM    */ {  TRUE,     TRUE, FALSE,    TRUE,      FALSE },
/* PLAYGROUND */ { FALSE,    FALSE, FALSE,   FALSE,      FALSE }
};

/*! \brief Obsługa zdarzenia pobrania danych za pomocą mechanizmu DND
*/
static void  
source_drag_data_get  (GtkWidget          *widget,
						GdkDragContext    *context,
						GtkSelectionData  *selection_data,
						guint             info,
						guint             time,
						gpointer          data)
{
	/* Lista zmiennych nieużywanych */
	(void) widget;
	(void) context;
	(void) info;
	(void) time;

	gtk_selection_data_set (selection_data,
			    selection_data->target,
                            8, (guchar *) &data, sizeof (gpointer));  

}

/*! \brief Obsługa zdarzenia otrzymania danych za pomocą mechanizmu DND
*/
static void  
target_drag_data_received  (GtkWidget          *widget,
							GdkDragContext     *context,
							gint                x,
							gint                y,
							GtkSelectionData   *data,
							guint               info,
							guint               time,
							gpointer           userdata)

{
	/* List of unused variables */
	(void) widget;
	(void) x;
	(void) y;
	(void) info;
	
	if ((data->length == 0) || (data->format != 8)) {
    	gtk_drag_finish (context, FALSE, FALSE, time);
 
    	return;
    }
	
	LWPiece *sourcepiece = * ((LWPiece **) data->data);
	LWPiece *destpiece = (LWPiece *) userdata;

	if (TRUE == sourcepiece->getRow()->isPieceDummy(sourcepiece)) {
		
		/* Don't touch additional pieces at the end of lines,
		 * in programming mode */
		gtk_drag_finish (context, TRUE, FALSE, time);
		return;
	}
	
	/* Moving single piece in line to the same line
	 * doesn't make a sens, additionaly it can produce seg. fault
	 * in some cases, don't allow such situation */
	if ((sourcepiece->row == destpiece->row) &&
		(NULL == sourcepiece->row->getPieceNth(1))) {
			gtk_drag_finish (context, TRUE, FALSE, time);
			return;
	}

	if (dndpermissions_table [sourcepiece->getRow()->getBoard()->getType()]
		[destpiece->getRow()->getBoard()->getType()] == FALSE) {
			gtk_drag_finish (context, TRUE, FALSE, time);
			return;
	}

	LWSymbol *symbol = sourcepiece->getSymbol();
								
	/* Negotiated action : copying */	
	if (context->action == GDK_ACTION_COPY || context->action == GDK_ACTION_LINK) {
		if (symbol != NULL)
			symbol->onDndCopy(sourcepiece, destpiece);
		else
		if (LW_TYPE_PROGRAM == destpiece->getRow()->getBoard()->getType()) {
			LWPiece *newpiece = new LWPiece (sourcepiece, destpiece->getRow());
			destpiece->getRow()->insertPieceBefore (newpiece, destpiece);
		} 
		else
			destpiece->setBackgroundPixmap (sourcepiece->getBackgroundPixmap());
	} 
	
	/* Negotiated action: moving */
	if (context->action == GDK_ACTION_MOVE) {
		if (symbol != NULL)
			symbol->onDndMove(sourcepiece, destpiece);		
		else
		if (sourcepiece->ifDNDWillMoveWizard()) {
			LWWizard *wizard = destpiece->getRow()->getBoard()->getWizard();
			g_return_if_fail (wizard != NULL);
			wizard->setPiece(destpiece);
		} 
		else 
		if (LW_TYPE_WORLD == destpiece->getRow()->getBoard()->getType()) {
			destpiece->setBackgroundPixmap (sourcepiece->getBackgroundPixmap());
			sourcepiece->setBackgroundPixmap(NULL);
		}
		else 
		{
			LWPiece *newpiece = new LWPiece (sourcepiece, destpiece->row);
			destpiece->getRow()->insertPieceBefore (newpiece, destpiece);
			sourcepiece->getRow()->removePiece (sourcepiece);			
		}
	}

	gtk_drag_finish (context, TRUE, FALSE, time);
}

/*! \brief Komponuj obrazek

	Komponuje obrazek z dwóch warstw
	\param image Kontrolka obrazka, w którym znajdzie się kompozycja
	\param forepixmap Obrazek na wierzchu
	\param backpixmap Obrazek pod spodem
	\param size Oczekiwany rozmiar obrazka
*/
static void
composite_image(GtkWidget *image,
				LWPixmap *forepixmap,
				LWPixmap *backpixmap,
				gint size)
{
	
	g_return_if_fail (image != NULL);
	g_return_if_fail (size > 0);
					
					
	GdkPixbuf *tmppixbuf=NULL;
	
	if ((backpixmap == NULL) && (forepixmap == NULL)) return;
	
	if (backpixmap != NULL) {
		if ((gdk_pixbuf_get_width (backpixmap->getPixbuf()) == size) &&
			(gdk_pixbuf_get_height (backpixmap->getPixbuf()) == size))
				tmppixbuf = gdk_pixbuf_copy (backpixmap->getPixbuf());
		else
				tmppixbuf = gdk_pixbuf_scale_simple (backpixmap->getPixbuf(),
					size, size, GDK_INTERP_BILINEAR);
	
		if (forepixmap != NULL)
			gdk_pixbuf_composite (forepixmap->getPixbuf(), tmppixbuf, 
						0, 0, /* Cooridinates of left top corner */
						size, size, /* Size of scaled image */
						0.0, 0.0, /* offset, don't crop left and top side of an image */
						(gdouble) size / (gdouble) gdk_pixbuf_get_width (forepixmap->getPixbuf()),
						(gdouble) size / (gdouble) gdk_pixbuf_get_height (forepixmap->getPixbuf()),
						GDK_INTERP_BILINEAR, /* interpolation */
						255); /* Alfa for background image */
	} else {
		if ((gdk_pixbuf_get_width (forepixmap->getPixbuf()) == size) &&
			(gdk_pixbuf_get_height (forepixmap->getPixbuf()) == size)) {
				tmppixbuf = gdk_pixbuf_copy (forepixmap->getPixbuf());
			} else {
		
				tmppixbuf = gdk_pixbuf_scale_simple (forepixmap->getPixbuf(),
					size, size, GDK_INTERP_BILINEAR);
			}
	}
		
	gtk_image_set_from_pixbuf (GTK_IMAGE (image), tmppixbuf);
	g_object_unref (G_OBJECT (tmppixbuf));
}

/*! \brief Tip

	Tip associated with a piece
*/
GtkTooltips *LWPiece::tooltips=NULL;

void LWPiece::init (LWRow *row)
{
	g_return_if_fail ( row != NULL );
	g_return_if_fail ( row->getBoard() != NULL );

	guint size = row->getBoard()->getPieceSize();
	
	widget = gtk_event_box_new();
	gtk_widget_set_size_request (widget, size, size);
	gtk_widget_show (widget);
	
	attach(row);
	
	/* Tworzy grupę wskazówek, jeśli jeszcze nie istneje,
	nie trzeba dbać o usuwanie grupy, ponieważ ze zniszczeniem
	ostatniej kontrolki która korzysta z grupy, sama grupa
	ulegnie zniszczeniu */
		
	if (tooltips == NULL)
		tooltips = gtk_tooltips_new ();
}

/*! \brief Konstruktor

	Tworzy klocek
	\param row Klocek będzie przynależał do wiersza
*/
LWPiece::LWPiece (LWRow *row):
	backpixmap(NULL), forepixmap(NULL), symbol(NULL), image(NULL), inverted_grounds(FALSE)
{
	init (row);	
}

/*! \brief Konstruktor

	Tworzy klocek
	\param row Klocek będzie przynależał do wiersza
	\param pixmap Klocek będzie zawierał ikonę 
*/
LWPiece::LWPiece (LWRow *row, LWPixmap *pixmap): 
	backpixmap(NULL), forepixmap(NULL), symbol(NULL), image(NULL), inverted_grounds(FALSE)
{
	init (row);
	
	setBackgroundPixmap (pixmap);
}

/*! \brief Konstruktor kopiujący

	Kopiuje klocek
	\param piece Kopiowany klocek
	\param pixmap Klocek będzie zawierał ikonę 
*/
LWPiece::LWPiece (const LWPiece *piece, LWRow *row): 
	backpixmap(NULL), forepixmap(NULL), symbol(NULL), image(NULL), inverted_grounds(FALSE)
{
	init (row);
	
	copy ((LWPiece *) piece);
}

/*! \brief Destruktor

	Niszczy klocek
*/
LWPiece::~LWPiece ()
{
	LWWizard *wizard = row->getBoard()->getWizard();
		
	if (wizard != NULL)
		if (wizard->getPiece() == this) {
			guint x = row->getPieceIndex(this);
			g_assert (x > 0);
			wizard->setPiece (row->getPieceNth (x-1));
		}
		
	clear ();
		
	gtk_widget_destroy (widget);
}

/*! \brief Odtwarza klocek z węzła XML

	Odtwarza wygląd klocka na podstawie węzła drzewa XML
	\param node Węzeł na podstawie zostanie dokonana rekonstrukcja
*/
void LWPiece::restoreFromXML (xmlNode *node)
{
	g_return_if_fail (LWEnvironment::getPixmapSet() != NULL);
	g_return_if_fail (!xmlStrcasecmp (node->name, (xmlChar *) "Piece"));
	
	clear();
	
	guint id = 0;
	
	gchar *name = (gchar *) xmlGetProp (node, (xmlChar *) "pixmap");

	if (name != NULL) {
		setBackgroundPixmap (LWEnvironment::getPixmapSet()->getPixmap (name));

#if BACKWARD_COMPATIBILITY
	/* Kompatybilność wsteczna - Backward compatibilty with ver 1.0.x*/
		if (name[1] == '\0')
			if ((name[0] >= 'A') && (name[0] <= 'Z'))
				id = name[0] - 'A' + 1;
#endif
		
		xmlFree (name);
	}

	gchar *symbolname = (gchar *) xmlGetProp (node, (xmlChar *) "symbol");

#if BACKWARD_COMPATIBILITY
	
	/* Kompatybilność wsteczna - Backward compatibilty with ver 1.0.x*/
	if (symbolname == NULL)
		symbolname = (gchar *) xmlGetProp (node, (xmlChar *) "cmd");
	else 
		id = 0;
#endif
	
	if (symbolname != NULL) {
		gchar *idstr = (gchar *) xmlGetProp (node, (xmlChar *) "id");

		if (idstr != NULL) {
			id = atoi (idstr);
			xmlFree (idstr);
		}
		
		if (id > 0)
			setSymbolWithId (symbolname, id);	
		else
			setSymbol (symbolname);
			
		xmlFree (symbolname);
	}
	
#if BACKWARD_COMPATIBILITY
	/* Kompatybilność wsteczna - Backward compatibilty with ver 1.0.x*/
	else if (id > 0)
			setSymbolWithId ("variable", id);
#endif
	
}

/*! \brief Zapisanie informacji o klocku w postaci węzła

	Zapisuje wygląd klocka w węźle drzewa XML. Węzeł będzie dzieckiem 
	rodzica który zostaje przekazany do metody.
	\param parent_node Węzeł rodzica
*/
void LWPiece::storeToXML (xmlNode *parent_node)
{
	g_return_if_fail (!xmlStrcasecmp (parent_node->name, (xmlChar *) "Row"));
	
	xmlNode *node = xmlNewChild(parent_node, NULL, (xmlChar *) "Piece", NULL);

	if (backpixmap != NULL)
		xmlNewProp (node, (xmlChar *) "pixmap", (xmlChar *) backpixmap->getName());

	if (symbol != NULL) {
		xmlNewProp (node, (xmlChar *) "symbol", (xmlChar *) symbol->getName());
		
		if (0 < symbol->getId()) {
			gchar *s = g_strdup_printf ("%u", symbol->getId());
			xmlNewProp (node, (xmlChar *) "id", (xmlChar *) s);
			g_free (s);
		}
	}
			
}

/*! \brief Pobranie wiersza, do którego należy klocek

	Pobiera wiersz, do którego należy klocek
	\return Pobrany wiersz
*/
LWRow *LWPiece::getRow()
{
	return row;
}

/*! \brief Pobranie symbolu

	Pobiera symbol skojarzony z klockiem
	\return Pobrany symbol
*/
LWSymbol *LWPiece::getSymbol ()
{
	return symbol;
}

/*! \brief Ustawienie symbolu

	Ustawia symbol, który będzie skojarzone z klockiem
	\param cmd Polecenie
*/
void LWPiece::setSymbol (const gchar *symbolname)
{
	g_return_if_fail (symbolname != NULL);

	if (symbol != NULL && TRUE == symbol->canClone())
		delete symbol;
		
	symbol = LWSymbol::factory (symbolname);

	g_return_if_fail (symbol != NULL);
		
	symbol->onAttach(this);
	
	gtk_tooltips_set_tip (tooltips, widget, 
			symbol->getHint(), symbol->getHint());
}

/*! \brief Ustawienie symbolu

	Ustawia symbol, który będzie skojarzone z klockiem.
	Funkcja uwzględnia identyfikator symbolu.
	
	\param cmd Polecenie
*/
void LWPiece::setSymbolWithId (const gchar *symbolname, guint id)
{
	g_return_if_fail (symbolname != NULL);

	if (symbol != NULL && TRUE == symbol->canClone())
		delete symbol;
	
	symbol = LWSymbol::factoryId (symbolname, id);

	g_return_if_fail (symbol != NULL);
	
	symbol->onAttach(this);
	
	gtk_tooltips_set_tip (tooltips, widget, 
			symbol->getHint(), symbol->getHint());
}

/*! \brief Pobranie ikony pod klockiem

	Pobiera ikonę, która znajduje się pod klockiem
	\return Pobrana ikona
*/
LWPixmap *LWPiece::getBackgroundPixmap ()
{
	return backpixmap;
}

/*! \brief Ustawia ikonę na klocku

	Ustawia ikonę, która znajdzie się na klocku
	\param pixmap Ikona do ustawienia
*/
void LWPiece::setForegroundPixmap (LWPixmap *pixmap)
{
   	forepixmap = pixmap;
	updateImage ();
}

/*! \brief Ustawia ikonę pod klockiem

	Ustawia ikonę, która znajdzie się pod klockiem
	\param pixmap Ikona do ustawienia
*/
void LWPiece::setBackgroundPixmap (LWPixmap *pixmap)
{
   	backpixmap = pixmap;
	updateImage();
}

/*! \brief Aktalizacja obrazu zawartego na klocku

	Aktalizuje obraz zawarty na klocku
*/
void LWPiece::updateImage ()
{
	guint size = row->getBoard()->getPieceSize();
	gint w, h;

	gtk_widget_get_size_request (widget, &w, &h);
	
	if ((gint) size != w || (gint) size != h)
		gtk_widget_set_size_request (widget, size, size);

	if ((forepixmap == NULL) && (backpixmap == NULL)) {
		clear ();
		return;
	}
		
	if (image == NULL) {
		image = gtk_image_new ();
		gtk_container_add (GTK_CONTAINER (widget), image);
		gtk_widget_show (image);
	}
	
	if (inverted_grounds == FALSE)
		composite_image (image, forepixmap, backpixmap, size);
	else
		composite_image (image, backpixmap, forepixmap, size);
	
	if (LW_TYPE_WORLD == row->getBoard()->getType())
		if (ifDNDWillMoveWizard()) {
			gtk_drag_source_set (widget, GDK_BUTTON1_MASK,
				target_table, n_targets, 
	    		GDK_ACTION_MOVE);
			gtk_drag_source_set_icon_pixbuf (widget, forepixmap->getPixbuf());
			return;
		} 
		else
		{
			gint source_actions = GDK_ACTION_COPY | GDK_ACTION_MOVE;
				
			gtk_drag_source_set (widget, GDK_BUTTON1_MASK,
				target_table, n_targets, 
	    	(GdkDragAction) source_actions);
		}
	
	if (backpixmap != NULL)
		if (LW_TYPE_PLAYGROUND != row->getBoard()->getType())
			gtk_drag_source_set_icon_pixbuf (widget, backpixmap->getPixbuf());
}

/*! \brief Włącza odwrócenie warstw

	Włącza lub wyłącza odwrócone interpretowanie warstw. To co
	było na klocku znajdzie się pod nim i vice versa
	\param inverted Jeśli prawda włączone zostanie odwrotne
	interpretowane warstw, jeśli fałsz wyłączone
*/
void LWPiece::enableInvertedGrounds (gboolean inverted)
{
	inverted_grounds = inverted;
	
	if (backpixmap != NULL && forepixmap != NULL)
		updateImage();
}

/*! \brief Wyczyszczenie zawartości klocka

	Czyści zawartość klocka
*/
void LWPiece::clear ()
{
	forepixmap = NULL;
	backpixmap = NULL;

	if (symbol != NULL)
		if (TRUE == symbol->canClone()) {
			delete symbol;
			symbol = NULL;
		}
	
	if (image != NULL) {
		gtk_widget_destroy (image);
		image = NULL;
	}

	if (LW_TYPE_PLAYGROUND != row->getBoard()->getType() &&
		FALSE == row->isPieceDummy (this))
			gtk_drag_source_set_icon_stock (widget, GTK_STOCK_DND);
}

/*! \brief Dostowuje zawartość klocka

	Dostosowuje zawartość klockado wygladu klocka zawartego w piece
	\param piece Klocek na podstawie którego nastąpi dostosowanie wyglądu
*/
void LWPiece::copy (LWPiece *piece)
{
	g_return_if_fail (piece != NULL);
	
	setForegroundPixmap(piece->forepixmap);
	setBackgroundPixmap(piece->backpixmap);
		
	if (piece->symbol != NULL) {
		
		if (symbol != NULL)
			if (TRUE == symbol->canClone())
				delete symbol;
		
		symbol = (TRUE == piece->symbol->canClone()) ? 
			piece->symbol->clone() : piece->symbol;
		
		g_return_if_fail (symbol != NULL);
		
		symbol->onAttach (this);
	
		gtk_tooltips_set_tip (tooltips, widget, 
								symbol->getHint(), symbol->getHint());
	} else
	piece->symbol = NULL;
}	

/*! \brief Kojarzenie klocka z wierszem

	Kojarzy klocek z wierszem, inicjacja drag'n'drop (metoda prywatna)
	\param row Wiersz z którym klocek zostanie skojarzony
*/
void LWPiece::attach (LWRow *a_row)
{
gint source_actions=0;
gint dest_actions=0;
	
	g_return_if_fail (a_row != NULL);
	g_return_if_fail (a_row->getBoard() != NULL);
	g_return_if_fail (symbol == NULL);
	
	switch (a_row->getBoard()->getType ()) {
		case LW_TYPE_ICONS: 
			source_actions = GDK_ACTION_COPY;
			dest_actions = 0;
			break;
		
		case LW_TYPE_COMMANDS:
			source_actions = GDK_ACTION_COPY;
			dest_actions = 0; 
			break;
		
		case LW_TYPE_WORLD:
			source_actions = GDK_ACTION_COPY | GDK_ACTION_MOVE;
			dest_actions = GDK_ACTION_COPY | GDK_ACTION_MOVE;
			break;
		
		case LW_TYPE_PROGRAM:
			source_actions = GDK_ACTION_MOVE | GDK_ACTION_LINK;
			dest_actions = GDK_ACTION_COPY | GDK_ACTION_MOVE | GDK_ACTION_LINK; 
		break;
		
		case LW_TYPE_PLAYGROUND: 
			source_actions = 0;
			dest_actions = 0;
			break;
	}

    if (dest_actions != 0) {
 		gtk_drag_dest_set (widget,
		    GTK_DEST_DEFAULT_ALL,
		   	target_table, n_targets,
		   	(GdkDragAction) dest_actions);

    	g_signal_connect ( G_OBJECT(widget), "drag_data_received",
	 		G_CALLBACK(target_drag_data_received), (gpointer) this);
	}

	if (source_actions != 0) {
		gtk_drag_source_set (widget, GDK_BUTTON1_MASK,
			target_table, n_targets, 
	    	(GdkDragAction) source_actions);

		g_signal_connect (G_OBJECT (widget), "drag_data_get",
			G_CALLBACK(source_drag_data_get), (gpointer) this);
	
	}
	
	row = a_row;
}

gboolean LWPiece::ifDNDWillMoveWizard()
{
	return ((inverted_grounds == FALSE || backpixmap == NULL) && forepixmap != NULL) ? TRUE : FALSE;
}

/*! \brief Pobranie kontrolki

	Pobiera kontrolkę
	\return Pobrana kontrolka
*/
GtkWidget *LWPiece::getWidget ()
{
	return widget;
}
