/*
 * dlock.c - SCO lslk lock functions
 *
 * Vic Abell
 * Purdue University Computing Center
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell.
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */
#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dlock.c,v 1.10 99/11/10 15:01:04 abe Exp $";
#endif


#include "lslk.h"


/*
 * Local structure definitions
 */

struct pidcmd {				/* PID<->command_name cache structure */

	unsigned long pid;		/* PID */
	char *cmd;			/* command name */
	struct pidcmd *next;		/* next entry in same hash bucket */
};

#if	defined(HAS_NFS)
struct l_NFS_lock {			/* local NFS lock information */

	struct inode *ip;		/* inode's kernel address */
	pid_t ppid;			/* pseudo PID */
	unsigned long rpid;		/* remote PID on NFS client */
	char *hn;			/* NFS client host name */
	struct in_addr *iap;		/* network address */
	struct l_NFS_lock *forw;	/* forward link */
};
#endif	/* defined(HAS_NFS) */


/*
 * Local function prototypes
 */

#if	defined(HAS_NFS)
_PROTOTYPE(static struct l_NFS_lock *alloc_l_NFS_lock,(char *hn));
_PROTOTYPE(static int lkup_NFS_lock,(struct inode *ik));
_PROTOTYPE(static int open_kmem,(int nx));
_PROTOTYPE(static char *rd_clnt_hn,(netobj *c));
#endif	/* defined(HAS_NFS) */

#if	!defined(N_UNIX)
_PROTOTYPE(static int is_boot,(char *p));
#endif	/* !defined(N_UNIX) */

_PROTOTYPE(static void savelock,(struct inode *i, struct inode *ik, struct filock *f));


/*
 * Local macro definitions
 */

#define	hashpid(pid)	((((int)pid * 31415) >> 3) & PidMask)


/*
 * Local static variables.
 */

#if	defined(HAS_NFS)
static struct l_NFS_lock *Li;		/* local NFS lock information */
static int Lis = 0;			/* status of Li: 0 = not loaded
					 *		 1 = loaded or not
					 *		     available */
#endif	/* defined(HAS_NFS) */

static struct pidcmd **Pidhash = (struct pidcmd **)NULL;
					/* PID to command hash buckets */
static int PidMask = 0;			/* mask for hashpid macro */
static struct var Var;			/* system configuration information */


#if	defined(HAS_NFS)
/*
 * alloc_l_NFS_lock() - allocate l_NFS_lock structure
 */

struct l_NFS_lock *
alloc_l_NFS_lock(hn)
	char *hn;			/* client host name of allocation */
{
	struct l_NFS_lock *lnp;

	if ((lnp = (struct l_NFS_lock *)malloc(sizeof(struct l_NFS_lock))))
	    return(lnp);
	(void) fprintf(stderr,
	    "%s: can't allocate NFS lock struct for: %s\n", Pn, hn);
	Exit(1);
}
#endif	/* defined(HAS_NFS) */


/*
 * gather_lock_info() -- gather lock information
 */

void
gather_lock_info()
{
	KA_T ff, fp;
	struct filock fl;
	struct hinode hi, *hip;
	int hx;
	struct inode i, *ip;
/*
 * Loop through the inode hash buckets.
 */
	for (hx = 0, hip = (struct hinode *)Nl[X_HINODE].n_value;
	     hx < Var.v_hinode;
	     hx++, hip++)
	{
	    if (kread((KA_T)hip, (char *)&hi, sizeof(hi))) {
		if (Owarn)
		    (void) fprintf(stderr, "%s: can't read hinode from %#x\n",
			Pn, hip);
		return;
	    }
	/*
	 * Loop through the inodes of a hash bucket.
	 */
	    for (ip = hi.i_forw; ip && ip != (struct inode *)hip; ip = i.i_forw)
	    {
		if (kread((KA_T)ip, (char *)&i, sizeof(i))) {
		    if (Owarn)
			(void) fprintf(stderr,
			    "%s: premature exit from inode chain\n", Pn);
		    return;
		}
		if (!i.i_count || !i.i_filocks)
		    continue;
	    /*
	     * If the inode has a non-NULL i_filocks pointer and a non-zero
	     * reference count, read the associated filock structure list,
	     * and process its entries.
	     */
		ff = fp = (KA_T)i.i_filocks;
		do {
		    if (kread(fp, (char *)&fl, sizeof(fl)))
			break;
		    (void) savelock(&i, ip, &fl);
		    if (is_lock_sel()) {
			NLockU++;
			Lp = (struct llock_info *)NULL;
		    }
		} while ((fp = (KA_T)fl.next) && fp != ff);
	    }
	}
}


/*
 * get_cmdnm() -- get command name
 */

char *
get_cmdnm(lp)
	struct llock_info *lp;		/* local lock structure */
{
	char *cmd;
	int i, ph;
	KA_T ka;
	static KA_T kp = (KA_T)0;
	MALLOC_S len;
	static struct proc *p = (struct proc *)NULL;
	struct pidcmd *pc;
	static int px = 0;
	static struct user *u = (struct user *)NULL;
/*
 * If no PID<->command_name has table has been defined, define it.
 */
	if (PidMask == 0) {
	    for (PidMask = 128; PidMask < (2 * Var.v_proc); )
		PidMask = PidMask << 1;
	    if ((Pidhash = (struct pidcmd **)calloc((size_t)PidMask,
			     sizeof(struct pidcmd)))
	    == (struct pidcmd **)NULL)
	    {
		(void) fprintf(stderr,
		    "%s: no space for PID<->command_name hash table\n", Pn);
		Exit(1);
	    }
	    PidMask--;
	    ph = hashpid(lp->pid);
	} else {

	/*
	 * See if the PID's command name is already hashed.
	 */
	    ph = hashpid(lp->pid);
	    for (pc = Pidhash[ph]; pc; pc = pc->next) {
		if (pc->pid == lp->pid && pc->cmd)
		    return(pc->cmd);
	    }
	}

/*
 * If this is the first time, set up to scan the proc table.
 */
	if (kp == (KA_T)0) {
	
	/*
	 * Set proc table base address.
	 */
	    kp = (KA_T)Nl[X_PROC].n_value;
	/*
	 * Allocate proc structure buffer address.
	 */
	    if ((p = (struct proc *)malloc(sizeof(struct proc)))
	    == (struct proc *)NULL)
	    {
		(void) fprintf(stderr, "%s: no space for proc struct buffer\n",
		    Pn);
		Exit(1);
	    }
	/*
	 * Allocate user structure buffer address.
	 */
	    if ((u = (struct user *)malloc(MAXUSIZE * NBPC))
	    == (struct user *) NULL)
	    {
		(void) fprintf(stderr, "%s: no space for user struct buffer\n",
		    Pn);
		Exit(1);
	    }
	}
/*
 * Resume reading the kernel proc structure table.  If the process isn't
 * located, reread the proc structure table from its beginning.
 */
	for (i = px ? 0 : 1; i < 2; i++) {
	    for (; px < Var.v_proc; px++) {
		ka = kp + (px * sizeof(struct proc));
		if (kread(ka, (char *)p, sizeof(struct proc)))
		    continue;
		if ((unsigned long)p->p_pid != lp->pid)
		    continue;
		if (sysi86(RDUBLK, p->p_pid, (char *)u, MAXUSIZE * NBPC) == -1)
		    continue;
	    /*
	     * The target process has been located along with its user area.
	     * Allocate a PID<->command_name structure, fill it, and link it
	     * to its hash bucket.
	     */
		if ((len = strlen(u->u_comm)) < 1)
		    continue;
		if ((cmd = (char *)malloc(len + 1)) == (char *)NULL) {
		    (void) fprintf(stderr,
			"%s: PID %ld; can't allocate command space: %s\n",
			Pn, (unsigned long)lp->pid, u->u_comm);
		    Exit(1);
		}
		(void) strcpy(cmd, u->u_comm);
		if ((pc = (struct pidcmd *)malloc(sizeof(struct pidcmd)))
		    == (struct pidcmd *)NULL)
		{
		    (void) fprintf(stderr,
			"%s: PID %ld; can't allocate hash space: %s\n",
			Pn, lp->pid, u->u_comm);
		    Exit(1);
		}
		pc->pid = lp->pid;
		pc->cmd = cmd;
		pc->next = Pidhash[ph];
		Pidhash[ph] = pc;
		return(cmd);
	    }
	    px = 0;
	}
	return("(unknown)");
}


/*
 * get_kernel_access() -- get access to kernel information
 */

void
get_kernel_access()
{
	int err, i;
	MALLOC_S len;
/*
 * See if the name list file is readable.
 */
	if (Nmlst && !is_readable(Nmlst, 1))
		Exit(1);
/*
 * Access kernel symbols.
 */

#if	defined(N_UNIX)
	if (nlist(Nmlst ? Nmlst : N_UNIX, Nl) < 0)
#else	/* !defined(N_UNIX) */
	if (!get_nlist_path(0))
#endif	/* defined(N_UNIX) */

	{
	    (void) fprintf(stderr, "%s: can't read kernel name list\n", Pn);
	    Exit(1);
	}
	for (err = i = 0; Nl[i].n_name[0]; i++) {
	    if (i == X_LCKSVR)
		continue;
	    if (!Nl[i].n_value) {
		(void) fprintf(stderr, "%s: no nlist address for %s\n",
		    Pn, Nl[i].n_name);
		err++;
	    }
	}
	if (err)
	    Exit(1);
/*
 * Open kernel memory access.
 */
	(void) open_kmem(0);
	if ((Kd = open(KMEM, O_RDONLY, 0)) < 0) {
	    (void) fprintf(stderr, "%s: can't open %s: %s\n",
		Pn, KMEM, strerror(errno));
	    Exit(1);
	}
/*
 * Drop setgid permission.
 */
	(void) dropgid();
/*
 * Read system configuration information -- the kernel's var struct.
 */
	if (kread((KA_T)Nl[X_VAR].n_value, (char *)&Var, sizeof(Var))) {
	    (void) fprintf(stderr,
		"%s: can't read system configuration info\n", Pn);
	    Exit(1);
	}

#if	OSRV>=40
/*
 * Get extended device table parameters.  These are needed by the
 * kernel versions of the major() and minor() device number macroes.
 */
	if (Nl[X_NXD].n_value == (long)NULL
	||  kread((KA_T)Nl[X_NXD].n_value,(char *)&nxdevmaps,sizeof(nxdevmaps))
	||  nxdevmaps < 0)
	{
	    (void) fprintf(stderr,
		"%s: bad extended device table size (%d) at 0x%#x.\n",
		Pn, nxdevmaps, Nl[X_NXD].n_value);
	    Exit(1);
	}
	len = (MALLOC_S)((nxdevmaps + 1) * sizeof(struct XDEVMAP));
	if ((Xdevmap = (struct XDEVMAP *)malloc(len)) == (struct XDEVMAP *)NULL)
	{
	    (void) fprintf(stderr, "%s: no space for %d byte xdevmap table\n",
		Pn, len);
	    Exit(1);
	}
	if (Nl[X_XDEV].n_value == (long)NULL
	||  kread((KA_T)Nl[X_XDEV].n_value, (char *)Xdevmap, len))
	{
	    (void) fprintf(stderr,
		"%s: can't read %d byte xdevmap table at 0x%#x.\n",
		Pn, len, Nl[X_XDEV].n_value);
	    Exit(1);
	}
#endif	/* OSRV>=40 */

}


#if	!defined(N_UNIX)
/*
 * get_nlist_path() - get kernel nlist() path
 *
 * As a side effect on a successful return (non-NULL character pointer), the
 * boot path's name list will have been loaded into Nl[].
 */

char *
get_nlist_path(ap)
	int ap;				/* on success, return an allocated path
					 * string pointer if 1; return a
					 * constant character pointer if 0;
					 * return NULL if failure */
{
	FILE *bf;
	char *bfp, b1[MAXPATHLEN+1], b2[MAXPATHLEN+1], *pp, *tp;
	struct dirent *de;
	char *dir[] = { "/", "/stand/", NULL };
	DIR *dp;
	int i;
	MALLOC_S len;
/*
 * If a kernel name list file was specified, use it.
 */
	if (Nmlst) {
	    if (is_boot(Nmlst))
		return(Nmlst);
	    return((char *)NULL);
	}
/*
 * If it's possible to open /etc/ps/booted system, search it for a preferred
 * boot path, defined by the value of a line that begins with "KERNEL=".
 */
	bfp = pp = (char *)NULL;
	if ((bf = fopen("/etc/ps/booted.system", "r"))) {
	    len = strlen("KERNEL=");
	    while (fgets(b1, sizeof(b1), bf)) {
		if (strncmp(b1, "KERNEL=", len) != 0)
		    continue;
		if ((tp = strrchr(&b1[len], '\n'))) {
		    *tp = '\0';
		    if (b1[len]) {
			bfp = &b1[len];
			if (is_boot(bfp)) {
			    pp = bfp;
			    (void) fclose(bf);
			    goto get_nlist_return_path;
			}
			break;
		    }
		}
	    }
	    (void) fclose(bf);
	}
/*
 * Look for possible unix* boot paths.
 *
 * Use the preferred boot path from /etc/ps/booted.system as the first
 * possible boot path.
 */
	for (i = 0; dir[i]; i++) {
	    if (!(dp = opendir(dir[i])))
		continue;
	    while ((de = readdir(dp))) {

	    /*
	     * Use the next entry that begins with "unix".
	     */
		if (strncmp("unix", de->d_name, 4) != 0)
		    continue;
	    /*
	     * Construct a temporary copy of the path name,
	     * If it matches the preferred boot name, skip it.
	     */
		len = strlen(dir[i]) + strlen(de->d_name) + 1;
		if (len >= sizeof(b2))
		    continue;
		(void) sprintf(b2, "%s%s", dir[i], de->d_name);
		if (bfp && strcmp(b2, bfp) == 0)
		    continue;
	    /*
	     * See if it's the bppted kernel.
	     */
		if (is_boot(b2)) {
		    (void) closedir(dp);
		    pp = b2;

get_nlist_return_path:

		/*
		 * A boot path has been located.  As requested return a
		 * malloc'd pointer to it.
		 */
		    if (!ap)
			return("");
		    len = (MALLOC_S)(strlen(pp) + 1);
		    if (!(tp = (char *)malloc(len))) {
			(void) fprintf(stderr,
			    "%s: can't allocate %d bytes for: %s\n",
			    Pn, len , pp);
			Exit(1);
		    }
		    (void) strcpy(tp, pp);
		    return(tp);
		}
	    }
	    if (dp)
		(void) closedir(dp);
	}
	return((char *)NULL);
}
#endif	/* !defined(N_UNIX) */


/*
 * initialize() -- initialize
 */

void
initialize()
{
	(void) get_kernel_access();
}


#if	!defined(N_UNIX)
/*
 * is_boot() - does the specified path lead to a booted kernel
 */

is_boot(p)
	char *p;			/* specified path */
{
	int i;
	KA_T ka;
	union {
		struct scoutsname s;
		unsigned char sc[sizeof(struct scoutsname)];
	} s1, s2;
/*
 * Get the scoutsname structure via __scoinfo() to use as a reference against
 * the one obtained via kread()'ing from the nlist(<possible_kernel>) address.
 *  If __scoinfo() fails, return the default boot path.
 */
	if (__scoinfo(&s1.s, sizeof(s1.s)) < 0)
	    return 0;
/*
 * Get the name list for this boot path.  Using the scoutsname address, read
 * the scoutsname structure and compare it to the _s_scoinfo() one.  If the
 * two match, this is the boot path.
 */
	for (i = 0; Nl[i].n_name && Nl[i].n_name[0]; i++) {
	    Nl[i]. n_value = Nl[i]. n_scnum = Nl[i].n_type = Nl[i].n_sclass
			   = Nl[i].n_numaux = 0;
	}
	if (nlist(p, Nl) < 0)
	    return(0);
	if (!(ka = (KA_T)Nl[X_SCOUTS].n_value))
	    return(0);
	if (Kd < 0) {
	    if (open_kmem(1))
		return(0);
	}
	if (kread(ka, (char *)&s2.s, sizeof(s2.s)))
	    return(0);
	for (i = 0; i < sizeof(struct scoutsname); i++) {
		if (s1.sc[i] != s2.sc[i])
		    return(0);
	}
	return(1);
}
#endif	/* !defined(N_UNIX) */


/*
 * kread() -- read kernel memory
 */

int
kread(addr, buf, len)
	KA_T addr;			/* kernel address */
	char *buf;			/* local receiving buffer address */
	int len	;			/* length to read */
{
	int br;

	if (lseek(Kd, (off_t)addr, SEEK_SET) == (off_t)-1L)
		return(1);
	if ((br = read(Kd, buf, len)) < 0)
		return(1);
	return((br == len) ? 0 : 1);
}


#if	defined(HAS_NFS)
/*
 * lkup_NFS_lock() - look up NFS lock information
 */

static int
lkup_NFS_lock(ik)
	struct inode *ik;		/* inode's kernel address */
{
	int err;
	char *hn;
	struct in_addr *iap;
	KA_T klb, klp;
	struct lhaddr *lh;
	struct l_NFS_lock *lnp;

# if	OSRV<500
	struct NFS_lock_info kl;
# else	/* OSRV>=500 */
	KA_T krb, krp;
	struct reclock rl;
	struct srvlock sl;
# endif	/* OSRV<500 */

	if (!Lis) {

	/*
	 * If the kernel's NFS lock information hasn't been loaded, do it now.
	 *
	 * If there's no NFS lock information pointer or the pointer can't be
	 * read, record that the NFS lock information is unavailable.  Issue
	 * a warning on a read failure.  Return a non-found response.
	 */
	    if (!Nl[X_LCKSVR].n_value
	    ||  kread((KA_T)Nl[X_LCKSVR].n_value, (char *)&klb, sizeof(klb)))
	    {
		(void) fprintf(stderr, "%s: can't read %s from: %#x\n",
		    Pn, Nl[X_LCKSVR].n_name, Nl[X_LCKSVR].n_value);
		Li = (struct l_NFS_lock *)NULL;
		return(0);
	    }
	    Lis = 1;

# if	OSRV<500
	    if ((klp = klb)) {

	    /*
	     * Read kernel lock structures and enter their values in
	     * l_NFS_lock structures.
	     */
		do {
		    if (kread(klp, (char *)&kl, sizeof(kl)))
			break;
		    if ((hn = rd_clnt_hn(&kl.ci)) == (char *)NULL)
			continue;
		    lnp = alloc_l_NFS_lock(hn);
		    lnp->ip = kl.ip;
		    lnp->ppid = kl.l_pid;
		    lnp->rpid = kl.f_pid;
		    lnp->hn = hn;
		    lnp->iap = (struct in_addr *)NULL;
		    lnp->forw = Li;
		    Li = lnp;
		} while ((klp = (KA_T)kl.forw) && klp != klb); }
# else	/* OSRV>=500 */
	    if ((klp = klb)) {

	    /*
	     * Read server lock structures.
	     */
		do {
		    if (kread(klp, (char *)&sl, sizeof(sl)))
			break;
		    if (!(krb = (KA_T)sl.srv_rlp))
			continue;
		    krp = krb;
		/*
		 * Read record lock structures and enter their values in
		 * l_NFS_lock structures.
		 */
		    do {
			if (kread(krp, (char *)&rl, sizeof(rl)))
			    continue;
			if ((hn = rd_clnt_hn(&rl.rl_clnt)) == (char *)NULL)
			    continue;
			lnp = alloc_l_NFS_lock(hn);
			lnp->ip = rl.rl.ip;
			lnp->ppid = rl.rl_lock.l_pid;
			lnp->rpid = (unsigned long)rl.rl_lock.l_sysid;
			lnp->hn = hn;
			lnp->iap = (struct in_addr *)NULL;
			lnp->forw = Li;
			Li = lnp;
		    } while ((krp = (KA_T)rl.lck_nxt) && krb != krb);
		} while ((klp = (KA_T)sl.srv_nxt) && klp != klb);
	    }
# endif	/* OSRV<500 */

	    else
		Li = (struct l_NFS_lock *)NULL;
	}
/*
 * Search the NFS lock information for a match on inode address and
 * pseudo PID.
 */
	for (lnp = Li; lnp; lnp = lnp->forw) {
	    if (ik == lnp->ip && Lp->pid == lnp->ppid) {

	    /*
	     * Update PID and host name information from the local NFS lock
	     * information structure.
	     */
		Lp->pid = lnp->rpid;
		Lp->hn = lnp->hn;
	    /*
	     * Add a network address to the l_NFS_lock structure, as
	     * required.  Propagate it to the local lock info structure.
	     */
		if (lnp->iap) {
		    Lp->iap = lnp->iap;
		    return(1);
		} else if (!lnp->hn || !Oconv)
		    return(1);
		lh = get_haddr(0, lnp->hn, (unsigned long)0, &err);
		if (!err && lh->na) {
		    iap = alloc_in_addr();
		    iap->s_addr = lh->na;
		    Lp->iap = lnp->iap = iap;
		}
		return(1);
	    }
	}
	return(0);
}
#endif	/* defined(HAS_NFS) */


/*
 * open_kmem() - open kernel memory access
 */

static int
open_kmem(nx)
	int nx;				/* no Exit(1) if 1 */
{
	if (Kd >= 0)
	    return(0);
/*
 * Open kernel memory access.
 */
	if ((Kd = open(KMEM, O_RDONLY, 0)) < 0) {
	    if (nx)
		return(1);
	    (void) fprintf(stderr, "%s: can't open %s: %s\n", Pn,
		KMEM, strerror(errno));
	    Exit(1);
	}
	return(0);
}


/*
 * print_dev() -- print device number
 */

char *
print_dev(lp)
	struct llock_info *lp;		/* local lock structure */
{
	static char buf[128];

	(void) sprintf(buf, "%d,%d", major(lp->dev), minor(lp->dev));
	return(buf);
}


#if	defined(HAS_NFS)
/*
 * rd_clnt_hn() - read client host name
 */

static char *
rd_clnt_hn(c)
	netobj *c;			/* client name net object */
{
	char *hn;

	if (c->n_len == 0 || !c->n_bytes)
	    return((char *)NULL);
/*
 * Allocate space for client host name and read it.
 */
	if ((hn = (char *)malloc((MALLOC_S)(c->n_len + 1))) == (char *)NULL) {
	    (void) fprintf(stderr,
		"%s: can't allocate %d bytes for NFS client name\n",
	        Pn, c->n_len + 1);
	    Exit(1);
	}
	if (kread((KA_T)c->n_bytes, hn, (int)c->n_len))
	    return((char *)NULL);
	hn[c->n_len] = '\0';
	return(hn);
}
#endif	/* defined(HAS_NFS) */


/*
 * savelock() - save lock information
 */

static void
savelock(i, ik, f)
	struct inode *i;		/* local inode pointer */
	struct inode *ik;		/* inode's kernel address  */
	struct filock *f;		/* filock pointer */
{

/*
 * Allocate a local lock structure.
 */
	(void) alloc_llock();
/*
 * Save: inode number; device; process ID; and size.
 */
	Lp->inum = i->i_number;
	Lp->dev = i->i_dev;
	Lp->pid = (unsigned long)f->set.l_pid;
	Lp->sz = (u_long)i->i_size;
	Lp->szs = 1;
/*
 * Save the lock description: mandatory status; type; start; length;
 * and whence.
 *
 * Release 3.0 mandatory locks are indicated by IXLOCKED in the inode; thus
 * it isn't possible to know which of multiple locks is mandatory and which is
 * advisory.  Lslk shows all locks whose inode has IXLOCKED set as mandatory
 * locks.
 *
 * Release 5.0 mandatory locks are indicated in the individual filock
 * structures with a flag value of F_XOUT.  Thus lslk can show that some locks
 * for the same inode are mandatory and others are advisory.
 */

	Lp->mand = 

#if	OSRV<500
		    (i->i_flag & IXLOCKED)
#else	/* OSRV>=500 */
		    (f->flags & F_XOUT)
#endif	/* OSRV<300 */

		    ? 1 : 0;

	if (f->set.l_type == F_WRLCK)
	    Lp->type = 2;
	else if (f->set.l_type == F_RDLCK)
	    Lp->type = 1;
	else
	    Lp->type = 0;
	Lp->whence = f->set.l_whence;
	Lp->start = f->set.l_start;
	Lp->len = f->set.l_len;
	Lp->ls = Lp->ss = Lp->ws = 1;
/*
 * Set remote status if l_pid contains a remote PID.
 */

#if	defined(HAS_NFS)
	if (f->set.l_pid >= MAXPID && lkup_NFS_lock(ik))
	    Lp->src = 1;
	else
#endif	/* defined(HAS_NFS) */

	    Lp->src = 0;
}
