/* ---------------------------------- global.cpp ---------------------------------------------------------------------------
 file containing all variables & functions used globaly

===============================================================================================================================
===============================================================================================================================
     This file is part of "luckyBackup" project
     Copyright 2008-2010, Loukas Avgeriou
     luckyBackup is distributed under the terms of the GNU General Public License
     luckyBackup is free software: you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation, either version 3 of the License, or
     (at your option) any later version.
 
     luckyBackup is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
 
     You should have received a copy of the GNU General Public License
     along with luckyBackup.  If not, see <http://www.gnu.org/licenses/>.

 project version	: Please see "main.cpp" for project version

 developer 		: luckyb 
 last modified 		: 14 Apr 2010
===============================================================================================================================
===============================================================================================================================
*/

#include "operationClass.h"
#include <iostream>
#include "global.h"
using namespace std;

// argumentsTest===================================================================================================================
// tests the arguments given when LB executed at command-line
bool argumentsTest(int ArgsNo, char **arg)
{
	console = FALSE;
	NoQuestions = FALSE;
	SkipCritical = FALSE;
	DryRun = FALSE;
	silentMode = FALSE;
	runImmediately = FALSE;
	
	if (ArgsNo == 1)		// if just luckybackup is given without argumets just run the gui
		return TRUE;

	int NoOfArgs = ArgsNo-1;	//the number of arguments given minus the command luckybackup
	string stdArgs[5] = { "-c",	"--no-questions",	"--skip-critical",	"--dry-run",	"--silent"};
	bool argCheck = FALSE;	// if a specific argument is ok this becomes true

	int count = 1;
	while (count < NoOfArgs)
	{
		if (arg[count] == stdArgs[0])
			{ console = TRUE; argCheck = TRUE; }
		if (arg[count] == stdArgs[1])
			{ NoQuestions = TRUE; console = TRUE; argCheck = TRUE; }
		if (arg[count] == stdArgs[2])
			{ SkipCritical = TRUE; argCheck = TRUE; }
		if (arg[count] == stdArgs[3])
			{ DryRun = TRUE; argCheck = TRUE; }
		if (arg[count] == stdArgs[4])
			{ silentMode = TRUE; argCheck = TRUE; }
		if (!argCheck)			// if the argument is unknown
			{ help(); return FALSE;	}	
		argCheck = FALSE;
		count++;
	}
	
	const char *profileNameChar = arg[NoOfArgs];	//const char* used cause argument in terminal is in utf-8 and this helps to convert it to QString
	QString profileNameArg = arg[NoOfArgs];
	if ( (profileNameArg == "--help") || (profileNameArg == "-h") )
		{ help(); return FALSE;	}
	
	//currentProfile = profileNameArg;	// currentProfile holds the full path+filename of the current profile
	currentProfile = QString::fromUtf8(profileNameChar);	// currentProfile holds the full path+filename of the current profile
	if (!profileNameArg.endsWith(".profile",Qt::CaseSensitive))
		currentProfile.append(".profile");
	if (!profileNameArg.startsWith("/"))
		currentProfile.prepend(profileDir);
	
	//current profile QFile
	profile.setFileName(currentProfile);
	
	//current profile's name (QString) - this just holds the profile name. No path & no extension
	profileName = currentProfile;
	profileName = profileName.right(profileName.size() - profileName.lastIndexOf("/") - 1);
	profileName.chop(8);
	
	runImmediately = TRUE;
	return TRUE;	// all arguments ok
}

// loadCurrentProfile===================================================================================================================
// loads an existing profile - non-gui call
bool loadCurrentProfile()
{
	QString currentProfileUTF8 = QString(currentProfile.toUtf8());
	
	cout << "\n============================================================================================\n";
	cout << "		Loading profile " << currentProfileUTF8.toStdString() << "\n";
	cout << "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n";
	
	QString source, dest;

	//read the file and fill all arrays
	int loadOK = loadProfile(currentProfileUTF8);	// try to load the currentProfile
	if (loadOK == 1)		// if it cannot open
	{
		cout << "	** Unable to open profile : " << currentProfileUTF8.toStdString() << " **\n";
		cout << "	" << profile.errorString().toStdString();
		cout << "\n\n";
		return FALSE;					//do nothing more
	}

	if (loadOK == 2)			// if it is not a valid profile
	{
		cout << "	** The profile you are trying to open is not a valid luckyBackup v."
		<< countStr.setNum(appVersion).toStdString() << " profile **\n\n";
		return FALSE;	//do nothing more
	}

	// if all went ok (profile loaded)
	currentOperation = 0;
	while (currentOperation < TotalOperations)
	{
		source = Operation[currentOperation]->GetSource();
		dest = Operation[currentOperation]->GetDestination();
		
		if ( (Operation[currentOperation] -> GetRemote()) && (Operation[currentOperation] -> GetRemoteSource()) )
		{
			if (Operation[currentOperation] -> GetRemoteModule())
				source.prepend(":");
			source.prepend((Operation[currentOperation] -> GetRemoteHost())+":");
			if (Operation[currentOperation] -> GetRemoteUser()!="")
				source.prepend((Operation[currentOperation] -> GetRemoteUser())+"@");
		}
	
		if ( (Operation[currentOperation] -> GetRemote()) && (Operation[currentOperation] -> GetRemoteDestination()) )
		{
			if (Operation[currentOperation] -> GetRemoteModule())
				dest.prepend(":");
			dest.prepend((Operation[currentOperation] -> GetRemoteHost())+":");
			if (Operation[currentOperation] -> GetRemoteUser()!="")
				dest.prepend((Operation[currentOperation] -> GetRemoteUser())+"@");
		}
		cout << "\n* operation name		: " << QString((Operation[currentOperation]->GetName()).toUtf8()).toStdString();
		cout << "\n* source			: " << QString(source.toUtf8()).toStdString();
		cout << "\n* destination			: " << QString(dest.toUtf8()).toStdString();

		if (Operation[currentOperation]->GetIncluded())
			cout << "\n* This task is included\n";
		else
			cout << "\n* This task is NOT included\n";	

		currentOperation++;
	}
	cout << "\n\n			** Profile loaded successfuly ... **\n\n";
	return TRUE;
}

// check_list===================================================================================================================
// checks the loaded operations list for errors (eg nothing is included)
// calls global function checkTaskList()
bool check_list()
{
	cout << "\n============================================================================================\n";
	cout << "				Task list check \n";
	cout << "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n";



	cout << "\n* Checking if the task list is empty...					done";
	cout << "\n* Checking if 2 connected tasks have been selected for execution...	done";
	cout << "\n* Checking if no task is included...					done";
	cout << "\n* Checking if 2 or more identical destination directories are declared\n  & 'Backup dir contents' is checked...					done\n";

	checkTaskList();

	if (ask)
	{
		cout << messageCLI.toStdString();;
		return FALSE;
	}
	else
	{
		cout << "\n\n			** Task list looks ok... **\n";
		return TRUE;
	}
	return TRUE;
}

// check_dirs===================================================================================================================
// checks all declared directories for errors by calling checkBackupDirs or checkSyncDirs
//This is called from the console
bool check_dirs()
{
	cout << "\n============================================================================================\n";
	cout << "				Directories check\n";
	cout << "~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n";

	checkDeclaredDirs(FALSE);

	if (ask == FALSE)	//if all the dirs are ok prepend  this lines to the dialog message
	{
		CheckedDataCLI.prepend("\n		(Have in mind that checks are not performed for remote data)\n\n"
		"~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n");
		CheckedDataCLI.prepend("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
		"		All data declared apear to be ok - You are ready to go !!"
		"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n");
	}
	else			//else prepend the following
	{
		CheckedDataCLI.prepend("			Errors have been found\n"
		"		Please have a good look at the following messages\n\n"
		"	WARNING means that the task is NOT going to be performed\n"
		"	CRITICAL means that the task is going to be performed normally\n\n"
		"	If a directory is empty or does not exist,\n"
		"	there is a possibility that you 've forgotten to mount a partition/drive\n	or have just mistyped a path !!\n"
		"\n	BEWARE if a destination is empty or non-existent\n"
		"	and it is not the first time you perform the specific task(s)\n\n"
		"	Also have in mind that checks are not performed for remote data\n"
		"~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n");

		CheckedDataCLI.prepend("\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n"
		"		Source & destination data check results"
		"\n~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~\n\n");
	}

	cout << CheckedDataCLI.toStdString();

	cout <<"\n\n";
	if (NothingToDo) //if there is nothing to Do anyway then just display a message
	{
		cout << "\n\n		** ..nothing to do !! **\n\n";
		return FALSE;
	}
	return TRUE;
}

// help===================================================================================================================
// displays command line usage instructions
void help()
{
	cout << "\n============================================================================================\n";
	cout << "\n * usage :		luckybackup [options] [filename]";
	cout << "\n -------------------------------------------------------------------------------------------\n";
	cout << " [options] can be :\n\n";
	cout << " --help 		: displays this help message\n\n";

	cout << " --skip-critical	: does not execute tasks that appear with a 'CRITICAL' warning message.\n";
	cout << " --dry-run		: executes luckybackup in dry-run (simulation) mode\n";
	cout << " --silent		: executes luckybackup in silent mode (just tray notification icon shown)\n";
	cout << "\n";
	cout << " -c 			: console mode. Use this, if there is no graphical envrironment available\n";
	cout << " --no-questions		: skips confirmation questions asked to user. Implies -c (console mode).\n\n";

	cout << " [filename] is the already created profile that is going to be executed\n";
	cout << " -------------------------------------------------------------------------------------------\n";
	cout << " * examples:\n";
	cout << " Execute luckybackup gui :\n";
	cout << " $ luckybackup\n\n";
	
	cout << " Execute luckybackup in silent mode for profile '~/.luckyBackup/profiles/BackupHome.profile' :\n";
	cout << " $ luckybackup --silent ~/.luckyBackup/profiles/BackupHome.profile\n\n";
	
	cout << " Execute luckybackup in console mode for profile '~/.luckyBackup/profiles/BackupHome.profile' :\n";
	cout << " $ luckybackup -c BackupHome.profile\n\n";
	
	cout << " Execute luckybackup in console mode for profile '~/.luckyBackup/profiles/BackupHome.profile' :\n";
	cout << " Do not ask any questions and skip all operations that apear CRITICAL after the checks\n";
	cout << " $ luckybackup --skip-critical --no-questions ~/.luckyBackup/profiles/BackupHome.profile";
	cout << "\n -------------------------------------------------------------------------------------------\n";
	cout << " see also luckybackup man page:\n";
	cout << " $ man luckybackup";
	cout << "\n============================================================================================\n\n";
}
// loadProfile =====================================================================================================================================
// loads an existing profile
int loadProfile(QString profileToLoad)
{
	int count;
	profile.setFileName(profileToLoad);

	if (!profile.open(QIODevice::ReadOnly))		//if the profile cannot be opened
	{
		profile.close();
		return 1;
	}

	QDataStream in(&profile);
	in.setVersion(QDataStream::Qt_4_3);

	QVariant v;					//we will import everything as QVariant using this temp variable
	QString vString;				//temp variable to import "labels" of real data
	QString tempAppName = "asxeto";
	double tempAppVersion=0;
	in>>v;	if (v.toString()=="appName")
		in >> v;	tempAppName = v.toString();	//input the application name & version--------------------------
	in>>v;	if (v.toString()=="appVersion")
		in >> v;	tempAppVersion = v.toDouble();

	if ( (tempAppName != appName) || (tempAppVersion < validProfileVersion) )//check if the file is a valid luckybackup profile
	{
		profile.close();
		return 2;		//profile is not valid
	}

	in>>v;	if (v.toString()=="TotalOperations")
		in >> v;	TotalOperations = v.toInt();	//input the size of the operations list 

	currentOperation = 0;	vString="";
	while (currentOperation < TotalOperations)
	{
		operation *tempOp = new operation;

		in>>v;	vString = v.toString();	in >> v;	//input a label in vString and real data in v
		while (vString != "operation end")
		{
			//cout << "\ntask: " <<currentOperation << "\nvString: " << vString.toStdString() << "\n";
			if (vString == "Name")			tempOp	-> SetName(v.toString());
			if (vString == "Args")			tempOp	-> SetArgs(v.toStringList());

			if (vString == "ConnectRestore")	tempOp	-> SetConnectRestore(v.toString());

			if (vString == "LastExecutionTime")	tempOp	-> SetLastExecutionTime(v.toDateTime());
			if (vString == "LastExecutionErrors")	tempOp	-> SetLastExecutionErrors(v.toInt());
			if (vString == "KeepSnapshots")		tempOp	-> SetKeepSnapshots(v.toInt()); //see also "update the max.." few lines below
			if (vString == "SnapshotsListSize")
			{
				int snapshotsListSize = v.toInt();
				count = 0;
				while ( count < snapshotsListSize)
				{
					in >> v;		// load the string SnapshotsListItem1,2....XX
					in >> v; 		tempOp	-> AddSnapshotsListItem(v.toString());
					count++;
				}
			}

			if (vString == "TypeDirContents")	tempOp	-> SetTypeDirContents(v.toBool());
			if (vString == "TypeDirName")		tempOp	-> SetTypeDirName(v.toBool());
			if (vString == "TypeSync")		tempOp	-> SetTypeSync(v.toBool());
	
			if (vString == "Source")		tempOp	-> SetSource(v.toString());
			if (vString == "Destination")		tempOp	-> SetDestination(v.toString());

			if (vString == "ExcludeTemp")		tempOp	-> SetExcludeTemp(v.toBool());
			if (vString == "ExcludeFromFile")	tempOp	-> SetExcludeFromFile(v.toBool());
			if (vString == "ExcludeFile")		tempOp	-> SetExcludeFile(v.toString());
			if (vString == "ExcludeCache")		tempOp	-> SetExcludeCache(v.toBool());
			if (vString == "ExcludeBackup")		tempOp	-> SetExcludeBackup(v.toBool());
			if (vString == "ExcludeMount")		tempOp	-> SetExcludeMount(v.toBool());
			if (vString == "ExcludeLostFound")	tempOp	-> SetExcludeLostFound(v.toBool());
			if (vString == "ExcludeSystem")		tempOp	-> SetExcludeSystem(v.toBool());
			if (vString == "ExcludeTrash")		tempOp	-> SetExcludeTrash(v.toBool());
			if (vString == "ExcludeListSize")
			{
				int excludeListSize = v.toInt();
				count = 0;
				while ( count < excludeListSize)
				{
					in >> v; 		tempOp	-> AddExcludeListItem(v.toString());
					count++;
				}
			}
			tempOp	-> SetExclude();

			if (vString == "IncludeListSize")
			{
				int IncludeListSize = v.toInt();
				count = 0;
				while ( count < IncludeListSize)
				{
					in >> v; 		tempOp	-> AddIncludeListItem(v.toString());
					count++;
				}
			}
			if (vString == "IncludeModeNormal")	tempOp	-> SetIncludeModeNormal(v.toBool());
			if (vString == "IncludeFromFile")	tempOp	-> SetIncludeFromFile(v.toBool());
			if (vString == "IncludeFile")		tempOp	-> SetIncludeFile(v.toString());
			tempOp	-> SetInclude();

			if (vString == "Remote")			tempOp	-> SetRemote(v.toBool());
			if (vString == "RemoteModule")		tempOp	-> SetRemoteModule(v.toBool());
			if (vString == "RemoteDestination")	tempOp	-> SetRemoteDestination(v.toBool());
			if (vString == "RemoteSource")		tempOp	-> SetRemoteSource(v.toBool());
			if (vString == "RemoteSSH")			tempOp	-> SetRemoteSSH(v.toBool());
			if (vString == "RemoteHost")		tempOp	-> SetRemoteHost(v.toString());
			if (vString == "RemoteUser")		tempOp	-> SetRemoteUser(v.toString());
			if (vString == "RemotePassword")	tempOp	-> SetRemotePassword(v.toString());
			if (vString == "RemoteSSHPassword")	tempOp	-> SetRemoteSSHPassword(v.toString());
			if (vString == "RemoteSSHPort")		tempOp	-> SetRemoteSSHPort(v.toInt());

			if (vString == "OptionsUpdate")		tempOp	-> SetOptionsUpdate(v.toBool());
			if (vString == "OptionsDelete")		tempOp	-> SetOptionsDelete(v.toBool());
			if (vString == "OptionsRecurse")	tempOp	-> SetOptionsRecurse(v.toBool());
			if (vString == "OptionsOwnership")	tempOp	-> SetOptionsOwnership(v.toBool());
			if (vString == "OptionsSymlinks")	tempOp	-> SetOptionsSymlinks(v.toBool());
			if (vString == "OptionsPermissions") tempOp	-> SetOptionsPermissions(v.toBool());
			if (vString == "OptionsDevices")	tempOp	-> SetOptionsDevices(v.toBool());
			if (vString == "OptionsCVS")		tempOp	-> SetOptionsCVS(v.toBool());
			if (vString == "OptionsHardLinks")	tempOp	-> SetOptionsHardLinks(v.toBool());
			if (vString == "OptionsFATntfs")	tempOp	-> SetOptionsFATntfs(v.toBool());
			if (vString == "OptionsListSize")
			{
				int OptionsListSize = v.toInt();
				count = 0;
				while ( count < OptionsListSize)
				{
					in >> v;			tempOp	-> AddOptionsListItem(v.toString());
					count++;
				}
			}
			if (vString == "ExecuteBeforeListSize")
			{
				int ExecuteBeforeListSize = v.toInt();
				count = 0;
				while ( count < ExecuteBeforeListSize)
				{
					in >> v;			tempOp	-> AddExecuteBeforeListItem(v.toString());
					count++;
				}
			}
			if (vString == "ExecuteAfterListSize")
			{
				int ExecuteAfterListSize = v.toInt();
				count = 0;
				while ( count < ExecuteAfterListSize)
				{
					in >> v;			tempOp	-> AddExecuteAfterListItem(v.toString());
					count++;
				}
			}
			if (vString == "ByPassWarning")		tempOp	-> SetByPassWARNING(v.toBool());
			if (vString == "CloneWarning")		tempOp	-> SetCloneWARNING(v.toBool());
			//import include checkbox state
			if (vString == "IncludeState")		tempOp	-> SetIncluded(v.toBool());
			in>>v;	vString = v.toString();
			if (vString!="operation end")
				in >> v;
			else			// all properties have been loaded
			{
				// first fix some icompatibilities with older versions that did not save the following properties:
				// 1. update the max "keep snapshots" number if it is < 1 (or not defined !!) or if the task is a sync task
				if ((tempOp -> GetTypeSync()) || (tempOp -> GetKeepSnapshots() < 1) )
					tempOp -> SetKeepSnapshots(1);
				// 2. Add a snapshot if last executiontime exists but snapshot list is empty
				QString LastTime = ( tempOp -> GetLastExecutionTime() ).toString("yyyyMMddhhmmss");
				if ( !(LastTime == "") && (tempOp -> SnapshotsListIsEmpty()) )
					tempOp -> AddSnapshotsListItem (LastTime);

				Operation[currentOperation] = tempOp;	// update the currentOperation
			}
		}
		currentOperation++;
	}
	profile.close();
	return 0;
}

// saveProfile =====================================================================================================================================
// saves an existing profile
bool saveProfile(QString profileToSave)
{
	int count;

	profile.setFileName(profileToSave);
	
	if (!profile.open(QIODevice::WriteOnly))	// if the profile cannot be saved (or fails to create)
	{
		profile.close();
		return FALSE;
	}

	//write arrays to profile file
	QDataStream out(&profile);
	out.setVersion(QDataStream::Qt_4_3);

	QVariant v;					//we will export everything as QVariant using this temp variable
	v = "appName";		out << v;	v = appName;		out << v;	//output the application name 
	v = "appVersion";	out << v;	v = appVersion;		out << v;	//output the application version 
	v = "TotalOperations";	out << v;	v = TotalOperations;	out << v;	//output the size of the opeartions list 


	currentOperation = 0;

	while ( currentOperation < TotalOperations )
	{
		v = "Name";			out << v;	v = Operation[currentOperation] -> GetName();			out << v;
		v = "Args";			out << v;	v = Operation[currentOperation] -> GetArgs();			out << v;

		v = "ConnectRestore";		out << v;	v = Operation[currentOperation] -> GetConnectRestore();		out << v;

		v = "LastExecutionTime";	out << v;	v = Operation[currentOperation] -> GetLastExecutionTime();	out << v;
		v = "LastExecutionErrors";	out << v;	v = Operation[currentOperation] -> GetLastExecutionErrors();	out << v;
		v = "KeepSnapshots";		out << v;	v = Operation[currentOperation] -> GetKeepSnapshots();		out << v;
		v = "SnapshotsListSize";	out << v;	v = Operation[currentOperation] -> GetSnapshotsListSize();	out << v;
		count = 0;
		while ( count < (Operation[currentOperation] -> GetSnapshotsListSize()) )
		{
			v = "SnapshotsListItem" + countStr.setNum(count);		out << v;
			v = Operation[currentOperation] -> GetSnapshotsListItem(count); out << v;
			count++;
		}

		v = "TypeDirContents";		out << v;	v = Operation[currentOperation] -> GetTypeDirContents();	out << v;
		v = "TypeDirName";		out << v;	v = Operation[currentOperation] -> GetTypeDirName();		out << v;
		v = "TypeSync";			out << v;	v = Operation[currentOperation] -> GetTypeSync();		out << v;

		v = "Source";			out << v;	v = Operation[currentOperation] -> GetSource();			out << v;
		v = "Destination";		out << v;	v = Operation[currentOperation] -> GetDestination();		out << v;

		v = "Exclude";			out << v;	v = Operation[currentOperation] -> GetExclude();		out << v;
		v = "ExcludeFromFile";		out << v;	v = Operation[currentOperation] -> GetExcludeFromFile();	out << v;
		v = "ExcludeFile";		out << v;	v = Operation[currentOperation] -> GetExcludeFile();		out << v;
		v = "ExcludeTemp";		out << v;	v = Operation[currentOperation] -> GetExcludeTemp();		out << v;
		v = "ExcludeCache";		out << v;	v = Operation[currentOperation] -> GetExcludeCache();		out << v;
		v = "ExcludeBackup";		out << v;	v = Operation[currentOperation] -> GetExcludeBackup();		out << v;
		v = "ExcludeMount";		out << v;	v = Operation[currentOperation] -> GetExcludeMount();		out << v;
		v = "ExcludeLostFound";		out << v;	v = Operation[currentOperation] -> GetExcludeLostFound();	out << v;
		v = "ExcludeSystem";		out << v;	v = Operation[currentOperation] -> GetExcludeSystem();		out << v;
		v = "ExcludeTrash";		out << v;	v = Operation[currentOperation] -> GetExcludeTrash();		out << v;
		v = "ExcludeListSize";		out << v;	v = Operation[currentOperation] -> GetExcludeListSize();	out << v;
		count = 0;
		while ( count < (Operation[currentOperation] -> GetExcludeListSize()) )
		{
			v = Operation[currentOperation] -> GetExcludeListItem(count); out << v;
			count++;
		}

		v = "Include";			out << v;	v = Operation[currentOperation] -> GetInclude();		out << v;
		v = "IncludeFromFile";		out << v;	v = Operation[currentOperation] -> GetIncludeFromFile();	out << v;
		v = "IncludeModeNormal";	out << v;	v = Operation[currentOperation] -> GetIncludeModeNormal();	out << v;
		v = "IncludeFile";		out << v;	v = Operation[currentOperation] -> GetIncludeFile();		out << v;
		v = "IncludeListSize";		out << v;	v = Operation[currentOperation] -> GetIncludeListSize();	out << v;
		count = 0;
		while ( count < (Operation[currentOperation] -> GetIncludeListSize()) )
		{
			v = Operation[currentOperation] -> GetIncludeListItem(count); out << v;
			count++;
		}

		v = "Remote";			out << v;	v = Operation[currentOperation] -> GetRemote();			out << v;
		v = "RemoteModule";		out << v;	v = Operation[currentOperation] -> GetRemoteModule();		out << v;
		v = "RemoteDestination";	out << v;	v = Operation[currentOperation] -> GetRemoteDestination();	out << v;
		v = "RemoteSource";		out << v;	v = Operation[currentOperation] -> GetRemoteSource();		out << v;
		v = "RemoteSSH";		out << v;	v = Operation[currentOperation] -> GetRemoteSSH();		out << v;
		v = "RemoteHost";		out << v;	v = Operation[currentOperation] -> GetRemoteHost();		out << v;
		v = "RemoteUser";		out << v;	v = Operation[currentOperation] -> GetRemoteUser();		out << v;
		v = "RemotePassword";		out << v;	v = Operation[currentOperation] -> GetRemotePassword();		out << v;
		v = "RemoteSSHPassword";	out << v;	v = Operation[currentOperation] -> GetRemoteSSHPassword();	out << v;
		v = "RemoteSSHPort";		out << v;	v = Operation[currentOperation] -> GetRemoteSSHPort();		out << v;

		v = "OptionsUpdate";		out << v;	v = Operation[currentOperation] -> GetOptionsUpdate();		out << v;
		v = "OptionsDelete";		out << v;	v = Operation[currentOperation] -> GetOptionsDelete();		out << v;
		v = "OptionsRecurse";		out << v;	v = Operation[currentOperation] -> GetOptionsRecurse();		out << v;
		v = "OptionsOwnership";		out << v;	v = Operation[currentOperation] -> GetOptionsOwnership();	out << v;
		v = "OptionsSymlinks";		out << v;	v = Operation[currentOperation] -> GetOptionsSymlinks();	out << v;
		v = "OptionsPermissions";	out << v;	v = Operation[currentOperation] -> GetOptionsPermissions();	out << v;
		v = "OptionsDevices";		out << v;	v = Operation[currentOperation] -> GetOptionsDevices();		out << v;
		v = "OptionsCVS";			out << v;	v = Operation[currentOperation] -> GetOptionsCVS();		out << v;
		v = "OptionsHardLinks";		out << v;	v = Operation[currentOperation] -> GetOptionsHardLinks();	out << v;
		v = "OptionsFATntfs";		out << v;	v = Operation[currentOperation] -> GetOptionsFATntfs();		out << v;
		v = "OptionsListSize";		out << v;	v = Operation[currentOperation] -> GetOptionsListSize();	out << v;
		count = 0;
		while ( count < (Operation[currentOperation] -> GetOptionsListSize()) )
		{
			v = Operation[currentOperation] -> GetOptionsListItem(count); out << v;
			count++;
		}
		v = "ExecuteBeforeListSize";	out << v;	v = Operation[currentOperation] -> GetExecuteBeforeListSize();	out << v;
		count = 0;
		while ( count < (Operation[currentOperation] -> GetExecuteBeforeListSize()) )
		{
			v = Operation[currentOperation] -> GetExecuteBeforeListItem(count); out << v;
			count++;
		}
		v = "ExecuteAfterListSize";	out << v;	v = Operation[currentOperation] -> GetExecuteAfterListSize();	out << v;
		count = 0;
		while ( count < (Operation[currentOperation] -> GetExecuteAfterListSize()) )
		{
			v = Operation[currentOperation] -> GetExecuteAfterListItem(count); out << v;
			count++;
		}
		v = "ByPassWarning";		out << v;	v = Operation[currentOperation] -> GetByPassWARNING();		out << v;
		v = "CloneWarning";		out << v;	v = Operation[currentOperation] -> GetCloneWARNING();		out << v;
		//export include state
		v = "IncludeState";		out << v;	v = Operation[currentOperation] -> GetIncluded();		out << v;

		v = "operation end";	out << v;	
		currentOperation++;
	}
	v = "profile end";	out << v;

	profile.close();
	return TRUE;
}

// checkTaskList =====================================================================================================================================
// Checks if the Task list is ok to proceed
bool checkTaskList()
{
	int count;
	QString tempConnect="";
	message="";
	messageCLI="";
	ask = FALSE;
	NothingIncluded=TRUE;

	if (TotalOperations==0)						//check if the operations list is empty ------------------------------
	{
		message.append("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~<br><b>" +
				QObject::tr("The task list is empty")+" !!</b><br>..."+
			QObject::tr("nothing to start"));
		messageCLI.append("\nThe task list is empty !!\n"
			"nothing to start !!\n\n");
		ask = TRUE;
		return FALSE;
	}

	//first check if 2 connected operations have been selected for execution--------------------------------------------------------------
	//set NothingInluded to FALSE if 1 or more operation is included
	bool askTemp = FALSE;	QString dirNames="";	QString dirNamesCLI="";	currentOperation=0;
	while  (currentOperation < TotalOperations)
	{
		if (Operation[currentOperation] -> GetIncluded())	//if the operations is "included"
		{
			NothingIncluded = FALSE;

			tempConnect = Operation[currentOperation] -> GetConnectRestore();
			if (tempConnect != "")							//if it is connected to another operation
			{
				count = currentOperation+1;
				while (count < TotalOperations)
				{
					if  ( (Operation[count]->GetName() == tempConnect)	//if 2 connected operations are both included
					&& (Operation[count] -> GetIncluded()) )
					{
						dirNames.append("* " + Operation[currentOperation]->GetName() + "<br>* " + tempConnect + "<br>");
						dirNamesCLI.append(Operation[currentOperation]->GetName() + "\n" + tempConnect + "\n\n");
						askTemp = TRUE; ask=TRUE;
						break;
					}
					count++;
				}
			}
		}
		currentOperation++;
	}
	if (askTemp)
	{
		message.append("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~<br>" +
				QObject::tr("You have included both of the following connected tasks:")+"<b><br>" + dirNames + 
				"</b>" + QObject::tr("this is not allowed for obvious reasons")+" !!<br>"+
				QObject::tr("Please <b>uncheck appropriate include boxes</b> before proceeding")+"... <br>");
		messageCLI.append("\nYou have included both of the following connected tasks:\n" + QString(dirNamesCLI.toUtf8()) + 
				"this is not allowed for obvious reasons !!\n"
				"Please uncheck appropriate include boxes before proceeding...\n\n");
	}

	if ( (NothingIncluded) && (TotalOperations!=0) )		//if no checkboxes are selected show this message ---------------------
	{
		message.append("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~<br><b>"+
			QObject::tr("You have not included any tasks")+" !!</b><br>"
			"..." + QObject::tr("nothing to start"));
		messageCLI.append("\nYou have not included any tasks !!\n\n"
			"nothing to start !!\n\n");
		ask = TRUE;
		return FALSE;
	}

	//check if 2 or more identical destination directories are decleared & "Backup dir contents" is checked -------------------------------
	currentOperation=0;	askTemp = FALSE;	dirNames="";	dirNamesCLI="";
	QString dest1="", dest2="";
	//the following loop compares an included operation (currentoperation) with all the other included
	//to check if they share the same destination
	while  (currentOperation < TotalOperations)
	{
		dest1 = Operation[currentOperation] -> GetDestination();
		if (dest1.endsWith("/")) dest1.chop(1);

		if (Operation[currentOperation] -> GetIncluded())
		{
			count = 0;
			while (count < TotalOperations)
			{
				if (count == currentOperation) count++;
				else
				{
					dest2 = Operation[count] -> GetDestination();
					if (dest2.endsWith("/")) dest2.chop(1);
				
				//if this operation's destination is identical to another one's which is included and is of type 'Backup dir contents'
					if ( (Operation[count] -> GetIncluded()) 
					&& ( dest1 == dest2 )
					&& ( ((Operation[count] -> GetTypeDirContents()) && (!Operation[count] -> GetInclude())) ||
					((Operation[currentOperation] -> GetTypeDirContents()) && (!Operation[currentOperation] -> GetInclude()))) )
					{
						dirNames.append("* " + Operation[currentOperation]->GetName() + "<br>");
						dirNamesCLI.append(Operation[currentOperation]->GetName() + "\n");
						askTemp = TRUE; ask=TRUE;
						break;
					}
					count++;
				}
			}
		}
		currentOperation++;
	}
	if (askTemp)
	{
		message.append("~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~<br>" +
				QObject::tr("The following included tasks <b>share the same destination directory</b> (all together or at groups) and at least one of them will backup the contents of a directory")+" :<br><b>"
				+ dirNames +
				"</b>" + QObject::tr("this will lead to one task canceling the other")+" !!<br>"+
				QObject::tr("Please uncheck appropriate include boxes or change all task types to other than '<b>Backup dir contents</b>' "
				" or use 'Backup dir contents' together with the '<b>Only include</b>' option, before proceeding")+"...");

		messageCLI.append("\nThe following included tasks share the same destination directory (all together or at groups) and at least one of them will backup the contents of a directory :\n"
				+ QString(dirNamesCLI.toUtf8()) +
				"\nthis will lead to one task canceling the other !!\n"
				"Please uncheck appropriate include boxes or change all task types to other than 'Backup dir contents'\n "
				"or use 'Backup dir contents' together with the 'Only include' option, before proceeding...\n\n");
	}

	return TRUE;
}

// checkDeclaredDirs =================================================================================================================================
//Check if the declared data are ok by calling checkBackupDirs or checkSyncDirs
//If guiExec is TRUE this is called from the gui, otherwise from console
bool checkDeclaredDirs(bool guiExec)
{
	int count;
	ask=FALSE;
	NothingToDo = TRUE;

	QString source, dest;

	count = 0;
	currentOperation = 0;
	CheckedData = "";
	CheckedDataCLI = "";
	while (currentOperation < TotalOperations)
	{
		Operation[currentOperation] -> SetOK(FALSE);
		Operation[currentOperation] -> SetWARNING(FALSE);
		Operation[currentOperation] -> SetCRITICAL(FALSE);
		Operation[currentOperation] -> SetSourcePerms(FALSE);
		Operation[currentOperation] -> SetDestPerms(FALSE);

		if (Operation[currentOperation] -> GetIncluded())	//if the operations is "included"
		{
			//first set  variables source & dest as well as itsPerform which will finaly decide if the task will be preformed
			source = Operation[currentOperation] -> GetSource();
			dest = Operation[currentOperation] -> GetDestination();
			if (!guiExec)	//this is used for compatibility issues with console
			{
				source=QString(source.toUtf8());
				dest=QString(dest.toUtf8());
			}
			Operation[currentOperation] -> SetIncluded(TRUE);
			Operation[currentOperation] -> SetPerform(TRUE);	//this will change at the next commands

			if ( (Operation[currentOperation] -> GetTypeDirContents()) || (Operation[currentOperation] -> GetTypeDirName()) )	
				checkBackupDirs(source,dest);			//if the operation is of type "backup dir ...'
			if (Operation[currentOperation] -> GetTypeSync())	//if the operation is of type "sync dirs'
				checkSyncDirs(source,dest);
		}
		else
		{
			Operation[currentOperation] -> SetIncluded(FALSE);
			Operation[currentOperation] -> SetPerform(FALSE);	
		}
		currentOperation++;
	}
	return TRUE;
}

//===================================================================================================================================================
//Check if the directories to be synced are empty or don't exist
void checkSyncDirs(QString source, QString dest)
{
	Operation[currentOperation] -> SetOK(FALSE);
	Operation[currentOperation] -> SetWARNING(FALSE);
	Operation[currentOperation] -> SetCRITICAL(FALSE);
	Operation[currentOperation] -> SetSourcePerms(FALSE);
	Operation[currentOperation] -> SetDestPerms(FALSE);

	bool remoteSource = FALSE;
	bool remoteDest = FALSE;

	if ( (Operation[currentOperation] -> GetRemote()) && (Operation[currentOperation] -> GetRemoteSource()) )
	{
		if (Operation[currentOperation] -> GetRemoteModule())
			source.prepend(":");
		source.prepend((Operation[currentOperation] -> GetRemoteHost())+":");
		if (Operation[currentOperation] -> GetRemoteUser()!="")
			source.prepend((Operation[currentOperation] -> GetRemoteUser())+"@");
		remoteSource = TRUE;
	}

	if ( (Operation[currentOperation] -> GetRemote()) && (Operation[currentOperation] -> GetRemoteDestination()) )
	{
		if (Operation[currentOperation] -> GetRemoteModule())
			dest.prepend(":");
		dest.prepend((Operation[currentOperation] -> GetRemoteHost())+":");
		if (Operation[currentOperation] -> GetRemoteUser()!="")
			dest.prepend((Operation[currentOperation] -> GetRemoteUser())+"@");
		remoteDest = TRUE;
	}

	QFileInfo dirAFile(source);
	QFileInfo dirBFile(dest);
	QString currentOpNameCLI = QString((Operation[currentOperation] -> GetName())).toUtf8();
	
	int lastErrors = Operation[currentOperation] -> GetLastExecutionErrors();
	QString lastTimeReadable = (Operation[currentOperation] -> GetLastExecutionTime()).toString(Qt::DefaultLocaleLongDate);
	QString lastTime = (Operation[currentOperation] -> GetLastExecutionTime()).toString();

	//If user does not have sufficient permissions for the dirA or dirB, skip it
	if ( (dirAFile.exists()) && ((!dirAFile.isReadable()) || (!dirAFile.isExecutable())) && (!remoteSource) )
	{
		CheckedData.append(QObject::tr("Last execution time") + ": <b>");
		if (lastTime == "")
			CheckedData.append(QObject::tr("not available") + "</b><br>");
		else
		{
			CheckedData.append(lastTimeReadable +"</b>");
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedData.append(" (" + QObject::tr("no errors") + ")");
				else
					CheckedData.append(" (" + QObject::tr("errors found") + ")");
			}
			CheckedData.append("<br>");
		}
		CheckedData.append(QObject::tr("status") + ": <font color=red><b>"+QObject::tr("WARNING")+"</font></b><br>");
		CheckedData.append(QObject::tr("I do not have the permission to read/enter sync directory A")+":<b><br>");
		CheckedData.append(source);
		if (Operation[currentOperation] -> GetByPassWARNING())
			CheckedData.append("</b><br>"+QObject::tr("I will <font color=red><b>NOT SKIP</b></font> this task because you have enabled the \"by-pass WARNING\" option"));
		else
			CheckedData.append("</b><br>"+QObject::tr("This task will be <font color=red><b>skipped</b></font> to avoid errors"));

		

		CheckedDataCLI.append("[WARNING]	->	");
		CheckedDataCLI.append(currentOpNameCLI);
		CheckedDataCLI.append("\nLast execution time	: ");
		if (lastTime == "")
			CheckedDataCLI.append("not available");
		else
		{
			CheckedDataCLI.append(lastTimeReadable);
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedDataCLI.append(" (no errors)");
				else
					CheckedDataCLI.append(" (errors found)");
			}
		}
		CheckedDataCLI.append("\nI do not have the permission to read/enter sync directory A :\n");
		CheckedDataCLI.append(source);
		if (Operation[currentOperation] -> GetByPassWARNING())
			CheckedDataCLI.append("\nThis task will NOT be skipped because you have enabled the \"by-pass WARNING\" option\n");
		else
			CheckedDataCLI.append("\nThis task will be skipped to avoid errors\n");
		CheckedDataCLI.append("_____________________________________________________________________________________________\n");

		if (!Operation[currentOperation] -> GetByPassWARNING())
			Operation[currentOperation] -> SetPerform(FALSE);	//don't perform this operation if the "bypass WARNING" OPTION is disabled
		else
			NothingToDo = FALSE;
		Operation[currentOperation] -> SetSourcePerms(TRUE);
		ask=TRUE;	//ask the user if he/she wants to continue
		return;
	}

	if ( (dirBFile.exists()) && ((!dirBFile.isReadable()) || (!dirBFile.isExecutable())) && (!remoteDest) )
	{
		CheckedData.append(QObject::tr("Last execution time") + ": <b>");
		if (lastTime == "")
			CheckedData.append(QObject::tr("not available") + "</b><br>");
		else
		{
			CheckedData.append(lastTimeReadable +"</b>");
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedData.append(" (" + QObject::tr("no errors") + ")");
				else
					CheckedData.append(" (" + QObject::tr("errors found") + ")");
			}
			CheckedData.append("<br>");
		}
		CheckedData.append(QObject::tr("status") + ": <font color=red><b>"+QObject::tr("WARNING")+"</font></b><br>");
		CheckedData.append(QObject::tr("I do not have the permission to read/enter sync directory B")+":<b><br>");
		CheckedData.append(dest);
		if (Operation[currentOperation] -> GetByPassWARNING())
			CheckedData.append("</b><br>"+QObject::tr("I will <font color=red><b>NOT SKIP</b></font> this task because you have enabled the \"by-pass WARNING\" option"));
		else
			CheckedData.append("</b><br>"+QObject::tr("This task will be <font color=red><b>skipped</b></font> to avoid errors"));

		CheckedDataCLI.append("[WARNING]	->	");
		CheckedDataCLI.append(currentOpNameCLI);
		CheckedDataCLI.append("\nLast execution time	: ");
		if (lastTime == "")
			CheckedDataCLI.append("not available");
		else
		{
			CheckedDataCLI.append(lastTimeReadable);
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedDataCLI.append(" (no errors)");
				else
					CheckedDataCLI.append(" (errors found)");
			}
		}
		CheckedDataCLI.append("\nI do not have the permission to read/enter sync directory B :\n");
		CheckedDataCLI.append(dest);
		if (Operation[currentOperation] -> GetByPassWARNING())
			CheckedDataCLI.append("\nThis task will NOT be skipped because you have enabled the \"by-pass WARNING\" option\n");
		else
			CheckedDataCLI.append("\nThis task will be skipped to avoid errors\n");
		CheckedDataCLI.append("_____________________________________________________________________________________________\n");

		ask=TRUE;	//ask the user if he/she wants to continue
		if (!Operation[currentOperation] -> GetByPassWARNING())
			Operation[currentOperation] -> SetPerform(FALSE);	//don't perform this operation if the "bypass WARNING" OPTION is disabled
		else
			NothingToDo = FALSE;
		Operation[currentOperation] -> SetDestPerms(TRUE);
		return;
	}

	QDir dirA (source);
	QDir dirB (dest);
	QStringList dirAList = dirA.entryList(QDir::AllEntries | QDir::Hidden | QDir::NoDotAndDotDot);
	QStringList dirBList = dirB.entryList(QDir::AllEntries | QDir::Hidden | QDir::NoDotAndDotDot);
	bool dirAIsEmpty, dirBIsEmpty;
	if (dirAList.size() == 0)
		dirAIsEmpty = TRUE;
	else
		dirAIsEmpty = FALSE;
	if (dirBList.size() == 0)
		dirBIsEmpty = TRUE;
	else
		dirBIsEmpty = FALSE;

	//Do directories exist and are not emty ??(skip remote)								->	[ Ok ]
	if ( ( ((dirA.exists())  && !(dirAIsEmpty)) || (remoteSource) )
	&& (((dirB.exists()) && !(dirBIsEmpty)) ||(remoteDest) ) )
	{
		CheckedData.append(QObject::tr("Last execution time") + ": <b>");
		if (lastTime == "")
			CheckedData.append(QObject::tr("not available") + "</b><br>");
		else
		{
			CheckedData.append(lastTimeReadable +"</b>");
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedData.append(" (" + QObject::tr("no errors") + ")");
				else
					CheckedData.append(" (" + QObject::tr("errors found") + ")");
			}
			CheckedData.append("<br>");
		}
		CheckedData.append(QObject::tr("status") + ": <font color=green><b>" + QObject::tr("OK") + "</font></b><br>");
		CheckedData.append(QObject::tr("Sync directory")+" A: <b>");
		CheckedData.append(source);
		if (remoteSource)
			CheckedData.append("<br><font color=magenta>("+QObject::tr("Using remote, check is skipped")+"...)</font>");
		CheckedData.append("</b><br>" + QObject::tr("Sync directory")+" B: <b>");
		CheckedData.append(dest);
		if (remoteDest)
			CheckedData.append("<br><font color=magenta>("+QObject::tr("Using remote, check is skipped")+"...)</font>");
		CheckedData.append("</b>");

		CheckedDataCLI.append("[Ok]		->	");
		CheckedDataCLI.append(currentOpNameCLI);
		CheckedDataCLI.append("\nLast execution time	: ");
		if (lastTime == "")
			CheckedDataCLI.append("not available");
		else
		{
			CheckedDataCLI.append(lastTimeReadable);
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedDataCLI.append(" (no errors)");
				else
					CheckedDataCLI.append(" (errors found)");
			}
		}
		CheckedDataCLI.append("\nSync directory A	: ");
		CheckedDataCLI.append(source);
		CheckedDataCLI.append("\nSync directory B	: ");
		CheckedDataCLI.append(dest);
		CheckedDataCLI.append("\n_____________________________________________________________________________________________\n");

		NothingToDo = FALSE;
		Operation[currentOperation] -> SetOK(TRUE);
	}
	
	//is one or both sync dirs non-existent ??									->	[ WARNING ]
	else if ( !(dirA.exists()) || !(dirB.exists()) )
	{
		CheckedData.append(QObject::tr("Last execution time") + ": <b>");
		if (lastTime == "")
			CheckedData.append(QObject::tr("not available") + "</b><br>");
		else
		{
			CheckedData.append(lastTimeReadable +"</b>");
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedData.append(" (" + QObject::tr("no errors") + ")");
				else
					CheckedData.append(" (" + QObject::tr("errors found") + ")");
			}
			CheckedData.append("<br>");
		}
		CheckedData.append(QObject::tr("status") + ": <font color=red><b>"+QObject::tr("WARNING")+"</font></b><br>");
		CheckedData.append(QObject::tr("Directory")+" <b>");
		CheckedData.append(source);
		CheckedData.append("</b><br>"+QObject::tr("and/or")+" <b>");
		CheckedData.append(dest);
		CheckedData.append("</b><br><font color=red><b>"+QObject::tr("does not exist")+"</b></font>.<br>");
		if (Operation[currentOperation] -> GetByPassWARNING())
			CheckedData.append(QObject::tr("I will <font color=red><b>NOT SKIP</b></font> this task because you have enabled the \"by-pass WARNING\" option"));
		else
			CheckedData.append(QObject::tr("I will <font color=red><b>skip</b></font> this task for that reason"));

		CheckedDataCLI.append("[WARNING]	->	");
		CheckedDataCLI.append(currentOpNameCLI);
		CheckedDataCLI.append("\nLast execution time	: ");
		if (lastTime == "")
			CheckedDataCLI.append("not available");
		else
		{
			CheckedDataCLI.append(lastTimeReadable);
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedDataCLI.append(" (no errors)");
				else
					CheckedDataCLI.append(" (errors found)");
			}
		}
		CheckedDataCLI.append("\nDirectory ");
		CheckedDataCLI.append(source);
		CheckedDataCLI.append("\nand/or ");
		CheckedDataCLI.append(dest);
		CheckedDataCLI.append("\ndoes not exist.\n");
		if (Operation[currentOperation] -> GetByPassWARNING())
			CheckedDataCLI.append("This task will NOT be skipped because you have enabled the \"by-pass WARNING\" option\n");
		else
			CheckedDataCLI.append("I will SKIP this task for that reason\n");
		CheckedDataCLI.append("_____________________________________________________________________________________________\n");

		if (!Operation[currentOperation] -> GetByPassWARNING())
			Operation[currentOperation] -> SetPerform(FALSE);	//don't perform this operation if the "bypass WARNING" OPTION is disabled
		else
			NothingToDo = FALSE;
		Operation[currentOperation] -> SetWARNING(TRUE);
		ask=TRUE;
	}

	//Is one or both of the directories empty ??									->	[ CRITICAL ]
	else
	{
		CheckedData.append(QObject::tr("Last execution time") + ": <b>");
		if (lastTime == "")
			CheckedData.append(QObject::tr("not available") + "</b><br>");
		else
		{
			CheckedData.append(lastTimeReadable +"</b>");
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedData.append(" (" + QObject::tr("no errors") + ")");
				else
					CheckedData.append(" (" + QObject::tr("errors found") + ")");
			}
			CheckedData.append("<br>");
		}

		CheckedData.append(QObject::tr("status") + ": <font color=orange><b>"+QObject::tr("CRITICAL")+"</font></b><br>");
		CheckedData.append(QObject::tr("Directory")+" <b>");
		CheckedData.append(source);
		CheckedData.append("</b><br>"+QObject::tr("and/or")+" <b>");
		CheckedData.append(dest);
		CheckedData.append("</b><br><font color=orange><b>"+QObject::tr("is empty")+"</b></font>.<br>"+
		QObject::tr("I will <font color=orange><b>not skip</b></font> this task. Synchronizing is going to be performed anyway"));

		CheckedDataCLI.append("[CRITICAL]	->	");
		CheckedDataCLI.append(currentOpNameCLI);
		CheckedDataCLI.append("\nLast execution time	: ");
		if (lastTime == "")
			CheckedDataCLI.append("not available");
		else
		{
			CheckedDataCLI.append(lastTimeReadable);
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedDataCLI.append(" (no errors)");
				else
					CheckedDataCLI.append(" (errors found)");
			}
		}
		CheckedDataCLI.append("\nDirectory ");
		CheckedDataCLI.append(source);
		CheckedDataCLI.append("\nand/or ");
		CheckedDataCLI.append(dest);
		CheckedDataCLI.append("\nis empty.\n"
		"I will NOT skip this task (unless '--skip-critical' is given as argument).\n"
		"Synchronizing is going to be performed anyway !!\n"
		"_____________________________________________________________________________________________\n");

		ask=TRUE;	//ask the user if he/she wants to continue
		Operation[currentOperation] -> SetCRITICAL(TRUE);
		if (SkipCritical)						// if a --skip-critical is given as argument
			Operation[currentOperation] -> SetPerform(FALSE);	//don't perform this operation
		else
			NothingToDo = FALSE;
	}

}

//===================================================================================================================================================
//Check if the source and destination backup directories are empty or don't exist
void checkBackupDirs(QString source, QString dest)
{
	Operation[currentOperation] -> SetOK(FALSE);
	Operation[currentOperation] -> SetWARNING(FALSE);
	Operation[currentOperation] -> SetCRITICAL(FALSE);
	Operation[currentOperation] -> SetSourcePerms(FALSE);
	Operation[currentOperation] -> SetDestPerms(FALSE);

	bool remoteSource = FALSE;
	bool remoteDest = FALSE;

	//first check if a remote source or dest is used so that to skip checks
	if ( (Operation[currentOperation] -> GetRemote()) && (Operation[currentOperation] -> GetRemoteSource()) )
	{
		if (Operation[currentOperation] -> GetRemoteModule())
			source.prepend(":");
		source.prepend((Operation[currentOperation] -> GetRemoteHost())+":");
		if (Operation[currentOperation] -> GetRemoteUser()!="")
			source.prepend((Operation[currentOperation] -> GetRemoteUser())+"@");
		remoteSource = TRUE;
	}
	if ( (Operation[currentOperation] -> GetRemote()) && (Operation[currentOperation] -> GetRemoteDestination()) )
	{
		if (Operation[currentOperation] -> GetRemoteModule())
			dest.prepend(":");
		dest.prepend((Operation[currentOperation] -> GetRemoteHost())+":");
		if (Operation[currentOperation] -> GetRemoteUser()!="")
			dest.prepend((Operation[currentOperation] -> GetRemoteUser())+"@");
		remoteDest = TRUE;
	}

	//If user does not have sufficient permissions for the source or dest, skip it
	QFileInfo destFile(dest);
	QFileInfo sourceFile(source);
	QString currentOpNameCLI = QString((Operation[currentOperation] -> GetName())).toUtf8();
	
	int lastErrors = Operation[currentOperation] -> GetLastExecutionErrors();
	QString lastTimeReadable = (Operation[currentOperation] -> GetLastExecutionTime()).toString(Qt::DefaultLocaleLongDate);
	QString lastTime = (Operation[currentOperation] -> GetLastExecutionTime()).toString();

	if ( (sourceFile.exists()) && ((!sourceFile.isReadable()) || (!sourceFile.isExecutable())) && (!remoteSource) )
	{
		CheckedData.append(QObject::tr("Last execution time") + ": <b>");
		if (lastTime == "")
			CheckedData.append(QObject::tr("not available") + "</b><br>");
		else
		{
			CheckedData.append(lastTimeReadable +"</b>");
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedData.append(" (" + QObject::tr("no errors") + ")");
				else
					CheckedData.append(" (" + QObject::tr("errors found") + ")");
			}
			CheckedData.append("<br>");
		}
		CheckedData.append(QObject::tr("status") + ": <font color=red><b>"+QObject::tr("WARNING")+"</font></b><br>");
		CheckedData.append(QObject::tr("I do not have the permission to read/enter the source directory")+":<b><br>");
		CheckedData.append(source);
		if (Operation[currentOperation] -> GetByPassWARNING())
			CheckedData.append("</b><br>"+QObject::tr("I will <font color=red><b>NOT SKIP</b></font> this task because you have enabled the \"by-pass WARNING\" option"));
		else
			CheckedData.append("</b><br>"+QObject::tr("This task will be <font color=red><b>skipped</b></font> to avoid errors"));

		CheckedDataCLI.append("[WARNING]	->	");
		CheckedDataCLI.append(currentOpNameCLI);
		CheckedDataCLI.append("\nLast execution time	: ");
		if (lastTime == "")
			CheckedDataCLI.append("not available");
		else
		{
			CheckedDataCLI.append(lastTimeReadable);
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedDataCLI.append(" (no errors)");
				else
					CheckedDataCLI.append(" (errors found)");
			}
		}
		CheckedDataCLI.append("\nI do not have the permission to read/enter the source directory :\n");
		CheckedDataCLI.append(source);
		if (Operation[currentOperation] -> GetByPassWARNING())
			CheckedDataCLI.append("\nThis task will NOT be skipped because you have enabled the \"by-pass WARNING\" option\n");
		else
			CheckedDataCLI.append("\nThis task will be skipped to avoid errors\n");
		CheckedDataCLI.append("_____________________________________________________________________________________________\n");

		if (!Operation[currentOperation] -> GetByPassWARNING())
			Operation[currentOperation] -> SetPerform(FALSE);	//don't perform this operation if the "bypass WARNING" OPTION is disabled
		else
			NothingToDo = FALSE;

		ask=TRUE;	//ask the user if he/she wants to continue
		Operation[currentOperation] -> SetSourcePerms(TRUE);
		return;
	}

	if ( (destFile.exists()) && ((!destFile.isReadable()) || (!destFile.isExecutable())) && (!remoteDest) )
	{
		CheckedData.append(QObject::tr("Last execution time") + ": <b>");
		if (lastTime == "")
			CheckedData.append(QObject::tr("not available") + "</b><br>");
		else
		{
			CheckedData.append(lastTimeReadable +"</b>");
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedData.append(" (" + QObject::tr("no errors") + ")");
				else
					CheckedData.append(" (" + QObject::tr("errors found") + ")");
			}
			CheckedData.append("<br>");
		}
		CheckedData.append(QObject::tr("status") + ": <font color=red><b>"+QObject::tr("WARNING")+"</font></b><br>");
		CheckedData.append(QObject::tr("I do not have the permission to read/enter the destination directory")+":<b><br>");
		CheckedData.append(dest);
		if (Operation[currentOperation] -> GetByPassWARNING())
			CheckedData.append("</b><br>"+QObject::tr("I will <font color=red><b>NOT SKIP</b></font> this task because you have enabled the \"by-pass WARNING\" option"));
		else
			CheckedData.append("</b><br>"+QObject::tr("This task will be <font color=red><b>skipped</b></font> to avoid errors"));

		CheckedDataCLI.append("[WARNING]	->	");
		CheckedDataCLI.append(currentOpNameCLI);
		CheckedDataCLI.append("\nLast execution time	: ");
		if (lastTime == "")
			CheckedDataCLI.append("not available");
		else
		{
			CheckedDataCLI.append(lastTimeReadable);
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedDataCLI.append(" (no errors)");
				else
					CheckedDataCLI.append(" (errors found)");
			}
		}
		CheckedDataCLI.append("\nI do not have the permission to read/enter the destination directory :\n");
		CheckedDataCLI.append(dest);
		if (Operation[currentOperation] -> GetByPassWARNING())
			CheckedDataCLI.append("\nThis task will NOT be skipped because you have enabled the \"by-pass WARNING\" option\n");
		else
			CheckedDataCLI.append("\nThis task will be skipped to avoid errors\n");
		CheckedDataCLI.append("_____________________________________________________________________________________________\n");

		if (!Operation[currentOperation] -> GetByPassWARNING())
			Operation[currentOperation] -> SetPerform(FALSE);	//don't perform this operation if the "bypass WARNING" OPTION is disabled
		else
			NothingToDo = FALSE;
		ask=TRUE;	//ask the user if he/she wants to continue
		Operation[currentOperation] -> SetDestPerms(TRUE);
		return;
	}

	QDir sourceDir (source);
	QDir destDir (dest);
	QStringList sourceList = sourceDir.entryList(QDir::AllEntries | QDir::Hidden | QDir::NoDotAndDotDot);
	QStringList destList = destDir.entryList(QDir::AllEntries | QDir::Hidden | QDir::NoDotAndDotDot);
	bool SourceIsEmpty, DestIsEmpty;
	if (sourceList.size() == 0)
		SourceIsEmpty = TRUE;
	else
		SourceIsEmpty = FALSE;
	if (destList.size() == 0)
		DestIsEmpty = TRUE;
	else
		DestIsEmpty = FALSE;

	//Does the source directory exist and it is not empty OR we're using remote source?? 
	if  ( ((sourceDir.exists())  && !(SourceIsEmpty)) || (remoteSource) )
	{
		//Does the destination directory exist and it is not empty ?? 								->	[ Ok ]
		if ( ((destDir.exists())  && !(DestIsEmpty)) || (remoteDest) )
		{
			CheckedData.append(QObject::tr("Last execution time") + ": <b>");
			if (lastTime == "")
				CheckedData.append(QObject::tr("not available") + "</b><br>");
			else
			{
				CheckedData.append(lastTimeReadable +"</b>");
				if (lastErrors > -1)	// if there is error information available
				{
					if (lastErrors == 0)
						CheckedData.append(" (" + QObject::tr("no errors") + ")");
					else
						CheckedData.append(" (" + QObject::tr("errors found") + ")");
				}
				CheckedData.append("<br>");
			}
		
			CheckedData.append(QObject::tr("status") + ": <font color=green><b>" + QObject::tr("OK") + "</font></b><br>");
			CheckedData.append(QObject::tr("Source directory")+": <b>");
			CheckedData.append(source+"</b>");
			if (remoteSource)
				CheckedData.append("<br><font color=magenta>("+QObject::tr("Using remote, check is skipped")+"...)</font>");
			CheckedData.append("<br>"+QObject::tr("Destination directory")+": <b>");
			CheckedData.append(dest+"</b>");
			if (remoteDest)
				CheckedData.append("<br><font color=magenta>("+QObject::tr("Using remote, check is skipped")+"...)</font>");

			CheckedDataCLI.append("[Ok]		->	");
			CheckedDataCLI.append(currentOpNameCLI);
			CheckedDataCLI.append("\nLast execution time	: ");
			if (lastTime == "")
				CheckedDataCLI.append("not available");
			else
			{
				CheckedDataCLI.append(lastTimeReadable);
				if (lastErrors > -1)	// if there is error information available
				{
					if (lastErrors == 0)
						CheckedDataCLI.append(" (no errors)");
					else
						CheckedDataCLI.append(" (errors found)");
				}
			}
			CheckedDataCLI.append("\nSource directory	: ");
			CheckedDataCLI.append(source);
			if (remoteSource)
				CheckedDataCLI.append("	(Using remote, check is skipped...)");
			CheckedDataCLI.append("\nDestination directory	: ");
			CheckedDataCLI.append(dest);
			if (remoteDest)
				CheckedDataCLI.append("	(Using remote, check is skipped...)");

			CheckedDataCLI.append("\n_____________________________________________________________________________________________\n");

			Operation[currentOperation] -> SetOK(TRUE);
			NothingToDo = FALSE;
		}
		else //The destination directory does not exist or it is empty							->	[ CRITICAL ]
		{
			CheckedData.append(QObject::tr("Last execution time") + ": <b>");
			if (lastTime == "")
				CheckedData.append(QObject::tr("not available") + "</b><br>");
			else
			{
				CheckedData.append(lastTimeReadable +"</b>");
				if (lastErrors > -1)	// if there is error information available
				{
					if (lastErrors == 0)
						CheckedData.append(" (" + QObject::tr("no errors") + ")");
					else
						CheckedData.append(" (" + QObject::tr("errors found") + ")");
				}
				CheckedData.append("<br>");
			}
			CheckedData.append(QObject::tr("status") + ": <font color=orange><b>"+QObject::tr("CRITICAL")+"</font></b><br>");
			CheckedData.append(QObject::tr("Destination directory")+": <b>");
			CheckedData.append(dest);
			CheckedData.append("<br><font color=orange> "+QObject::tr("is empty or does not exist")+"</font>.</b><br>"+
			QObject::tr("This task will <font color=orange><b>not be skipped")+"</b></font> <br>"+
			QObject::tr("The Destination Directory will be created if it doesn't exist and filled with new backup data")+".");

			CheckedDataCLI.append("[CRITICAL]	->	");
			CheckedDataCLI.append(currentOpNameCLI);
			CheckedDataCLI.append("\nLast execution time	: ");
			if (lastTime == "")
				CheckedDataCLI.append("not available");
			else
			{
				CheckedDataCLI.append(lastTimeReadable);
				if (lastErrors > -1)	// if there is error information available
				{
					if (lastErrors == 0)
						CheckedDataCLI.append(" (no errors)");
					else
						CheckedDataCLI.append(" (errors found)");
				}
			}
			CheckedDataCLI.append("\nSource directory	: ");
			CheckedDataCLI.append(source);
			if (remoteSource)
				CheckedDataCLI.append(" (Using remote, check is skipped...)");
			CheckedDataCLI.append("\nDestination directory	: ");
			CheckedDataCLI.append(dest);
			CheckedDataCLI.append(" is empty or does not exist.\n"
			"This task will NOT be skipped (unless '--skip-critical' is given as argument)\n"
			"The Destination Directory will be created if it doesn't exist and filled with new backup data.\n"
			"_____________________________________________________________________________________________\n");

			ask=TRUE;	//ask the user if he/she wants to continue
			NothingToDo = FALSE;
			Operation[currentOperation] -> SetCRITICAL(TRUE);
			
			if (SkipCritical)						// if a --skip-critical is given as argument
				Operation[currentOperation] -> SetPerform(FALSE);	//don't perform this operation
		}
	}
	else //The source directory does not exist or it is empty								->	[WARNING ]
	{
		CheckedData.append(QObject::tr("Last execution time") + ": <b>");
		if (lastTime == "")
			CheckedData.append(QObject::tr("not available") + "</b><br>");
		else
		{
			CheckedData.append(lastTimeReadable +"</b>");
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedData.append(" (" + QObject::tr("no errors") + ")");
				else
					CheckedData.append(" (" + QObject::tr("errors found") + ")");
			}
			CheckedData.append("<br>");
		}
		CheckedData.append(QObject::tr("status") + ": <font color=red><b>"+QObject::tr("WARNING")+"</font></b><br>");
		CheckedData.append(QObject::tr("Source directory")+": <b>");
		CheckedData.append(source);
		CheckedData.append("<br><font color=red> "+QObject::tr("is empty or does not exist")+"</font>.</b><br>");
		if (Operation[currentOperation] -> GetByPassWARNING())
			CheckedData.append(QObject::tr("I will <font color=red><b>NOT SKIP</b></font> this task because you have enabled the \"by-pass WARNING\" option"));
		else
			CheckedData.append(QObject::tr("This task will be <font color=red><b>skipped</b></font> to protect your backup data (if any !!)"));

		CheckedDataCLI.append("[WARNING]	->	");
		CheckedDataCLI.append(currentOpNameCLI);
		CheckedDataCLI.append("\nLast execution time	: ");
		if (lastTime == "")
			CheckedDataCLI.append("not available");
		else
		{
			CheckedDataCLI.append(lastTimeReadable);
			if (lastErrors > -1)	// if there is error information available
			{
				if (lastErrors == 0)
					CheckedDataCLI.append(" (no errors)");
				else
					CheckedDataCLI.append(" (errors found)");
			}
		}
		CheckedDataCLI.append("\nSource directory	: ");
		CheckedDataCLI.append(source);
		CheckedDataCLI.append(" is empty or does not exist.\n"
				"Destination directory	: ");
		CheckedDataCLI.append(dest);
		if (remoteDest)
			CheckedDataCLI.append("	(Using remote, check is skipped...)");
		if (Operation[currentOperation] -> GetByPassWARNING())
			CheckedDataCLI.append("\nThis task will NOT be skipped because you have enabled the \"by-pass WARNING\" option\n");
		else
			CheckedDataCLI.append("\nThis task will be SKIPPED to protect your backup data (if any !!)\n");
		CheckedDataCLI.append("_____________________________________________________________________________________________\n");

		if (!Operation[currentOperation] -> GetByPassWARNING())
			Operation[currentOperation] -> SetPerform(FALSE);	//don't perform this operation if the "bypass WARNING" OPTION is disabled
		else
			NothingToDo = FALSE;
		Operation[currentOperation] -> SetWARNING(TRUE);
		ask=TRUE;	//ask the user if he/she wants to continue
	}

}

// /AppendArguments =======================================================================================================
//function to append "arguments" object at a given operation according to the fields of the modify window
QStringList AppendArguments(operation *operationToAppend)
{
	QStringList arguments; int count;
	int operationToAppendCurrentSnaps = operationToAppend -> GetSnapshotsListSize();	// this is the current number of snapshots
	int operationToAppendMaxSnaps = operationToAppend -> GetKeepSnapshots();		// this is the max number of snapshots to keep
	
	arguments << "-h" << "--progress" << "--stats";	//These are the standard arguments used by rsync

	//add rsync arguments	--------------------------------------------------------------------------------------------------
	if (operationToAppend -> GetOptionsRecurse())		arguments.append("-r");
	if (operationToAppend -> GetOptionsFATntfs())
	{
		arguments.append("-t");
		arguments.append("--modify-window=1");
	}
	else
	{
		if (operationToAppend -> GetOptionsOwnership())		arguments.append("-tgo");
		if (operationToAppend -> GetOptionsPermissions())	arguments.append("-p");
	}
	if (operationToAppend -> GetOptionsSymlinks())		arguments.append("-l");
	if (operationToAppend -> GetOptionsDevices())		arguments.append("-D");
	if (operationToAppend -> GetOptionsCVS())		arguments.append("-C");
	if (operationToAppend -> GetOptionsHardLinks())		arguments.append("-H");
	if (operationToAppend -> GetOptionsUpdate())		arguments.append("--update");
	if (operationToAppend -> GetOptionsDelete())		arguments.append("--delete-after");
	count =0;
	while ( count < (operationToAppend -> GetOptionsListSize()) )
	{
		arguments.append(operationToAppend -> GetOptionsListItem(count));
		count++;
	}
	
	//add included items------------------------------------------------------------------------------
	if (operationToAppend -> GetInclude())
	{
		// from file
		if ( (operationToAppend -> GetIncludeFromFile()) && !(operationToAppend -> GetIncludeFile()=="") )
			arguments.append("--include-from=" + (operationToAppend -> GetIncludeFile()) );
		// from "only include" tab
		count =0;
		while ( count < (operationToAppend -> GetIncludeListSize()) )
		{
			arguments.append("--include=" + (operationToAppend -> GetIncludeListItem(count)) );
			count++;
		}
		// if "Only Include" mode is used add the following
		if (!operationToAppend -> GetIncludeModeNormal())
		{
			arguments.append("--include=*/");
			arguments.append("--exclude=*");
			arguments.append("--prune-empty-dirs");
		}
	}
	
	//add excluded items------------------------------------------------------------------------------
	if (operationToAppend -> GetExclude())
	{
		if (operationToAppend -> GetOptionsDelete())	arguments.append("--delete-excluded");
		if ( (operationToAppend -> GetExcludeFromFile()) && !(operationToAppend -> GetExcludeFile()=="") )
			arguments.append("--exclude-from=" + (operationToAppend -> GetExcludeFile()) );
		if (operationToAppend -> GetExcludeTemp())	arguments.append("--exclude=**/*tmp*/");
		if (operationToAppend -> GetExcludeCache()){	arguments.append("--exclude=**/*cache*/");
								arguments.append("--exclude=**/*Cache*/");}
		if (operationToAppend -> GetExcludeBackup())	arguments.append("--exclude=**~");
		if (operationToAppend -> GetExcludeMount()){	arguments.append("--exclude=/mnt/*/**");
								arguments.append("--exclude=/media/*/**");}
		if (operationToAppend -> GetExcludeLostFound())	arguments.append("--exclude=**/lost+found*/");
		if (operationToAppend -> GetExcludeSystem()){	arguments.append("--exclude=/var/**");
								arguments.append("--exclude=/proc/**");
								arguments.append("--exclude=/dev/**");
								arguments.append("--exclude=/sys/**");}
		if (operationToAppend -> GetExcludeTrash()){	arguments.append("--exclude=**/*Trash*/");
								arguments.append("--exclude=**/*trash*/");}
	
		//also read the custom exclude list
		count =0;
		while ( count < (operationToAppend -> GetExcludeListSize()) )
		{
			arguments.append("--exclude=" + (operationToAppend -> GetExcludeListItem(count)) );
			count++;
		}
	}

	//set temp strings sourceString & destString accordingly if groupbox "remote" is checked--------------------------------------------
	//also add -e ssh if "ssh" is checked & --password-file=FILE
	QString sourceString, destString, remoteHost = "";	//temp strings
	if (operationToAppend -> GetRemote())
	{
		if (operationToAppend -> GetRemoteUser() != "")	//append remote user@ if applicable to temp string
			remoteHost.append(operationToAppend -> GetRemoteUser() + "@");

		if (operationToAppend -> GetRemoteModule())	//append remote host: (or :: if it's a module) to temp string
		{
			remoteHost.append(operationToAppend -> GetRemoteHost() + "::");
			//add --password-file=FILE if password file lineEdit is not empty
			if (operationToAppend -> GetRemotePassword() != "")
				arguments.append("--password-file=" + (operationToAppend -> GetRemotePassword()) );
		}
		else
			remoteHost.append(operationToAppend -> GetRemoteHost() + ":");

		if (operationToAppend -> GetRemoteDestination())	//set temp source & destination strings
		{	remoteHost.append(operationToAppend -> GetDestination());
			destString = remoteHost;
			sourceString 	= operationToAppend -> GetSource();}
		else {	remoteHost.append(operationToAppend -> GetSource());
			sourceString = remoteHost;
			destString 	= operationToAppend -> GetDestination();}

		//add argument for ssh if the checkbox is checked & ssh keyfile
		if (operationToAppend -> GetRemoteSSH())
		{
			if (operationToAppend -> GetRemoteSSHPassword() != "")
				//arguments.append("-e \"ssh -i " + uiM.lineEdit_sshPassword -> text() + "\"");
				if (operationToAppend -> GetRemoteSSHPort() != 0)
					arguments.append("-e ssh -i " + operationToAppend -> GetRemoteSSHPassword() +
							" -p " + countStr.setNum(operationToAppend -> GetRemoteSSHPort()) );
				else
					arguments.append("-e ssh -i " + operationToAppend -> GetRemoteSSHPassword());
			else
				if (operationToAppend -> GetRemoteSSHPort() != 0)
					arguments.append("-e ssh -p " + countStr.setNum(operationToAppend -> GetRemoteSSHPort()) );
				else
					arguments.append("-e ssh");
		}
	}	
	else		//Operate locally----------------------------------------------------------------------------------------
	{
		sourceString 	= operationToAppend -> GetSource();
		destString 	= operationToAppend -> GetDestination();
	}

	//add snapshot arguments - do not perform this if the function is called when "validate" button pressed
	//perform this if more than 1 snapshot is already made
	if  ( (operationToAppendCurrentSnaps > 1) && (!validation) )
	{
		//fix source and dest for snapshots directory
		QString snapSource=sourceString;	QString snapDest=destString;	//temp variables
		if (snapDest.contains(":"))	// this is normal for a remote directory
				snapDest = snapDest.right(snapDest.size()-snapDest.lastIndexOf(":")-1);	//this is the remote dest dir without the remote pc
		if (!snapSource.endsWith("/"))	// this means task is of type "backup dir by name"
		{
			QString sourceLast = snapSource;
			if (sourceLast.contains(":"))	// this is normal for a remote directory
				sourceLast = sourceLast.right(snapSource.size()-sourceLast.lastIndexOf(":")-1);	//this is the remote source dir without the remote pc
			if (snapSource.contains("/"))	// this is normal for a directory unless it is remote
				sourceLast = sourceLast.right(snapSource.size()-sourceLast.lastIndexOf("/")-1);	//this is the lowest dir of source
			snapSource.append("/");
			
			snapDest.append(sourceLast + "/");
		}
		QString prevSnapDateTime = operationToAppend -> GetSnapshotsListItem(operationToAppendCurrentSnaps-2);
		
		QString prevSnapDir = snapDest + snapDefaultDir + prevSnapDateTime + "/";	// This is where thy deleted files (== previous snapshot) will go
		
		// add arguments to backup files to be be deleted inside the snapshot direcotry
		arguments.append("--backup");
		arguments.append("--backup-dir=" + prevSnapDir);

		// protect the snapshots directories from being deleted. Do nothing for older snapshots dirs
/*		count = 0;
		while (count < operationToAppendCurrentSnaps)
		{
			arguments.append("--filter=protect " + snapDefaultDir + operationToAppend -> GetSnapshotsListItem(count) +"/");
			count++;
		}
*/
		arguments.append("--filter=protect " + snapDefaultDir);
	}
	
	// keep snapshot changes files only for backup task types, not sync
	if ( (!validation) && (operationToAppendMaxSnaps > 1) )
	{
		//Define a file to log new files transfered so that to exclude these when restoring previous snapshots
		arguments.append("--log-file=" + snapchangesfilename);
		arguments.append("--log-file-format=" + snapChangesString);
	}
	
	//set source & destination according to sourceString & destString ---------------------------------------------------------
	arguments.append(sourceString);
	arguments.append(destString);

	return arguments;
}

// logFileUpdate =====================================================================================================================================
// Updates the current logfile with some string
QString logFileUpdate(QString appendTYPE, QString appendTHIS, int currentPrePost)
{
	QTextStream out(&logfile);
	QString dirA = Operation[currentOperation] -> GetSource();
	QString dirB = Operation[currentOperation] -> GetDestination();
	
	if (appendTYPE == "pre-starting")
		appendTHIS = "\n<font color=magenta>" +
			QObject::tr("pre-task execution of command","Full phrase: pre-task execution of command: <COMMAND> starting")+
			"	: <b>" + Operation[currentOperation] -> GetExecuteBeforeListItem(currentPrePost) +
			"</b>, "+ QObject::tr("starting","Full phrase: pre-task execution of command: <COMMAND> starting")+"</font>\n";

	if (appendTYPE == "post-starting")
		appendTHIS = "\n<font color=magenta>" +
			QObject::tr("post-task execution of command","Full phrase: post-task execution of command: <COMMAND> starting")+
			"	: <b>" + Operation[currentOperation] -> GetExecuteAfterListItem(currentPrePost) +
			"</b>, "+QObject::tr("starting","Full phrase: post-task execution of command: <COMMAND> starting")+"</font>\n";
	
	if (appendTYPE == "rsync-starting-backup")
		appendTHIS = "\n=====================================<br><font color=magenta>" +
				QObject::tr("execution of task","Full phrase: execution of task: <TASKNAME> starting")+
				"	: <b>" + Operation[currentOperation] -> GetName() +
				"</b>, "+QObject::tr("starting","Full phrase: execution of task: <TASKNAME> starting")+"</font><br>"+
				QObject::tr("Source","Full phrase: source: <SOURCE_NAME")+"	: <b><font color=blue>" + dirA +
				"</font></b><br>"+QObject::tr("Destination","Full phrase: Destination: <DEST_NAME")+
				"	: <b><font color=blue>" + dirB + "</font></b>\n";

	if (appendTYPE == "rsync-starting-syncAB")
		appendTHIS = "\n=====================================<br><font color=magenta>" +
				QObject::tr("execution of 1st part of task","Full phrase: execution of 1st part of task: <TASKNAME> starting")+
				"	: <b>" + Operation[currentOperation] -> GetName() +
				"</b>, "+QObject::tr("starting","Full phrase: execution of 1st part of task: <TASKNAME> starting")+"</font><br>"+
				QObject::tr("Syncing","Full phrase: Syncing <DIR-A> to <DIR-B>")+"	: <b><font color=blue>" + dirA +
				"</font></b><br>"+QObject::tr("to","Full phrase: Syncing <DIR-A> to <DIR-B>")+
				"	: <b><font color=blue>" + dirB + "</font></b>\n";
		
	if (appendTYPE == "rsync-starting-syncBA")
		appendTHIS = "\n=====================================<br><font color=magenta>" +
				QObject::tr("execution of 2nd part of task","Full phrase: execution of 2nd part of task: <TASKNAME> starting")+
				"	: <b>" + Operation[currentOperation] -> GetName() +
				"</b>, "+QObject::tr("starting","Full phrase: execution of 2nd part of task: <TASKNAME> starting")+"</font><br>"+
				QObject::tr("Syncing","Full phrase: Syncing <DIR-B> to <DIR-A>")+"	: <b><font color=blue>" + dirB +
				"</font></b><br>"+QObject::tr("to","Full phrase: Syncing <DIR-B> to <DIR-A>")+"	: <b><font color=blue>" + dirA + "</font></b>\n";

	if (appendTYPE == "pre-finished")
		appendTHIS = "\n<font color=magenta>" +
			QObject::tr("pre-task execution of command","Full phrase: pre-task execution of COMMAND: <COMMANDNAME> finished")+
			"	: <b>" + Operation[currentOperation] -> GetExecuteBeforeListItem(currentPrePost) +
			"</b>, "+QObject::tr("finished","Full phrase: pre-task execution of COMMAND: <COMMANDNAME> finished")+"</font>\n";
	
	if (appendTYPE == "post-finished")
		appendTHIS = "\n<font color=magenta>" +
			QObject::tr("post-task execution of command","Full phrase: post-task execution of COMMAND: <COMMANDNAME> finished")+
			"	: <b>" + Operation[currentOperation] -> GetExecuteAfterListItem(currentPrePost) +
			"</b>, "+QObject::tr("finished","Full phrase: post-task execution of COMMAND: <COMMANDNAME> finished")+"</font><br>";

	if (appendTYPE == "rsync-finished-sync1")
		appendTHIS = "\n<font color=magenta>" +
				QObject::tr("execution of 1st part of task","Full phrase: execution of 1st part of task: <TASKNAME> finished")+
				"	: <b>" + Operation[currentOperation] -> GetName() +
				"</b>, "+QObject::tr("finished","Full phrase: execution of 1st part of task: <TASKNAME> finished") +
				"</font><br>=====================================<br>";

	if (appendTYPE == "rsync-finished")
		appendTHIS = "\n<font color=magenta>" +
				QObject::tr("execution of task","Full phrase: execution of task: <TASKNAME> finished")+
				"	: <b>" + Operation[currentOperation] -> GetName() +
				"</b>, "+QObject::tr("finished","Full phrase: execution of task: <TASKNAME> finished") +
				"</font>\n=====================================<br>";
				
	if (appendTYPE == "pre-task-exited-with-error")
		appendTHIS = "\n<font color=magenta>" +
				QObject::tr("execution of task","Full phrase: execution of task: <TASKNAME> finished because of pre-task command execution error")+
				"	: <b>" + Operation[currentOperation] -> GetName() +
				"</b>, "+QObject::tr("finished because of pre-task command execution error",
						     "Full phrase: execution of task: <TASKNAME> finished because of pre-task command execution error") +
				"</font>\n=====================================<br>";

	if (appendTYPE == "rsync-standard")
		appendTHIS = appendTHIS + "\n";

	if (appendTYPE == "rsync-error")
		appendTHIS = "<a name=\"error" + countStr.setNum(errorsFound) + "\"></a><font color=red>" + appendTHIS + "</font>\n";

	if (writeToLog)
	{
		if (console)	//this is used for console compatibility with utf-8
			appendTHIS = QString(appendTHIS.toUtf8());
		out << appendTHIS;
	}

	return appendTHIS;
}
// end of global.cpp ---------------------------------------------------------------------------

