/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

/**
 *   This file contains the class to deal with automatic dynamic memory
 *   allocation / desallocation.
 */

#ifndef TEPDISHAREDPTR_HPP
  #define TEPDISHAREDPTR_HPP

  template< class T >
  class TePDISharedPtr{
    public :
      /**
       * Default Constructor.
       *
       * @param pointer A pointer the the active object.
       * @param static_assignment A flag indication for a static
       * assignment (the pointed object will not be destroyed at the
       * end ).
       */
      TePDISharedPtr( T* pointer = 0, bool static_assignment = false );

      /**
       * Alternative Constructor.
       *
       * @param external External object reference.
       */
      TePDISharedPtr( const TePDISharedPtr& external );

      /**
       * Default Destructor
       */
      ~TePDISharedPtr();

      /**
       * Verifies if the current pointer points to an active object.
       *
       * @return true if active, false if not.
       */
      bool isActive() const;

      /**
       * Reset the active instance the the new pointer.
       *
       * @param pointer A pointer the the active object.
       * @param static_assignment A flag indication for a static
       * assignment (the pointed object will not be destroyed at the
       * end ).
       */
      void reset( T* pointer = 0, bool static_assignment = false );

      /**
       * Operator = overload.
       *
       * @param external External instance reference.
       */
      TePDISharedPtr< T >& operator=( const TePDISharedPtr< T >& external );

      /**
       * Operator * overload.
       *
       * @return The internal instance reference.
       */
      T& operator*();

      /**
       * Operator -> overload.
       *
       * @return The internal instance pointer.
       */
      T* operator->();

      /**
       * Operator == overload.
       *
       * @return true if the current pointer and the external pointer
       * are pointing to the same object.
       */
      bool operator==( const TePDISharedPtr& external );

      /**
       * Operator != overload.
       *
       * @return true if the current pointer and the external pointer
       * are not pointing to the same object.
       */
      bool operator!=( const TePDISharedPtr& external );

      /**
       * A Naked Pointer to the encapsulated object.
       *
       * @return The internal instance pointer.
       */
      T* NakedPointer();

    protected :

      /**
       * A pointer to the current number of active users of this pointer.
       */
      mutable unsigned long int* counter_;

      /**
       * A pointer to the current object encapsulated by this
       * shared pointer.
       */
      mutable T* reference_;

      /**
       * A flag indicating if this shared pointer was created by a
       * dynamic assignment ( false value ) or by a static
       * assignment( true value - the encapsulated object will not
       * be destroyed at the end.
       */
      mutable bool static_assignment_;
  };


#endif //TEPDISHAREDPTR_HPP
