/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeVisual.h
    This file deals with the visual presentation characteristics of geometries in TerraLib
*/
#ifndef  __TERRALIB_INTERNAL_VISUAL_H
#define  __TERRALIB_INTERNAL_VISUAL_H
#include "TeDataTypes.h"

class TeRasterTransform;

#include <string>
#include <map>

using namespace std;

//! Supports a color definition
struct TeColor
{
	//! Red component
	int red_;

	//! Green component 
	int green_; 	
	
	//! Blue component
	int blue_;	
	
	//! Color name
	string name_;
	
	//! Empty constructor
	TeColor () : red_(0),green_(0),blue_(0), name_("") {}

	//! Constructor with parameters
	TeColor (int r, int g, int b, const string& name="") : red_(r),green_(g),blue_(b), name_(name) {}

	//! Set parameters of colors
	void init (int r, int g, int b, const string& name="") {red_=r,green_=g,blue_=b;name_=name; }


	//! Returns TRUE if color1 is equal to color2 or FALSE if they are different.
	bool operator== (const TeColor& color)
	{
		if(red_==color.red_ && green_==color.green_ && blue_==color.blue_)
			return true;
		return false;
	}

	//! Assignment operator
	TeColor& operator= ( const TeColor& color )
	{
		if ( this != &color )
		{	
			red_ = color.red_;
			green_ = color.green_;
			blue_ = color.blue_;
			name_ = color.name_;
		}
		return *this;
	}
};


//! This class is used to store the presentation characteristics of vector geometries
class TeVisual
{
private :

	string		libName_;			//! Symbol library name used to geometry filling
	int			symbId_;			//! Symbol id that define the style for the geometry filling
	TeColor		color_;				//! Color used for geometry filling (line, point and text color)
	int			transparency_;		//! Percentage of transparency for the geometry filling
	int			width_;				//! Width for line geometry
	
	string		contourLibName_;		//! Symbol library name used to geometry outline
	int			contourSymbId_;			//! Symbol id that define the style for the geometry outline
	TeColor		contourColor_;			//! Contour color for the geometry outline
	int			contourTransparency_;	//! Percentage of transparency for the geometry outline
	int			contourWidth_;			//! Width for the geometry outline

	int			sizeValue_;
	int			ptAngle_;
	
	// Text attributes
	string		family_;			//! Text family
	bool		bold_;				//! Flag to indicate if text is bold
	bool		italic_;			//! Flag to indicate if text is italic
	double		alignmentVert_;		//! Position of vertical alignment
	double		alignmentHoriz_;	//! Position of horizontal alignmentool		fixedSize_;				//! Flag to indicate that element has a fixed size, dont follow zoom operations	
	int			tabSize_;			//! Number of spaces defined in a tab character
	int			lineSpace_;			//! Spacing between lines
	bool		fixedSize_;
	
	char*	pixmap_;

public :

	//! Empty constructor
	TeVisual(TeGeomRep rep = TePOLYGONS) : 
		libName_ (""),
		symbId_ (TePolyTypeFill),
		color_ (255,0,0),
	    transparency_ (0),
		width_ (0),
	   	contourLibName_ (""),
		contourSymbId_ (TeLnTypeContinuous),
		contourColor_(0,255,255),
		contourTransparency_ (0),
		contourWidth_ (1),
		sizeValue_ (0),
		ptAngle_ (0),
		family_ ("Verdana"),
		bold_ (false),
		italic_(false),
		alignmentVert_(0),
		alignmentHoriz_(0),
		tabSize_(6),
		lineSpace_(1),
		fixedSize_(false),
		pixmap_(0)
		{
			switch (rep)
			{
				case TePOLYGONS:
				case TeCELLS:
					symbId_= TePolyTypeFill;
					contourSymbId_= TeLnTypeContinuous;
					break;
				case TeLINES:
					symbId_= TeLnTypeContinuous;
					width_ = 1;
					break;
				case TePOINTS:
				case TeNODES:
					symbId_= TePtTypeBox;
					sizeValue_ = 3;
					break;
				case TeTEXT:
					sizeValue_= 12;
					break;
				default:
					break;
			}
		}

	//! Destructor
	~TeVisual () 
	{ if (pixmap_) delete pixmap_; }

	//! Copy constructor
	TeVisual& operator=(const TeVisual& vis)
	{
		if ( this == &vis )
			return *this;
		else
		{
			libName_ = vis.libName_;
			symbId_ = vis.symbId_;
			color_ = vis.color_;
			transparency_ = vis.transparency_;
			width_ = vis.width_;

			contourLibName_ = vis.contourLibName_;
			contourSymbId_ = vis.contourSymbId_;
			contourColor_ = vis.contourColor_;
			contourTransparency_ = vis.contourTransparency_;
			contourWidth_ = vis.contourWidth_;
			
			sizeValue_ = vis.sizeValue_;
			ptAngle_ = vis.ptAngle_;
			
			family_ = vis.family_;
			bold_ = vis.bold_;
			italic_ = vis.italic_;
			fixedSize_ = vis.fixedSize_;
		    alignmentVert_ = vis.alignmentVert_;
		    alignmentHoriz_ = vis.alignmentHoriz_;
		    tabSize_ = vis.tabSize_;
		    lineSpace_ = vis.lineSpace_;
			pixmap_ = vis.pixmap_;
		}
		return *this;
	}

	//! Returns TRUE if vis1 is equal to vis2 or FALSE if they are different.
	bool operator== (const TeVisual& vis)
	{
		if( libName_ == vis.libName_ &&
			symbId_ == vis.symbId_ &&
			color_ == vis.color_ &&
			transparency_ == vis.transparency_ &&
			width_ == vis.width_ &&
			contourLibName_ == vis.contourLibName_ &&
			contourSymbId_ == vis.contourSymbId_ &&
			contourColor_ == vis.contourColor_ &&
			contourTransparency_ == vis.contourTransparency_ &&
			contourWidth_ == vis.contourWidth_ &&
			sizeValue_ == vis.sizeValue_ &&
			ptAngle_ == vis.ptAngle_ &&
			family_ == vis.family_ &&
			bold_ == vis.bold_ &&
			italic_ == vis.italic_ &&
			fixedSize_ == vis.fixedSize_ &&
			alignmentVert_ == vis.alignmentVert_ &&
			alignmentHoriz_ == vis.alignmentHoriz_ &&
			tabSize_ == vis.tabSize_ && 
			lineSpace_ == vis.lineSpace_ &&
			pixmap_ == vis.pixmap_)
				return true;
		return false;
	}

	//! Copies visual parameters from another visual
	void setBasic (TeVisual &vis)
	{
		libName_ = vis.libName_;
		symbId_ = vis.symbId_;
		color_ = vis.color_;
		transparency_ = vis.transparency_;
		width_ = vis.width_;

		contourLibName_ = vis.contourLibName_;
		contourSymbId_ = vis.contourSymbId_;
		contourColor_ = vis.contourColor_;
		contourTransparency_ = vis.contourTransparency_;
		contourWidth_ = vis.contourWidth_;
		
		sizeValue_ = vis.sizeValue_;
		ptAngle_ = vis.ptAngle_;
		
		family_ = vis.family_;
		bold_ = vis.bold_;
		italic_ = vis.italic_;
		fixedSize_ = vis.fixedSize_;
		alignmentVert_ = vis.alignmentVert_;
		alignmentHoriz_ = vis.alignmentHoriz_;
		tabSize_ = vis.tabSize_;
		lineSpace_ = vis.lineSpace_;
		pixmap_ = vis.pixmap_;
	}

	//! Sets the color used for geometry filling and line, point and text color
	void color(TeColor& color) 
	{	color_ = color;}

	//! Returns the color used for geometry filling and line, point and text color
	TeColor& color() 
	{ return color_; }

	//! Sets symbol library name used for filling
	void libName (const string& name) { libName_ = name; }

	//! Returns symbol library name used for filling
	string libName () { return libName_; }

	//! Sets symbol library name used for contour
	void contourLibName (const string& name) { contourLibName_ = name; }

	//! Returns symbol library name used for contour
	string contourLibName () { return contourLibName_; }

	//! Sets the style for the filling of geometries
	void style(int s) { symbId_ = s; }

	//! Returns the style for the filling of geometries
	int	style() { return symbId_; }

	//! Sets the percentage of transparency for polygons and cells
	void transparency(int t)
	{ transparency_ = t; }

	//! Returns the percentage of transparency used in polygons and cells
	int transparency()
	{ return transparency_; }

	//! Sets the color for the polygons and cells outlines
	void contourColor(TeColor& color) 
	{ contourColor_ = color;}

	//! Returns the color used in the polygons and cells outlines
	TeColor& contourColor() 
	{ return contourColor_; }

	//! Sets the percentage of transparency for the polygon ans cells outlines
	void contourTransparency(int t) 
	{ contourTransparency_ = t;}

	//! Returns the color used in the polygons and cells outlines
	int	contourTransparency() 
	{ return contourTransparency_; }

	//! Sets the contour width for the polygons and cells outlines
	void contourWidth(int w) { contourWidth_ = w; }

	//! Returns the contour width used in the polygons and cells outlines
	int	contourWidth()  { return contourWidth_; }

	//! Sets the contour style for the polygons and cells outlines
	void contourStyle(int s) { contourSymbId_ = s; }

	//! Returns the contour style used in the polygons and cells outlines
	int	contourStyle() { return contourSymbId_; }

	//! Sets the width
	void width(int w) { width_ = w; }

	//! Returns the width
	int	 width ()  { return width_; }

	//! Sets the point size
	void size (int s)  { sizeValue_ = s; }

	//! Returns the point size
	int	size ()  { return sizeValue_; }

	//! Sets the text family for fonts
	void family (string f) { family_ = f; }

	//! Returns fonts family
	string family () { return family_ ; }

	//! Sets the bold style for texts
	void bold (bool b) { bold_ = b; }

	//! Returns the bold style used in the texts
	bool bold () { return bold_; }

	//! Sets the italic style for texts
	void italic (bool i) { italic_ = i; }

	//! Returns the italic style used in the texts
	bool italic () { return italic_; }

	//!  Sets a flag indicating if the text will have a fixed size
	void fixedSize (bool i) { fixedSize_ = i; }

	//!  Retuns a flag indicating if an text have a fixed size
	bool fixedSize () { return fixedSize_; }

	//!  Sets the vertical alignment of texts
	void alignmentVert(double a) { alignmentVert_ = a; }

	//!  Retuns the vertical alignment of texts
	double alignmentVert() { return alignmentVert_; }

	//!  Sets the horizontal alignment of texts
	void alignmentHoriz(double a) { alignmentHoriz_ = a; }

	//!  Retuns the horizontal alignment of texts
	double alignmentHoriz() { return alignmentHoriz_; }

	//! Sets the number of spaces defined in a tab character
	void tabSize (int s) { tabSize_ = s; }

	//! Returns the number of spaces defined in a tab character
	int	tabSize () { return tabSize_; }

	//! Sets the spacing between lines
	void lineSpace (int s) { lineSpace_ = s; }

	//! Returns the spacing between lines
	int	lineSpace () { return lineSpace_; }

	//! Returns the angle of points and symbols
	int   ptAngle() {return ptAngle_; }

	//! Sets the angle of points and symbols
	void  ptAngle(int p) {ptAngle_ = p;}

	//! Returns  pixmap used in the visual
	char*	pixmap () { return pixmap_; }

	//! Sets the pixmap used in the visual
	void	setPixmap( char* pix ) { pixmap_ = pix; }

};

typedef TeRasterTransform TeRasterVisual;

/*
//! This class is used to store the presentation characteristics of raster geometries
class TeRasterVisual
{
public:

	//! Constructor
	TeRasterVisual():
		mBand_(0),
		gain_(1.),
		offset_(0)
	{}

	int mBand_; 

	double gain_;
	double offset_;

	map<int,int> bandMap;

	vector<unsigned char> lutr_;
	vector<unsigned char> lutg_;
	vector<unsigned char> lutb_;
};
*/

#endif

