/* This file is part of the KDE project
   Copyright (C) 2006-2007 KovoKs <info@kovoks.nl>
   Copyright (C) 2007 Frode M. Døving <frode@lnix.net>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include <kdebug.h>
#include <kmdcodec.h>
#include <ktempfile.h>
#include <kstandarddirs.h>
#include <kmdcodec.h>
#include <krfcdate.h>

#include "../libkmime/kmime_message.h"
using KMime::Message;
using KMime::Content;

#include <qfile.h>
#include <qstylesheet.h>
#include <qtextcodec.h>

#include "imapmanager.h"
#include "global.h"
#include "linklocator.h"
#include "messagedata.h"

namespace Mailody {

MessageData::MessageData( QWidget* parent,  const char* name,
                        int uid, const QString& mb  )
    : QObject(parent, name),
    m_block(false),
    m_delete(false),
    m_uid(uid),
    m_xmsp(0),
    m_mb(mb)
{
    m_datamanager = ImapManager::instance();
    m_headers = m_datamanager->getHeader( mb, uid );
    initialise_needed();
}

MessageData::MessageData( QWidget* parent,  const char* name,
                        int uid, const QString& mb,
                        const QString& headers)
    : QObject(parent, name),
    m_block(false),
    m_delete(false),
    m_initialized_all(false),
    m_headers(headers),
    m_uid(uid),
    m_xmsp(0),
    m_mb(mb)
{
    m_datamanager = ImapManager::instance();
    initialise_needed();
}

MessageData::~MessageData()
{
    //    kdDebug() << k_funcinfo << m_uid << m_mb << endl;
}

void MessageData::initialise_needed()
{
    m_headers = "\n" + m_headers + "\n";
    int i = -1;
    if ((i = m_headers.find("\nmessage-id:",0,false)) != -1)
        m_messageID = m_headers.mid(i+13,(m_headers.find('\n',i+12))-i-13);
    if ((i = m_headers.find("\nin-reply-to:",0,false)) != -1)
        m_inreplyto = m_headers.mid(i+14,(m_headers.find('\n',i+13))-i-14);
    if ((i = m_headers.find("\ndate:",0,false)) != -1)
    {
        m_date.setTime_t(KRFCDate::parseDate(
                m_headers.mid(i+6,(m_headers.find('\n',i+5))-i-6)));
        m_sortDate = m_date;
    }
}

void MessageData::initialise_rest()
{
    m_initialized_all = true;

    if (m_headers.isEmpty())
        return;

    QString to;
    QStringList headerlist = QStringList::split("\n",m_headers);
    QStringList::iterator it;
    for (it = headerlist.begin(); it != headerlist.end(); ++it)
    {
        // Convert it to lower one, so the startsWith is fast, as we can do
        // a case sensitive compare.
        QString line = (*it).lower().stripWhiteSpace();

        if (line.startsWith("size:"))
            m_size = (*it).section(':',1).stripWhiteSpace().toInt();

        else if (line.startsWith("subject"))
        {
            const char* usedCS;
            QString t = (*it).section(':',1).stripWhiteSpace();
            m_subject =
                    KMime::decodeRFC2047String(t.local8Bit(), &usedCS, "", false);
        }
        else if (line.startsWith("from:"))
        {
            QString t = (*it).section(':',1).stripWhiteSpace();
            const char* usedCS;
            t  = KMime::decodeRFC2047String(t.local8Bit(), &usedCS, "", false);
            if (Global::myEmail(t))
                continue;
            m_sender = t;
            parseEmail(m_sender);
        }
        else if (m_sender.isEmpty() && line.startsWith("to:"))
        {
            // If there is no From address use the To address...
            const char* usedCS;
            QString t = (*it).section(':',1).stripWhiteSpace();
            m_sender =
                    KMime::decodeRFC2047String(t.local8Bit(), &usedCS, "", false);
            parseEmail(m_sender);
            m_sender = "To: " + m_sender; // no i18n, 'cause of comparisation
        }
    }
}

QString MessageData::subject()
{
    if (!m_initialized_all)
        initialise_rest();

    return m_subject;
}

QString MessageData::vDate() const
{
    KMime::DateFormatter* i = new KMime::DateFormatter();
    return i->dateString(m_date);
}

void MessageData::requestBody( bodyType i )
{
    m_bodyType = i;
    connect(m_datamanager,
            SIGNAL(message(const QString&, int, const QString&)),
            SLOT(slotBodyReady(const QString&, int, const QString&)));
    m_datamanager->getMessage(m_mb, m_uid);
}

void MessageData::slotBodyReady(const QString& mb, int uid, const QString& body)
{
    disconnect(m_datamanager,
            SIGNAL(message(const QString&, int, const QString&)),
            this, SLOT(slotBodyReady(const QString&, int, const QString&)));

    if (m_mb != mb || uid != m_uid)
        return;

    m_attachments.clear();
    m_datamanager->addFlag(m_mb, m_uid, "\\Seen");

    Message* m = new Message();
    m->setContent(KMime::CRLFtoLF(body.stripWhiteSpace().latin1()));
    m->parse();

    m_subject = m->subject()->asUnicodeString();
    m_sender_email = m->from()->email();
    m_sender_full = m->from()->asUnicodeString();

    KMime::Headers::Base *organization_raw= m->organization();
    if (organization_raw)
        m_company = organization_raw->asUnicodeString();

    KMime::Headers::To *to_raw= m->to();
    if (to_raw)
    {
        m_to = to_raw->asUnicodeString().stripWhiteSpace();

        m_to_addresses = new QStringList();
        if (!to_raw->isEmpty())
            to_raw->displayNames(m_to_addresses);
    }

    KMime::Headers::CC *cc_raw= m->cc();
    if (cc_raw)
    {
        m_cc = cc_raw->asUnicodeString().stripWhiteSpace();

        m_cc_addresses = new QStringList();
        if (!cc_raw->isEmpty())
            cc_raw->displayNames(m_cc_addresses);
    }

    KMime::Headers::Base* ua = m->getHeaderByType("User-Agent");
    if (ua)
        m_userAgent = ua->asUnicodeString();

    KMime::Headers::Base* mi = m->getHeaderByType("Message-id");
    if (mi)
        m_messageID = mi->asUnicodeString();

    KMime::Headers::Base* rt = m->getHeaderByType("Reply-To");
    if (rt)
        m_replyTo = rt->asUnicodeString();

    KMime::Headers::Base* rf = m->getHeaderByType("Resent-From");
    if (rf)
        m_resentFrom = rf->asUnicodeString();

    KMime::Headers::Base* rf2 = m->getHeaderByType("Resent-To");
    if (rf2)
        m_resentTo = rf2->asUnicodeString();

    KMime::Headers::Base* xms = m->getHeaderByType("X-Mailody-Signature");
    if(xms)
        m_xms = xms->asUnicodeString();

    KMime::Headers::Base* xmsp = m->getHeaderByType("X-Mailody-SP");
    if(xmsp)
        m_xmsp = xmsp->asUnicodeString().toInt();

    KMime::Content::List* lijstje = new KMime::Content::List;
    m->attachments( lijstje, true );

    QString plainPart;
    QString htmlPart;

    for(Content *c=lijstje->first(); c; c=lijstje->next())
    {
        QString type = c->contentType()->mimeType().lower();

        if (type == "text/html" && htmlPart.isEmpty() ||
            type == "text/plain" && plainPart.isEmpty() )
        {
            QString text = c->body();
            c->decodedText(text, true, true);

            if (type == "text/html")
                htmlPart = text.stripWhiteSpace();
            else
                plainPart = text.stripWhiteSpace();
        }
        else
        {
            // treat everything else as attachment...

            // try the content-id
            QString filename;
            KMime::Headers::Base *h = c->getHeaderByType("Content-ID");
            if (h)
                filename = h->asUnicodeString();

            if (!filename)
                filename = c->contentDisposition()->filename();

            // try the Content-type name parameter
            if (!filename)
            {
                KMime::Headers::ContentType *h = c->contentType(false);
                if (h)
                    filename = h->name();
            }

            // Ok, I give up...
            if (!filename)
                continue;

            QByteArray result = c->decodedContent();

            // Some stupid mailers...
            if (filename.startsWith("<") && filename.endsWith(">"))
                filename=filename.mid(1, filename.length()-2);

            // Save it
            KTempFile *tempFile =
                    new KTempFile(locateLocal( "tmp", "Mailody-" ));
            tempFile->setAutoDelete( true );
            tempFile->file()->writeBlock( result.data(), result.size() );
            tempFile->file()->close();

            m_attachments[tempFile->name()] = filename;

            // Add it to the mimefactory
            emit addContent(this, "cid:"+filename, tempFile->name());
            tempFile = 0;
        }
    }

    // replace all cid: entries with their filenames.
    QMap<QString,QString>::Iterator ita;
    for (ita = m_attachments.begin(); ita != m_attachments.end(); ++ita)
    {
        // kdDebug() << "cid:"+ita.key() << " -> " << ita.data() << endl;
        htmlPart.replace("cid:"+ita.data(),ita.key());
    }

    // Assign m_body

    using namespace LinkLocatorNS;
    const int flags = LinkLocator::PreserveSpaces |
                    LinkLocator::LinkLocator::HighlightText;
    if (m_bodyType == HTML)
    {
        if (htmlPart.stripWhiteSpace().isEmpty())
        {
            m_body = LinkLocator::convertToHtml(plainPart, flags);
        }
        else
            m_body = htmlPart;

        // when replying prefer plain
        !plainPart.isEmpty() ? m_raw = plainPart : m_raw = htmlPart;
    }
    else if (m_bodyType == Plain)
    {
        if (plainPart.stripWhiteSpace().isEmpty())
            m_body = LinkLocator::convertToHtml(htmlPart, flags);
        else
            m_body = LinkLocator::convertToHtml(plainPart, flags);

        // when replying prefer plain
        !plainPart.isEmpty() ? m_raw = plainPart : m_raw = htmlPart;
    }
    else if (m_bodyType == Source)
    {
        m_body = LinkLocator::convertToHtml( m->encodedContent() , flags);
        m_raw = body;
    }
    m_realraw = body;

    // make the quotation colors.
    m_body = Global::highlightText(m_body);

    emit messageData(this);
}

bool MessageData::isDeleted() const
{
    return m_datamanager->hasFlag(m_mb, m_uid, "\\Deleted");
}

bool MessageData::isNew() const
{
    return !m_datamanager->hasFlag(m_mb, m_uid, "\\Seen");
}

bool MessageData::hasUserTag( int i ) const
{
    return m_datamanager->hasFlag(m_mb, m_uid, "$Label" + QString::number(i));
}

void MessageData::parseEmail(QString& email)
{
    int start=0;
    int end=0;

    // first remove the " so: "toma" <toma@omat.nl>
    // becomes: toma <toma@omat.nl>.
    // that also solves: "" <toma@omat.nl>
    email = email.remove('"');

    // Maybe there is a: toma <toma@omat.nl>
    if ((start=email.find("<"))!=-1 && start > 2)
    {
        email = email.mid(0,start);
        return;
    }

    // Maybe there is a: <toma@omat.nl>
    else if ((start=email.find("<"))!=-1 && (end=email.findRev(">"))!=-1)
    {
        email = email.mid(start+1,end-start-1);
        return;
    }
    // Did not work out...
}

void MessageData::setBlocked( bool block )
{
    // kdDebug() << k_funcinfo << m_uid << m_mb << m_delete << endl;
    m_block=block;
    if (m_delete)
        deleteLater();
}

void MessageData::done()
{
    // kdDebug() << k_funcinfo << m_uid << m_mb << endl;
    if (!m_block)
        deleteLater();
    else
        m_delete = true;
}

}

#include "messagedata.moc"
