/* This file is part of Malaga, a system for Natural Language Analysis.
 * Copyright (C) 1995-1999 Bjoern Beutel
 *
 * Bjoern Beutel
 * Universitaet Erlangen-Nuernberg
 * Abteilung fuer Computerlinguistik
 * Bismarckstrasse 12
 * D-91054 Erlangen
 * e-mail: malaga@linguistik.uni-erlangen.de 
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/* description ==============================================================*/

/* Operations for files and file names. */

/* includes =================================================================*/

#define _XOPEN_SOURCE
#define _XOPEN_SOURCE_EXTENDED 1
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <unistd.h>
#include <sys/mman.h>
#include <pwd.h>
#include "basic.h"

#undef GLOBAL
#define GLOBAL

#include "files.h"

/* constants ================================================================*/

/* Define MAP_FILE if it's not defined in <sys/mman.h>. */
#ifndef MAP_FILE
#define MAP_FILE 0
#endif

/* file operations ==========================================================*/

GLOBAL bool_t file_exists (string_t file_name)
/* Return TRUE iff file <file_name> exists and can be read. */
{
  FILE *stream;

  stream = fopen (file_name, "r");
  if (stream == NULL) return FALSE;
  fclose (stream);
  return TRUE;
}

/*---------------------------------------------------------------------------*/

GLOBAL FILE *open_stream (string_t file_name, string_t stream_mode)
/* Open file <file_name> and create a stream from/to it in mode <stream_mode>.
 * Works like "fopen", but calls "error" if it doesn't work. */
{
  FILE *stream;
 
  stream = fopen (file_name, stream_mode);
  if (stream == NULL)
    error ("can't open \"%s\": %s", file_name, strerror (errno));

  return stream;
} 

/*---------------------------------------------------------------------------*/

GLOBAL void close_stream (FILE **stream_ptr, string_t file_name)
/* Close the stream <*stream_ptr> which is connected to the file <file_name>
 * and set <*stream_ptr> to NULL. Don't do anything if <*stream_ptr> == NULL.
 * Works like "fclose", but calls "error" if <file_name> != NULL and an error
 * occurs during closing. */
{
  FILE *stream = *stream_ptr;

  *stream_ptr = NULL;
  if (stream != NULL && fclose (stream) != 0 && file_name != NULL)
    error ("can't close \"%s\": %s", file_name, strerror (errno));
}

/*---------------------------------------------------------------------------*/

GLOBAL void write_vector (void *ptr, 
			  int_t item_size, 
			  int_t items, 
			  FILE *stream, 
			  string_t file_name)
/* Write <items> items, of size <item_size> each, stored at *<ptr>,
 * to <stream>, which is connected to file <file_name>.
 * Works like "fwrite", but calls "error" if it doesn't work. */
{
  if (fwrite (ptr, item_size, items, stream) < items)
    error ("can't write to \"%s\": %s", file_name, strerror (errno));
}

/*---------------------------------------------------------------------------*/

GLOBAL void read_vector (void *ptr, 
			 int_t item_size, 
			 int_t items, 
			 FILE *stream, 
			 string_t file_name)
/* Read <items> items, of size <item_size> each, from <stream>,
 * which is connected to file <file_name>, and store them at *<ptr>.
 * Works like "fread", but calls "error" if it doesn't work. */
{
  if (fread (ptr, item_size, items, stream) < items)
    error ("can't read from \"%s\": %s", file_name, strerror (errno));
}

/*---------------------------------------------------------------------------*/

GLOBAL void *read_new_vector (int_t item_size, 
			      int_t items, 
			      FILE *stream, 
			      string_t file_name)
/* Read <items> items, of size <items_size> each, from <stream>,
 * which is connected to file <file_name>, into allocated memory block,
 * and return a pointer to that block.
 * The block must be freed with "free" after use. */
{
  void *block;

  block = new_vector (item_size, items);
  read_vector (block, item_size, items, stream, file_name);
  return block;
}

/*---------------------------------------------------------------------------*/

GLOBAL void map_file (string_t file_name, void **address, int_t *length)
/* Map file "file_name" into the memory. It will be available in the 
 * memory region starting at *<address> and will occupy <length> bytes.
 * After usage, return the memory region via "unmap_file". */
{
  int file_descriptor;

  /* Get a file descriptor. */
  file_descriptor = open (file_name, O_RDONLY);
  if (file_descriptor == -1)
    error ("can't open \"%s\": %s", file_name, strerror (errno)); 

  /* Get file length. */
  *length = lseek (file_descriptor, 0, SEEK_END);
  if (*length == -1)
    error ("can't get length of \"%s\": %s", file_name, strerror (errno));

  *address = mmap (NULL, *length, PROT_READ, 
		   MAP_FILE | MAP_SHARED,
		   file_descriptor, 0);
  if (*address == (void *) -1)
    error ("can't read \"%s\": %s", file_name, strerror (errno));

  /* The file descriptor is no longer needed. */
  close (file_descriptor);
}

/*---------------------------------------------------------------------------*/

GLOBAL void unmap_file (void **address, int_t length)
/* Return the memory region that has been allocated by "map_file".
 * The region starts at <*address> and occupies <length> bytes. */
{
  munmap (*address, length);
  *address = NULL;
}

/* file name operations =====================================================*/

GLOBAL string_t name_in_path (string_t path_name)
/* Return the file name in <path_name>, i.e. the name after the last "/". */
{
  string_t file_name;

  file_name = strrchr (path_name, '/');
  if (file_name != NULL)
    file_name++;
  else
    file_name = path_name;

  return file_name;
}

/*---------------------------------------------------------------------------*/

LOCAL string_t get_env (string_t name)
/* Get the content of the environment variable <name>.
 * Emit an error if it is not defined. */
{
  string_t content = getenv (name);
      
  if (content == NULL)
    error ("can't read environment variable \"%s\"", name);

  return content;
}

/*---------------------------------------------------------------------------*/

LOCAL void tidy_path (string_t path)
/* Remove all superfluous "..", "." and "/" in <path>.
 * <path> must be absolute. */
{
  string_t src_ptr, dest_ptr;

  dest_ptr = src_ptr = path;
  while (*src_ptr != EOS) 
  {
    while (*src_ptr == '/')
      src_ptr++;
    
    *dest_ptr++ = '/';
    
    if (src_ptr[0] == '.' && src_ptr[1] == '.' 
	&& (src_ptr[2] == '/' || src_ptr[2] == EOS)) 
    {
      /* Walk up only if we are not on root level. */
      src_ptr += 2;
      if (dest_ptr > path + 1)
	dest_ptr -= 2;
      
      while (*dest_ptr != '/')
	dest_ptr--;
      
    } 
    else if (src_ptr[0] == '.' && (src_ptr[1] == '/' || src_ptr[1] == EOS)) 
    {
      src_ptr++;
      dest_ptr--;
    } 
    else 
    {
      while (*src_ptr != '/' && *src_ptr != EOS)
	*dest_ptr++ = *src_ptr++;
    }
  }
  
  if (dest_ptr > path + 1 && dest_ptr[-1] == '/')
    dest_ptr--;
  
  *dest_ptr = EOS;
}


/*---------------------------------------------------------------------------*/

GLOBAL string_t absolute_path (string_t src_path, string_t relative_to)
/* Return the absolute path name which is equivalent to <src_path>.
 * If <src_path> starts with "~", it's replaced by the home directory of the
 * user whose login name is following (current user if no login name).
 * If the file name starts with "$<x>/", then "$<x>" is replaced by
 * the content of the environment variable <x>.
 * If <relative_to> is not NULL, <src_name> is relative to that path name.
 * <relative_to> must be an absolute path name (a directory or a file).
 * The returned path must be freed after use. */
{
  text_t path = new_text ();
  string_t src_path_ptr, dest_path;

  /* Put a home directory in front. */
  src_path_ptr = src_path;
  if (*src_path_ptr == '~') /* Put a users home directory in front. */
  {
    string_t login_ptr;
    
    src_path_ptr++;
    login_ptr = src_path_ptr;
    while (*src_path_ptr != '/' && *src_path_ptr != EOS)
      src_path_ptr++;
    
    if (src_path_ptr == login_ptr) /* Put the home of current user in front. */
      add_to_text (path, get_env ("HOME"));
    else /* Put home directory of user <login> in front. */
    {
      string_t login = new_string (login_ptr, src_path_ptr);
      struct passwd *password = getpwnam (login);
      
      if (password == NULL)
	error ("can't find user \"%s\"", login);
      
      add_to_text (path, password->pw_dir);
      free_mem (&login);
    }
  } 
  else if (*src_path_ptr == '$') /* Put variable content in front. */
  {
    string_t variable_ptr;
    string_t variable;
    
    src_path_ptr++;
    variable_ptr = src_path_ptr;
    while (*src_path_ptr != '/' && *src_path_ptr != EOS)
      src_path_ptr++;
    
    variable = new_string (variable_ptr, src_path_ptr);
    add_to_text (path, get_env (variable));
    free_mem (&variable);
  }
  else if (*src_path_ptr != '/') 
  {
    if (relative_to != NULL) /* Put <relative_to> ahead (strip last name). */
    {
      string_t relative_end, relative_dir;

      relative_end = relative_to + strlen (relative_to);
      while (relative_end[-1] != '/') 
	relative_end--;
      relative_dir = new_string (relative_to, relative_end);
      add_to_text (path, relative_dir);
      free_mem (&relative_dir);
    } 
    else /* Put current directory in front. */
    {
      char current_dir[200];

      getcwd (current_dir, 200);
      add_to_text (path, current_dir);
    }
  }
  
  /* Copy rest of <dest_path>, clean it up and return it. */
  add_to_text (path, "/");
  add_to_text (path, src_path_ptr);

  dest_path = text_to_string (&path);
  if (*dest_path != '/')
    error ("path \"%s\" must be absolute", src_path);

  tidy_path (dest_path);
  return dest_path;
}

/*---------------------------------------------------------------------------*/

GLOBAL string_t replace_extension (string_t file_name, string_t extension)
/* Replace extension of <file_name> by <extension> (append if it has none).
 * The new string must be free after use. */
{
  string_t old_extension, new_name;
  int_t base_length; /* length of the base name including the "." */

  /* Find base length of file_name. */
  old_extension = strrchr (file_name, '.');
  if (old_extension == NULL)
    base_length = strlen (file_name);
  else
    base_length = old_extension - file_name;

  /* Allocate new name. */
  new_name = new_vector (sizeof (char), 
			 base_length + 1 + strlen (extension) + 1);
  strncpy (new_name, file_name, base_length);
  new_name[base_length] = '.';
  strcpy (new_name + base_length + 1, extension);

  return new_name;
}

/*---------------------------------------------------------------------------*/

GLOBAL bool_t has_extension (string_t file_name, string_t extension)
/* Test if <file_name> has extension <extension>. */
{
  string_t ext; /* the real extension of <file_name> (including ".") */

  ext = strrchr (file_name, '.');
  return (ext != NULL && strcmp (ext + 1, extension) == 0);
}

/*---------------------------------------------------------------------------*/

GLOBAL void set_file_name (string_t *file_name_ptr, 
			   string_t file_name, 
			   string_t extension)
/* Set *<file_name_ptr> to <file_name>, but with extension <extension>.
 * Print an error if *<file_name_ptr> is already set.
 * If <extension> == NULL, the file name is not modified.
 * The created file name must be freed with "free". */
{
  if (*file_name_ptr != NULL) 
    error ("file \"%s\" is redundant", file_name);

  if (extension != NULL)
    *file_name_ptr = replace_extension (file_name, extension);
  else
    *file_name_ptr = new_string (file_name, NULL);
}

/* end of file ==============================================================*/
