/* matchbox - a lightweight window manager

   Copyright 2002 Matthew Allum

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include "theme.h"

Theme* theme_new(wm *w, char *theme_conf)
{
    Theme *t;
    char *default_theme_conf = DEFAULTTHEME ;

    if ( (t = malloc(sizeof(Theme))) == NULL) err("err out of memory");
    
    if (theme_conf == NULL) theme_conf = default_theme_conf;
    XChangeProperty(w->dpy, w->root, w->mb_theme, XA_STRING, 8,
		    PropModeReplace, theme_conf, strlen(theme_conf));
    theme_load(w, t, theme_conf);
    return t;
}

void
theme_switch(Wm *w, Theme *t, char *new_theme_conf)
{
   Client *p;
   int orig_border = WBW(w->head_client);
   
   theme_load(w, t, new_theme_conf);
   t->win_border_width = orig_border;
   START_CLIENT_LOOP(w,p);
   if (p->type != docked)
      {
	 XSetWindowBorder(w->dpy, p->window, (t->bg_col).pixel);
	 XSetWindowBorder(w->dpy, p->frame, (t->bg_col).pixel);
	 /* -- TODO fix border size change on resize --
         XSetWindowBorderWidth(w->dpy, p->frame, WBW(p));
	 if (p->type == toolbar && client_get_state(p) == IconicState)
	 {
	    ;
	 } else {
	    p->configure(p);
	    p->move_resize(p);
	 }

	 client_deliver_config(p);
	 */
	 p->redraw(p,False);
      }
   END_CLIENT_LOOP(w,p);

}

int
theme_load(Wm *w, Theme *t, char *theme_conf)
{
   XGCValues gv, dummy_gv;
   Cursor curs;

#ifdef USE_XFT    
   XRenderColor colortmp;
   char *fontname = "mono";
   char *tbfontname = "mono";
#else
   char *fontname = "fixed";
   char *tbfontname = "fixed";
#endif
   
   dbg("opening %s\n", theme_conf);
   rc_load(theme_conf);

   if (rc_get("font:") != NULL) fontname = rc_get("font:");
   if (rc_get("toolbar.font:") != NULL) tbfontname = rc_get("toolbar.font:");

#ifdef USE_XFT    
    if (rc_get("xftfont:") != NULL) fontname = rc_get("xftfont:");
    if (rc_get("toolbar.xftfont:") != NULL)
       tbfontname = rc_get("toolbar.xftfont:");
    
    t->xftfont = XftFontOpenName(w->dpy, 0, fontname);
    if (!t->xftfont)
    {
       printf("xft font %s not found, trying mono font\n", fontname);
       if ((t->xftfont = XftFontOpenName(w->dpy, 0, "mono")) == NULL)
       {
	  printf("Mono font not found, giving up....\n"); exit(1);
       }
    }
    t->toolbar_xftfont = XftFontOpenName(w->dpy, 0, tbfontname);
    if (!t->toolbar_xftfont)
    {
       printf("xft font %s not found, trying mon font\n", tbfontname );
       if ((t->toolbar_xftfont = XftFontOpenName(w->dpy, 0, "mono")) == NULL)
       {
	  printf("Mono font not found, giving up....\n"); exit(1);
       }
    }
#else
    if (!(t->font = XLoadQueryFont(w->dpy, fontname)))
    {
       printf("font %s not found, trying fixed\n", fontname);
       if (!(t->font = XLoadQueryFont(w->dpy, "fixed")))
       {
	  printf("fixed font not found, giving up...\n");
	  exit(1);
       }
    }


    if (!(t->toolbar_font = XLoadQueryFont(w->dpy, tbfontname)))
    {
       printf("font %s not found, trying fixed\n", tbfontname);
       if (!(t->toolbar_font = XLoadQueryFont(w->dpy, "fixed")))
       {
	  printf("fixed font not found, giving up...\n");
	  exit(1);
       }
    }
#endif
    
    /* New */
    t->win_border_width = rc_get("win.border:") ? atoi(rc_get("win.border:")) : 2;
    t->padding          = rc_get("win.padding:")? atoi(rc_get("win.padding:")) : 2;;
    t->bevel = (rc_get("win.bevel:") && (strcmp(rc_get("win.bevel:"),"YES") == 0)) ? 1 : 0;
    //t->type = gradient;

    _XColorFromStr(w->dpy, &(t->bg_col), rc_get("bg.color:") ?
		   rc_get("bg.color:") : "black" );
    _XColorFromStr(w->dpy, &(t->text_col), rc_get("text.color:") ?
		   rc_get("text.color:") : "black");
    _XColorFromStr(w->dpy, &(t->fg_col), rc_get("fg.color:") ?
		   rc_get("fg.color:") : "white");
    _XColorFromStr(w->dpy, &(t->fg_start_col), rc_get("fg.start.color:") ?
		   rc_get("fg.start.color:") : "white");
    _XColorFromStr(w->dpy, &(t->fg_end_col), rc_get("fg.end.color:") ?
		   rc_get("fg.end.color:") : "white");

    /* text_gc, toolbar_text_gc, bg_gc, fg_gc, effects_gc; */

    gv.graphics_exposures = FALSE;
    gv.function   = GXcopy;
    gv.foreground = (t->text_col).pixel;
#ifndef USE_XFT
    gv.font       = t->font->fid;
#endif
    t->text_gc = XCreateGC(w->dpy, w->root,
		     GCGraphicsExposures|GCFunction|GCForeground|GCFont, &gv);
#ifndef USE_XFT
    gv.font = t->toolbar_font->fid;
#endif
    t->toolbar_text_gc = XCreateGC(w->dpy, w->root,
		     GCGraphicsExposures|GCFunction|GCForeground|GCFont, &gv);

    gv.foreground = (t->bg_col).pixel;
    gv.line_width = t->win_border_width;
    t->bg_gc = XCreateGC(w->dpy, w->root,
		GCGraphicsExposures|GCFunction|GCForeground|GCLineWidth, &gv);

    gv.foreground = (t->fg_col).pixel; /* line width ? */
    t->fg_gc = XCreateGC(w->dpy, w->root,
		    GCGraphicsExposures|GCFunction|GCForeground, &gv);

    gv.foreground  = (t->fg_start_col).pixel; 
    t->effects_gc = XCreateGC(w->dpy, w->root,
		    GCGraphicsExposures|GCFunction|GCForeground, &gv);

    t->mask_gc = XCreateGC(w->dpy, w->root, 0, &dummy_gv);

    gv.function = GXinvert;
    gv.subwindow_mode = IncludeInferiors;
    t->invert_gc = XCreateGC(w->dpy, w->root, GCFunction|GCSubwindowMode|GCLineWidth|GCFont, &gv);

#ifdef USE_XFT
    colortmp.red   = (t->text_col).red;
    colortmp.green = (t->text_col).green;
    colortmp.blue  = (t->text_col).blue;
    colortmp.alpha = 0xffff;
    XftColorAllocValue(w->dpy,
		       DefaultVisual(w->dpy, DefaultScreen(w->dpy)), 
		       DefaultColormap(w->dpy,DefaultScreen(w->dpy)),
		       &colortmp,
		       &t->xft_fg);

#endif
    
    curs = XCreateFontCursor(w->dpy, XC_right_ptr);
    XDefineCursor(w->dpy, w->root, curs);
    
    t->tile.width = 0; t->tile.height = 0;
    
    if (rc_get("xpm.tile:"))
       theme_load_pxm( w, rc_get("xpm.tile:"), &(t->tile) );
    else
    {
       //  memset(&(t->tile), 0, sizeof(Pxm));
       t->tile.width = 0; t->tile.height = 0;
    }
    theme_load_pxm(w, rc_get("xpm.prev:") ? rc_get("xpm.prev:") : XPM_PREV,
		   &t->buttons[BUTTON_PREV] );
    theme_load_pxm(w, rc_get("xpm.next:") ? rc_get("xpm.next:") : XPM_NEXT,
		   &t->buttons[BUTTON_NEXT] );
    theme_load_pxm(w, rc_get("xpm.close:") ? rc_get("xpm.close:") : XPM_CLOSE,
		   &t->buttons[BUTTON_CLOSE] );
    theme_load_pxm(w, rc_get("xpm.toolbar:")
		   ? rc_get("xpm.toolbar:") : XPM_TOOLBAR,
		   &t->buttons[BUTTON_TOOLBAR] );
    theme_load_pxm(w, rc_get("xpm.toolbar.close:")
		   ? rc_get("xpm.toolbar.close:") : XPM_TOOLBAR_CLOSE,
		   &t->buttons[BUTTON_TOOLBAR_CLOSE] );
    theme_load_pxm(w, rc_get("xpm.select:")
		   ? rc_get("xpm.select:") : XPM_MENU_SELECT,
		   &t->buttons[BUTTON_MENU_SELECT] );

    rc_destroy();

    return True;
}


void
theme_load_pixmap(wm *w, char *filename, Pixmap *pxm, Pixmap *mask,
		  int *width, int *height )
{
  XpmAttributes attrib;

  attrib.valuemask = 0; /* TODO: improve */

  if (XpmReadFileToPixmap( w->dpy, w->root, filename,
			   pxm, mask, &attrib) != XpmSuccess )
  {
     fprintf(stderr, "theme: failed loading image '%s'\n", filename);
     exit(1);
  }

  *width  = attrib.width;
  *height = attrib.height;
}

int
theme_load_pxm(Wm *w, char *filename, Pxm *pxm )
{
  XpmAttributes attrib;

  attrib.valuemask = 0; /* TODO: improve */

  if (XpmReadFileToPixmap( w->dpy, w->root, filename,
			   &(pxm->pixmap), &(pxm->mask), &attrib)
      != XpmSuccess )
  {
     fprintf(stderr, "theme: failed loading image '%s'\n", filename);
     exit(1);
  }

  pxm->width  = attrib.width;
  pxm->height = attrib.height;

  return 1;
}


void
theme_render_area(wm *w, Drawable dest, int x, int y,
		  int width, int height )
{
   Theme *t = w->theme;

   XFillRectangle(w->dpy, dest, t->fg_gc, x, y, width, height);

   if ( t->fg_start_col.pixel != t->fg_end_col.pixel )
   {
      XColor tmp_col;
      int i;
      
      float red_step = (t->fg_end_col.red - t->fg_start_col.red) / height;
      float green_step = (t->fg_end_col.green - t->fg_start_col.green)/ height;
      float blue_step  = (t->fg_end_col.blue - t->fg_start_col.blue ) / height;
      
      for(i=height;i;i--)
      {
	 tmp_col.red   = (t->fg_start_col.red + (red_step * (height-i) ));
	 tmp_col.green = (t->fg_start_col.green + ( green_step * (height-i) ));
	 tmp_col.blue  = (t->fg_start_col.blue  + ( blue_step * (height-i) ));
	 
	 XAllocColor(w->dpy, DefaultColormap(w->dpy, DefaultScreen(w->dpy)),
		     &tmp_col);
	 XSetForeground(w->dpy, t->effects_gc, tmp_col.pixel); 
	 XDrawLine(w->dpy, dest,t->effects_gc, x, y+height-i,
		   width, y+height-i);
      }
   }
   
   if (t->tile.width && t->tile.height) {  
      int x,y;
      int ww = t->tile.width;
      int h = t->tile.height;
      for(x=0;x<width;x+=ww)
	 for(y=0;y<height;y+=h)
	    theme_paint_pxm(w, dest, x, y, &t->tile);
	       /*
	    XCopyArea(w->dpy,
		      t->tile.pixmap,
		      dest, t->effects_gc,
		      0, 0, ww, h, x, y);
	       */
   } 

   if (t->bevel)
   {
      XSetForeground(w->dpy, t->effects_gc, t->fg_start_col.pixel); 
      XDrawLine(w->dpy, dest, t->effects_gc, x, y+height-1,
		                         x+width, y+height-1);
      XDrawLine(w->dpy, dest, t->effects_gc, x+width, y,
		                         x+width, y+height-1);
      XSetForeground(w->dpy, t->effects_gc, t->fg_end_col.pixel); 
      XDrawLine(w->dpy, dest, t->effects_gc, x, y,
		                         x+width, y);
      XDrawLine(w->dpy, dest, t->effects_gc, x, y,
		                         x, y+height-1);
   }

      
}

void
theme_paint_pxm(wm *w, Drawable dest, int x, int y, Pxm *p)
{
   XGCValues gc_vals;
   unsigned long valuemask = 0;
      
   XSetClipMask(w->dpy, w->theme->mask_gc, p->mask);
    
   gc_vals.clip_x_origin = x;
   gc_vals.clip_y_origin = y;
   
   valuemask =  GCClipXOrigin | GCClipYOrigin ;
   XChangeGC(w->dpy, w->theme->mask_gc, valuemask, &gc_vals);

   XCopyArea(w->dpy, p->pixmap, dest, w->theme->mask_gc,
	     0, 0, p->width, p->height, x, y );
}
    
void
theme_draw_gradient(wm *w, Drawable dest, GC gc, int x, int y,
		    int width, int height, 
		    XColor* start_col, XColor* end_col) /* TOGO ? */
{
   XColor tmp_col;
   int i;

   float red_step   = ( end_col->red - start_col->red)   / height;
   float green_step = (end_col->green - start_col->green ) / height;
   float blue_step  = (end_col->blue  - start_col->blue )  / height;

   for(i=height;i;i--)
   {
      tmp_col.red   = ( start_col->red   + ( red_step * (height-i) ));
      tmp_col.green = ( start_col->green + ( green_step * (height-i) ));
      tmp_col.blue  = ( start_col->blue  + ( blue_step * (height-i) ));
      
      XAllocColor(w->dpy, DefaultColormap(w->dpy, DefaultScreen(w->dpy)),
		  &tmp_col);
      XSetForeground(w->dpy, gc, tmp_col.pixel); 
      XDrawLine(w->dpy, dest, gc, x, y+height-i, width, y+height-i);
   }
   
}


int
_XColorFromStr(Display *display, XColor *col, const char *defstr)
{
  char *str;
  const char delim[] = ",:";
  char *token;
  XColor exact;

  if ((strchr(defstr, delim[0]) != NULL)
      || (strchr(defstr, delim[1]) != NULL) )
  {
     str = strdup(defstr);

     token = strsep (&str, delim); 
     col->red = ( atoi(token) * 65535 ) / 255; 
     token = strsep (&str, delim); 
     col->green = ( atoi(token) * 65535 ) / 255;
     token = strsep (&str, delim); 
     col->blue = ( atoi(token) * 65535 ) / 255;

     return XAllocColor(display,
			DefaultColormap(display, DefaultScreen(display)),
			col);
  } else {
     return XAllocNamedColor(display,
			     DefaultColormap(display, DefaultScreen(display)),
			     defstr, col, &exact);
  }
}

