/*
 * Math3d - The 3D Computer Graphics Math Library
 * Copyright (C) 1996-2000 by J.E. Hoffmann <je-h@gmx.net>
 * All rights reserved.
 *
 * This program is  free  software;  you can redistribute it and/or modify it
 * under the terms of the  GNU Lesser General Public License  as published by 
 * the  Free Software Foundation;  either version 2.1 of the License,  or (at 
 * your option) any later version.
 *
 * This  program  is  distributed in  the  hope that it will  be useful,  but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or  FITNESS FOR A  PARTICULAR PURPOSE.  See the  GNU Lesser General Public  
 * License for more details.
 *
 * You should  have received  a copy of the GNU Lesser General Public License
 * along with  this program;  if not, write to the  Free Software Foundation,
 * Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: m2d.cpp,v 1.6 2000/10/09 12:17:54 jeh Exp $
 */
#define _MATH3D_EXPORT
#include <math3d/m2d.h>
#include <math3d/m3d.h>
#include <cmath>
#include <iostream>


/*!
 *
 */
Math3d::M2d::M2d(const M2d& A)
{
  d_v[0]=A.d_v[0];
  d_v[1]=A.d_v[1];
}


/*!
 * Construction from homogenous coordinates
 * ([W*X, W*Y, W] with scale factor W !=0 -> [X, Y]).
 *
 * \param A 4-dimensional vector. 
 */
Math3d::M2d::M2d(const M3d& A)
{
  if (fabs(A.d_v[2])<EPSILON) {
    ASSERT(false);
    d_v[0]=d_v[1]=0.0;
  }

  double c=1/A.d_v[2];
  d_v[0]=A.d_v[0]*c;
  d_v[1]=A.d_v[1]*c;
}


/*!
 *
 */
const Math3d::M2d&
Math3d::M2d::operator=(const M2d& A)
{
  d_v[0]=A.d_v[0];
  d_v[1]=A.d_v[1];
  return(*this);
}


/*!
 *
 */
void
Math3d::M2d::zero()
{
  d_v[0]=d_v[1]=0.0;
}


/*!
 *
 */
void
Math3d::M2d::copy(const M2d& A)
{
  d_v[0]=A.d_v[0];
  d_v[1]=A.d_v[1];
}


/*!
 *
 */
double&
Math3d::M2d::get(int i)
{
  ASSERT(i>=0 && i<2);
  return(d_v[i]);
}


Math3d::M2d
Math3d::M2d::operator+(const M2d& A)
{
  return(M2d(
    d_v[0]+A.d_v[0],
    d_v[1]+A.d_v[1]
  ));
}


/*!
 *
 */
Math3d::M2d
Math3d::M2d::operator+()
{
  return(*this);
}


/*!
 *
 */
const Math3d::M2d&
Math3d::M2d::operator+=(const M2d& A)
{
  d_v[0]+=A.d_v[0];
  d_v[1]+=A.d_v[1];
  return(*this);
}


/*!
 *
 */
Math3d::M2d
Math3d::M2d::operator-(const M2d& A)
{
  return(M2d(
    d_v[0]-A.d_v[0],
    d_v[1]-A.d_v[1]
  ));
}


/*!
 *
 */
Math3d::M2d
Math3d::M2d::operator-()
{
  return(M2d(-d_v[0],-d_v[1]));
}


/*!
 *
 */
const Math3d::M2d&
Math3d::M2d::operator-=(const M2d& A)
{
  d_v[0]-=A.d_v[0];
  d_v[1]-=A.d_v[1];
  return(*this);
}


/*!
 *
 */
Math3d::M2d
Math3d::M2d::operator*(double k)
{
  return(M2d(
    d_v[0]*k,
    d_v[1]*k
  ));
}


/*!
 *
 */
const Math3d::M2d&
Math3d::M2d::operator*=(double k)
{
  d_v[0]*=k;
  d_v[1]*=k;
  return(*this);
}


/*!
 *
 */
void
Math3d::M2d::neg()
{
  d_v[0]=-d_v[0];
  d_v[1]=-d_v[1];
}


/*!
 *
 */
void
Math3d::M2d::abs()
{
  d_v[0]=fabs(d_v[0]);
  d_v[1]=fabs(d_v[1]);
}


/*!
 *
 */
void
Math3d::M2d::add(const M2d& A, const M2d& B)
{
  d_v[0]=A.d_v[0] + B.d_v[0]; 
  d_v[1]=A.d_v[1] + B.d_v[1]; 
}


/*!
 *
 */
void
Math3d::M2d::sub(const M2d& A, const M2d& B)
{
  d_v[0]=A.d_v[0] - B.d_v[0]; 
  d_v[1]=A.d_v[1] - B.d_v[1]; 
}


/*!
 *
 */
void
Math3d::M2d::scalar(double k)
{
  d_v[0]*=k;
  d_v[1]*=k;
}


/*!
 *
 */
void
Math3d::M2d::normalize()
{
  double l,c;

  l=length();
  if (fabs(l)<EPSILON) {
    d_v[0]=d_v[1]=0.0;
  }
  else {
    c=1.0/l;
    d_v[0]*=c;
    d_v[1]*=c;
  }
}


/*!
 *
 */
void
Math3d::M2d::lerp(const M2d& A, const M2d& B, double t)
{
  d_v[0]=A.d_v[0]+t*(B.d_v[0]-A.d_v[0]);
  d_v[1]=A.d_v[1]+t*(B.d_v[1]-A.d_v[1]);
}


/*!
 *
 */
void
Math3d::M2d::min(M2d& min) const
{
  if (d_v[0]<min.d_v[0]) min.d_v[0]=d_v[0];
  if (d_v[1]<min.d_v[1]) min.d_v[1]=d_v[1];
}


/*!
 *
 */
void
Math3d::M2d::max(M2d& max) const
{
  if (d_v[0]>max.d_v[0]) max.d_v[0]=d_v[0];
  if (d_v[1]>max.d_v[1]) max.d_v[1]=d_v[1];
}


/*!
 *
 */
void
Math3d::M2d::cubic(const M2d& A, const M2d& TA, const M2d& TB, 
  const M2d& B, double t)
{
  double a,b,c,d;   

  a=2*t*t*t - 3*t*t + 1;
  b=-2*t*t*t + 3*t*t;
  c=t*t*t - 2*t*t + t;
  d=t*t*t - t*t;
  d_v[0]=a*A.d_v[0] + b*B.d_v[0] + c*TA.d_v[0] + d*TB.d_v[0];
  d_v[1]=a*A.d_v[1] + b*B.d_v[1] + c*TA.d_v[1] + d*TB.d_v[1];
}


/*!
 *
 */
double
Math3d::M2d::get(int i) const
{
  ASSERT(i>=0 && i<2);
  return(d_v[i]);
}


/*!
 *
 */
bool
Math3d::M2d::operator==(const M2d& A) const
{
  return(cmp(A));
}


/*!
 *
 */
bool
Math3d::M2d::operator!=(const M2d& A) const
{
  return(!cmp(A));
}


/*!
 *
 */
double
Math3d::M2d::operator*(const M2d& A) const
{
  return(d_v[0]*A.d_v[0] + d_v[1]*A.d_v[1]);
}


/*!
 *
 */
double 
Math3d::M2d::dot(const M2d& A) const
{
  return(d_v[0]*A.d_v[0] + d_v[1]*A.d_v[1]);
}


/*!
 *
 */
bool 
Math3d::M2d::cmp(const M2d& A, double epsilon) const
{
  return(
    (fabs(d_v[0]-A.d_v[0])<epsilon) &&
    (fabs(d_v[1]-A.d_v[1])<epsilon)
  );
}


/*!
 *
 */
double
Math3d::M2d::squared() const
{
  return(d_v[0]*d_v[0] + d_v[1]*d_v[1]);
}


/*!
 *
 */
double
Math3d::M2d::length() const
{
  return(sqrt(d_v[0]*d_v[0] + d_v[1]*d_v[1]));
}


/*!
 *
 */
ostream& 
Math3d::operator << (ostream& co, const M2d& v)
{
  co << "(" << v[0] << ", " << v[1] << ")";
  return co;
}







