"""

Displays a surface map of any data.  It should work for any
dataset but is best if used for 2d surfaces (polydata and unstructured
surfaces).

This code is distributed under the conditions of the GPL (GNU General
Public License).  Look at the file LICENSE included with the
distribution for details.

Copyright (C) 2001, Prabhu Ramachandran.
"""

__author__ = "Prabhu Ramachandran <prabhu_r@users.sf.net>"
__version__ = "$Revision: 1.7 $"
__date__ = "$Date: 2001/11/02 12:16:29 $"

import Base.Objects, Common
import Tkinter, tkColorChooser
import vtkpython
import vtkPipeline.vtkMethodParser

debug = Common.debug

class SurfaceMap (Base.Objects.Module):

    """ Displays a surface map of any data.  It should work for any
    dataset but is best if used for 2d surfaces (polydata and
    unstructured surfaces). """

    def __init__ (self, mod_m):
        debug ("In SurfaceMap::__init__ ()")
        Common.state.busy ()
        Base.Objects.Module.__init__ (self, mod_m)
        self.cont_fil = vtkpython.vtkContourFilter ()
        self.mapper = self.map = vtkpython.vtkDataSetMapper ()
        self.map.SetInput (mod_m.GetOutput ())
        self.actor = self.act = vtkpython.vtkActor ()
        self.act.SetMapper (self.map)
        self.data_range = self.mod_m.get_scalar_data_range ()
        self.map.SetLookupTable (self.mod_m.get_scalar_lut ())
        self.map.SetScalarRange (self.data_range)
        self.map.ScalarVisibilityOn ()
        apply (self.act.GetProperty ().SetColor, Common.config.fg_color)
        self.act.GetProperty ().SetLineWidth (2)
        # surface mode.
        self.act.GetProperty ().SetRepresentation (2)
        # used for the pipeline browser
        self.pipe_objs = self.act
        self.renwin.add_actors (self.act)
        self._gui_init ()
        self.renwin.Render ()
        Common.state.idle ()

    def __del__ (self): 
        debug ("In SurfaceMap::__del__ ()")
        if self.act:
            self.renwin.remove_actors (self.act)
        self.renwin.Render ()

    def _gui_init (self):
        self._contour_init ()

    def _contour_init (self):
        debug ("In SurfaceMap::_contour_init ()")
        Base.Objects.Module._contour_init (self)
        self.contour_on.set (0)
        self.n_cnt.set (10)
        dr = self.mod_m.get_scalar_data_range ()        
        self.min_cnt.set (dr[0])
        self.max_cnt.set (dr[1])
        
    def SetInput (self, source): 
        debug ("In SurfaceMap::SetInput ()")
        Common.state.busy ()
        dr = self.mod_m.get_scalar_data_range ()
        self.map.SetInput (source)
        self.map.SetScalarRange (dr)
        self.data_range = dr        
        self.min_cnt.set (dr[0])
        self.max_cnt.set (dr[1])
        self.do_contour ()
        Common.state.idle ()

    def save_config (self, file): 
        debug ("In SurfaceMap::save_config ()")
        file.write ("%d, %d, %f, %f\n"%(self.contour_on.get (),
                                        self.n_cnt.get (),
                                        self.min_cnt.get (),
                                        self.max_cnt.get ()))
        p = vtkPipeline.vtkMethodParser.VtkPickler ()
        for obj in (self.cont_fil, self.map, self.act,
                    self.act.GetProperty ()):
            p.dump (obj, file)

    def load_config (self, file): 
        debug ("In SurfaceMap::load_config ()")
        cnt_on, n_cnt, min_cnt, max_cnt = eval (file.readline ())
        self.contour_on.set (cnt_on)
        self.n_cnt.set (n_cnt)
        self.min_cnt.set (min_cnt)
        self.max_cnt.set (max_cnt)
        p = vtkPipeline.vtkMethodParser.VtkPickler ()
        for obj in (self.cont_fil, self.map, self.act,
                    self.act.GetProperty ()):
            p.load (obj, file)

        self.do_contour ()
        
    def config_changed (self): 
        debug ("In SurfaceMap::config_changed ()")
        apply (self.act.GetProperty ().SetColor, Common.config.fg_color)

    def make_main_gui (self, master=None): 
        debug ("In SurfaceMap::make_main_gui ()")
        "Create the GUI configuration controls for this object."
        self.make_actor_gui ()
        self.make_contour_gui ()

    def do_contour (self, event=None):
        debug ("In SurfaceMap::do_contour ()")
        Common.state.busy ()
        if self.contour_on.get ():
            if not self.mod_m.get_scalar_data_name ():
                self.contour_on.set (0)
                msg = "Warning: No scalar data present to contour!"
                Common.print_err (msg)
                Common.state.idle ()
                return
            self.cont_fil.SetInput (self.mod_m.GetOutput ())
            self.map.SetInput (self.cont_fil.GetOutput ())
        else:
            self.map.SetInput (self.mod_m.GetOutput ())
        self.change_contour ()
        Common.state.idle ()

    def change_contour (self, event=None):
        debug ("In SurfaceMap::change_contour ()")
        Common.state.busy ()
        n_cnt = self.n_cnt.get ()
        min_cnt = self.min_cnt.get ()
        max_cnt = self.max_cnt.get ()    
        if max_cnt < min_cnt:
            msg = "Error: Max. contour less than min. contour. "\
                  "Setting to defaults."
            debug (msg)
            dr = self.data_range
            min_cnt, max_cnt = dr[0], dr[1]
            self.min_cnt.set (min_cnt)
            self.max_cnt.set (max_cnt)

        if self.contour_on.get ():
            self.cont_fil.GenerateValues (n_cnt, min_cnt, max_cnt)
            self.cont_fil.Update ()
        self.renwin.Render ()
        Common.state.idle ()

