""" A detail view of the selected resource. """


# Major package imports.
import wx

# Enthought library imports.
from enthought.envisage.resource.ui import ExplorerTree
from enthought.envisage.workbench import View
from enthought.naming.api import Context
from enthought.pyface.api import LayeredPanel


class ResourceDetailView(View):
    """ A detail view of the selected resource.

    This view is intended for developers to help in debugging etc.  It is
    NOT intended to be seen by the end user!

    """

    ###########################################################################
    # 'View' interface.
    ###########################################################################

    def create_control(self, parent):
        """ Creates the toolkit-specific control that represents the view.

        'parent' is the toolkit-specific control that is the view's parent.

        """

        sizer = wx.BoxSizer(wx.VERTICAL)
        panel = wx.Panel(parent, -1, style=wx.CLIP_CHILDREN)
        panel.SetSizer(sizer)
        panel.SetAutoLayout(True)

        self._details = LayeredPanel(panel)
        sizer.Add(self._details.control, 1, wx.EXPAND)

        # Resize the splitter to fit the sizer's minimum size.
        sizer.Fit(panel)

        # Listen to the window selection.
        self.window.on_trait_change(self._on_selection_changed, 'selection')

        return panel

    ###########################################################################
    # Private interface.
    ###########################################################################

    def _create_detail_layer(self, parent, binding):
        """ Creates the detail layer for an object. """


        application = self.window.application

        resource_manager = application.get_service(
            'enthought.envisage.resource.IResourceManager'
        )
        type_manager = resource_manager.type_manager

        obj = binding.obj
        if isinstance(obj, Context):
            context = obj

        else:
            context = binding.context

        resource_type    = resource_manager.get_type_of(obj)
        node_type        = resource_type.node_type
        context_adapter  = type_manager.object_as(
            obj, Context, context.environment, context
        )
        mro              = type_manager.type_system.get_mro(type(obj))

        detail = {
            'Value' : obj,
            'Type'  : type(obj),
            'Resource type - %s' % resource_type.__class__.__name__ : resource_type,
            'Node type - %s' % node_type.__class__.__name__ : node_type,
            'Context - %s' % context_adapter.__class__.__name__ : context_adapter,
            'MRO' : mro
        }

        detail = [
            obj,
            type(obj),
            resource_type,
            node_type,
            context_adapter,
            type_manager.type_system.get_mro(type(obj))
        ]

        tree = ExplorerTree(parent, obj=detail)
        for node in tree.model.get_children(tree.model.root):
            tree.expand(node)

        return tree.control

    #### Trait event handlers #################################################

    def _on_selection_changed(self, selection):
        """ Called when the selection is changed in the window. """

        if len(selection) > 0:
            obj  = selection[0]
            name = str(id(obj))

            if not self._details.has_layer(name):
                self._details.add_layer(
                    name,
                    self._create_detail_layer(self._details.control, obj)
                )

            self._details.show_layer(name)

        return

#### EOF ######################################################################
