/* -*- C -*- 
   mboxgrep - scan mailbox for messages matching a regular expression
   Copyright (C) 2000, 2001, 2002  Daniel Spiljar

   Mboxgrep is free software; you can redistribute it and/or modify it 
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   Mboxgrep is distributed in the hope that it will be useful, but 
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with mboxgrep; if not, write to the Free Software Foundation, 
   Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: mbox.c,v 1.14 2002/02/06 22:10:44 dspiljar Exp $ */

#include <config.h>

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <errno.h>
#include <time.h>
#ifdef HAVE_FLOCK
#include <sys/file.h>
#endif /* HAVE_FLOCK */
#ifdef HAVE_LIBZ
#include <zlib.h>
#endif /* HAVE_LIBZ */

#include "mboxgrep.h"
#include "mbox.h"
#include "misc.h"
#include "wrap.h"


FILE *
mbox_open (const char *path, const char *mode)
{
  FILE *fp;
  int fd;
#ifndef HAVE_FLOCK
  struct flock lck;
#endif /* HAVE_FLOCK */
  char buffer[BUFSIZ];

  if (mode[0] == 'r')
    fd = m_open (path, O_RDONLY, 0);
  if (mode[0] == 'w')
    fd = m_open (path, (O_WRONLY | O_CREAT | O_APPEND) , (S_IWUSR | S_IRUSR));

  if (fd == -1)
    {
      if (config.merr)
	{
	  fprintf (stderr, "%s: %s: ", APPNAME, path);
	  perror (NULL);
	}
      errno = 0;
      return NULL;
    }

  if (config.lock)
    {
#ifdef HAVE_FLOCK
      int op;

      if (mode[0] == 'r')
	op = LOCK_SH;
      else
	op = LOCK_EX;
      if (-1 == flock (fd, op))
#else
      memset (&lck, 0, sizeof (struct flock));
      lck.l_whence = SEEK_SET;
      if (mode[0] == 'r')
	lck.l_type = F_RDLCK;
      else
	lck.l_type = F_WRLCK;

      if (-1 == fcntl (fd, F_SETLK, &lck))
#endif /* HAVE_FLOCK */
	{
	  if (config.merr)
	    {
	      fprintf (stderr, "%s: %s: ", APPNAME, path);
	      perror (NULL);
	    }
	  errno = 0;
	  close (fd);
	  return NULL;
	}
    }

  if (mode[0] == 'r')
    fp = m_fdopen (fd, "r");
  else if (mode[0] == 'w')
    fp = m_fdopen (fd, "w");

  if (fp == NULL)
    {
      if (config.merr)
	{
	  fprintf (stderr, "%s: %s: ", APPNAME, path);
	  perror (NULL);
	}
      errno = 0;
      close (fd);
      return NULL;
    }

  if (mode[0] == 'r')
    {
      fgets (buffer, BUFSIZ, fp);
      if (0 != strncmp ("From ", buffer, 5))
	{
	  if (config.merr)
	    fprintf (stderr, "%s: %s: Not a mbox folder\n", APPNAME, path);
	  close (fd);
	  return NULL;
	}
      rewind (fp);
    }
  return fp;
}

void
mbox_close (FILE * fp)
{
  int fd;
/*  struct flock lck; */

/*  fd = fileno (fp); */
  fclose (fp);
/*  memset (&lck, 0, sizeof (struct flock));
  lck.l_type = F_UNLCK;
  fcntl (fd, F_SETLK, &lck);
  close (fd); */
}

message_t *
mbox_read_message (FILE * fp)
{
  int isheaders = 1, s;
  long int pos;
  char buffer[BUFSIZ];
  message_t *message;

  message = (message_t *) xmalloc (sizeof (message_t));

  message->headers = (char *) xmalloc (sizeof (char));
  message->headers[0] = '\0';
  message->hbytes = 0;

  message->body = (char *) xmalloc (sizeof (char));
  message->body[0] = '\0';
  message->bbytes = 0;

  for (;;)
    {
      if (fgets (buffer, BUFSIZ, fp) == NULL)
	{
	  if (isheaders)
	    return NULL;
	  else
	    return message;
	}
      s = strlen (buffer);
      pos = (ftell (fp) - s);

      if (buffer[0] == '\n' && isheaders == 1)
	{
	  isheaders = 0;
	  continue;
	}

      if (isheaders)
	{
	  if (0 == strncasecmp ("Date: ", buffer, 6) && isheaders == 1)
	    message->date = parse_date(buffer);
	  /* check as above */
	  message->headers =
	    (char *) realloc (message->headers,
			      ((1 + s + message->hbytes) * sizeof (char)));
	  strcpy (message->headers + message->hbytes, buffer);
	  message->hbytes += s;
	}			/* if */
      else
	{
	  if (0 == strncmp (buffer, "From ", 5))
	    {
	      fseek (fp, pos, SEEK_SET);
	      return message;
	    }
	  message->body =
	    (char *) realloc (message->body,
			      ((1 + s + message->bbytes) * sizeof (char)));
	  strcpy (message->body + message->bbytes, buffer);
	  message->bbytes += s;

	}
    }				/* for */
  return NULL;
}

FILE *
tmpfile_open (void)
{
  extern char *tmpfilename;
  char fname[] = "mboxgrepXXXXXX";
  char *tmpdir;
  int foo;
  FILE *bar;

  tmpdir = getenv ("TMPDIR");
  if (tmpdir == NULL)
    tmpdir = xstrdup ("/tmp");

  tmpfilename = (char *) xmalloc ((strlen (tmpdir) + (strlen (fname) + 3)
				    * sizeof (char)));
  sprintf (tmpfilename, "%s/%s", tmpdir, fname);
  foo = mkstemp (tmpfilename);
  if (-1 == foo)
    {
      if (config.merr)
	{
	  fprintf (stderr, "%s: %s: ", APPNAME, tmpfilename);
	  perror (NULL);
	}
      exit (2);
    }
  bar = m_fdopen (foo, "w");

  return bar;
}

#ifdef HAVE_LIBZ

gzFile *
gzmbox_open (const char *path, const char *mode)
{
  gzFile *fp;
  int fd;
#ifndef HAVE_FLOCK
  struct flock lck;
#endif /* HAVE_FLOCK */
  char buffer[BUFSIZ];

  if (mode[0] == 'r')
    fd = m_open (path, O_RDONLY, 0);
  else if (mode[0] == 'w')
    fd = m_open (path, O_WRONLY, 0);
  else
    return NULL;

  if (fd == -1)
    {
      if (config.merr)
	{
	  fprintf (stderr, "%s: %s: ", APPNAME, path);
	  perror (NULL);
	}
      errno = 0;
      return NULL;
    }

  if (config.lock)
    {
#ifdef HAVE_FLOCK
      int op;
      
      if (mode[0] == 'r')
	op = LOCK_SH;
      else
	op = LOCK_EX;
      if (-1 == flock (fd, op))
#else
      memset (&lck, 0, sizeof (struct flock));
      lck.l_whence = SEEK_SET;
      if (mode[0] == 'r')
	lck.l_type = F_RDLCK;
      else
	lck.l_type = F_WRLCK;
      
      if (-1 == fcntl (fd, F_SETLK, &lck))
#endif /* HAVE_FLOCK */
	{
	  if (config.merr)
	    {
	      fprintf (stderr, "%s: %s: ", APPNAME, path);
	      perror (NULL);
	    }
	  errno = 0;
	  close (fd);
	  return NULL;
	}
    }

  if (mode[0] == 'r')
    fp = (gzFile *) gzdopen (fd, "rb");
  else if (mode[0] == 'w')
    fp = (gzFile *) gzdopen (fd, "wb");
  if (fp == NULL)
    {
      if (config.merr)
	{
	  fprintf (stderr, "%s: %s: ", APPNAME, path);
	  perror (NULL);
	}
      errno = 0; 
      close (fd); 
      return NULL;
    }

  if (mode[0] == 'r')
    {
      gzgets (fp, buffer, BUFSIZ);
      if (0 != strncmp ("From ", buffer, 5))
	{
	  if (config.merr)
	    fprintf (stderr, "%s: %s: Not a mbox folder\n", APPNAME, path); 
	  close (fd);
	  return NULL;
	}
      gzrewind (fp); 
    }
  return fp;
}

void
gzmbox_close (gzFile * fp)
{
  int fd;
/*  struct flock lck; */

/*  fd = fileno (fp); */
  gzclose (fp);
/*  memset (&lck, 0, sizeof (struct flock));
  lck.l_type = F_UNLCK;
  fcntl (fd, F_SETLK, &lck);
  close (fd); */
}

message_t *
gzmbox_read_message (gzFile * fp)
{
  int isheaders = 1, s;
  long int pos;
  char buffer[BUFSIZ];
  message_t *message;

  message = (message_t *) xmalloc (sizeof (message_t));

  message->hbytes = 0;
  message->bbytes = 0;
  message->headers = NULL;
  message->body = NULL;

  for (;;)
    {
      if (gzgets (fp, buffer, BUFSIZ) == NULL)
	{
	  if (isheaders)
	    return NULL;
	  else
	    return message;
	}
      s = strlen (buffer);
      pos = (gztell (fp) - s);

      if (buffer[0] == '\n' && isheaders == 1)
	{
	  isheaders = 0;
	  continue;
	}

      if (isheaders)
	{
	  if (0 == strncasecmp ("Date: ", buffer, 6) && isheaders == 1)
	    message->date = parse_date(buffer);
	  /* check as above */
	  message->headers =
	    (char *) realloc (message->headers,
			      ((1 + s + message->hbytes) * sizeof (char)));
	  strcpy (message->headers + message->hbytes, buffer);
	  message->hbytes += s;
	}			/* if */
      else
	{
	  if (0 == strncmp (buffer, "From ", 5))
	    {
	      gzseek (fp, pos, SEEK_SET);
	      return message;
	    }
	  message->body =
	    (char *) realloc (message->body,
			      ((1 + s + message->bbytes) * sizeof (char)));
	  strcpy (message->body + message->bbytes, buffer);
	  message->bbytes += s;

	}
    }				/* for */
  return NULL;
}

gzFile *
gztmpfile_open (void)
{
  extern char *tmpfilename;
  char fname[] = "mboxgrepXXXXXX";
  char *tmpdir;
  int foo;
  static gzFile *bar;

  tmpdir = getenv ("TMPDIR");
  if (tmpdir == NULL)
    tmpdir = xstrdup ("/tmp");

  tmpfilename = (char *) xmalloc ((strlen (tmpdir) + (strlen (fname) + 3)
				    * sizeof (char)));
  sprintf (tmpfilename, "%s/%s", tmpdir, fname);
  foo = mkstemp (tmpfilename);
  if (-1 == foo)
    {
      if (config.merr)
	{
	  fprintf (stderr, "%s: %s: ", APPNAME, tmpfilename);
	  perror (NULL);
	}
      exit (2);
    }
  bar = gzdopen (foo, "wb");

  return bar;
}

#endif /* HAVE_LIBZ */
