/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */
 
#include "main_window.hh"
#include "application.hh"
#include "exception_handler.hh"
#include "scope_log.hh"
#include "xine_engine.hh"
#include "mutex_lock.hh"
#include "gdk_lock.hh"

#include <libgnomeui/libgnomeui.h>
#include <X11/Xlib.h>
#include <X11/extensions/XTest.h>
#include "config.h"

#define POKE_INTERVAL				60
#define CURSOR_HIDE_INTERVAL		3

#define COLUMN_CHANNEL_NAME			0
#define COLUMN_WEIGHT				1

MainWindow::MainWindow()
{		
	Application& application = Application::get_current();
	Configuration& configuration = application.get_configuration();
	Glade& glade = application.get_glade();
	
	audio_stream_group		= NULL;
	subtitle_stream_group	= NULL;
	
	window_main				= glade.get_widget("window_main");
	vpaned_main				= glade.get_widget("vpaned_main");
	statusbar				= glade.get_widget("statusbar");
	toolbar					= glade.get_widget("handlebox_toolbar");
	menubar					= glade.get_widget("menubar");
	dialog_program_details	= glade.get_widget("dialog_program_details");
	event_box_video			= glade.get_widget("event_box_video");
	
	epg_widget = NULL;
	epg_widget					= new GtkEpgWidget();
	preferences_dialog			= new PreferencesDialog();
	scheduled_recording_dialog	= new ScheduledRecordingDialog();
	scheduled_recordings_dialog	= new ScheduledRecordingsDialog();
	
	search_bar = new SearchBar();
	
	is_fullscreen = false;
	
	gtk_window_set_title(GTK_WINDOW(window_main), APPLICATION_NAME);
	
	g_signal_connect( G_OBJECT ( window_main ), "motion-notify-event", G_CALLBACK( on_window_main_motion_notify ), this );
	g_signal_connect( G_OBJECT ( window_main ), "delete-event", G_CALLBACK ( on_window_main_delete ), this );
	g_signal_connect( G_OBJECT ( window_main ), "key-press-event", G_CALLBACK( on_window_main_key_press_event ), this );
	g_signal_connect( G_OBJECT ( glade.get_widget("tool_button_mute") ), "toggled", G_CALLBACK ( on_tool_button_mute_toggled ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("tool_button_record") ), "toggled", G_CALLBACK ( on_tool_button_record_toggled ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("tool_button_auto_surf") ), "toggled", G_CALLBACK ( on_tool_button_auto_surf_toggled ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("tool_button_broadcast") ), "toggled", G_CALLBACK ( on_tool_button_broadcast_toggled ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("tool_button_scheduled_recordings") ), "clicked", G_CALLBACK ( on_tool_button_scheduled_recordings_clicked ), this );
	g_signal_connect( G_OBJECT ( glade.get_widget("tool_button_hide_controls") ), "clicked", G_CALLBACK ( on_tool_button_hide_controls_clicked ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("menu_item_file_quit") ), "activate", G_CALLBACK ( on_menu_item_file_quit_activate ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("menu_item_view_fullscreen") ), "activate", G_CALLBACK ( on_menu_item_view_fullscreen_activate ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("menu_item_view_preferences") ), "activate", G_CALLBACK ( on_menu_item_view_preferences_activate ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("menu_item_view_epg") ), "activate", G_CALLBACK ( on_menu_item_view_epg_activate ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("menu_item_view_program_search") ), "activate", G_CALLBACK ( on_menu_item_view_program_search_activate ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("menu_item_audio_mute") ), "toggled", G_CALLBACK ( on_menu_item_audio_mute_toggled ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("menu_item_help_contents") ), "activate", G_CALLBACK ( on_menu_item_help_contents_activate ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("menu_item_help_about") ), "activate", G_CALLBACK ( on_menu_item_help_about_activate ), this );	
	g_signal_connect( G_OBJECT ( event_box_video ), "button_press_event",	G_CALLBACK ( on_window_main_button_press ), this );
	
	g_signal_connect( G_OBJECT ( glade.get_widget("combo_box_record_type") ), "changed", G_CALLBACK ( on_combo_box_record_type_changed ), this );	
	g_signal_connect( G_OBJECT ( glade.get_widget("spinbutton_duration") ), "value-changed", G_CALLBACK ( on_spinbutton_duration_value_changed ), this );	

	g_signal_connect( G_OBJECT ( glade.get_widget("menu_item_audio_dual_language_disable") ), "toggled", G_CALLBACK ( on_menu_item_audio_dual_language_changed ), this );
	g_signal_connect( G_OBJECT ( glade.get_widget("menu_item_audio_dual_language_left") ), "toggled", G_CALLBACK ( on_menu_item_audio_dual_language_changed ), this );
	g_signal_connect( G_OBJECT ( glade.get_widget("menu_item_audio_dual_language_right") ), "toggled", G_CALLBACK ( on_menu_item_audio_dual_language_changed ), this );
	
	gint x		= configuration.get_int_value("x");
	gint y		= configuration.get_int_value("y");
	gint width	= configuration.get_int_value("width");
	gint height	= configuration.get_int_value("height");

	if (x != -1)
	{
		gtk_window_move(GTK_WINDOW(window_main), x, y);
		gtk_window_resize(GTK_WINDOW(window_main), width, height);
	}
	
	gtk_window_set_keep_above(GTK_WINDOW(window_main), configuration.get_boolean_value("always_on_top"));

	show_controls(configuration.get_boolean_value("show_controls"));
	show_epg(configuration.get_boolean_value("show_epg"));

	GtkComboBox* combo_box_channel = GTK_COMBO_BOX(glade.get_widget("combo_box_channel"));
	GSList* iterator = application.get_channel_manager().get_channels();
	while (iterator != NULL)
	{
		GtkTreeIter iter;
		Channel& channel = *(Channel*)iterator->data;	
		gtk_combo_box_append_text(combo_box_channel, channel.name.c_str());
		iterator = g_slist_next(iterator);
	}
	gtk_combo_box_set_active(combo_box_channel, 0);
	gtk_combo_box_set_active(GTK_COMBO_BOX(glade.get_widget("combo_box_start_day")), 0);
	
	last_motion_time = time(NULL);

	hidden_cursor = NULL;
	
	gchar     bits[] = {0};
	GdkColor  color = {0, 0, 0, 0};
	GdkPixmap* pixmap = gdk_bitmap_create_from_data(NULL, bits, 1, 1);
	hidden_cursor = gdk_cursor_new_from_pixmap(pixmap, pixmap, &color, &color, 0, 0);

	is_cursor_visible = true;
}

MainWindow::~MainWindow()
{
	Application& application = Application::get_current();
	Configuration& configuration = application.get_configuration();
	
	if (get_epg_visible())
	{
		configuration.set_int_value("video_position",
			gtk_paned_get_position(GTK_PANED(vpaned_main)));
	}
	
	configuration.set_boolean_value("fullscreen", is_fullscreen);
	configuration.set_boolean_value("show_controls", get_controls_visible());
	configuration.set_boolean_value("show_epg", get_epg_visible());

	if (!is_fullscreen)
	{
		gint x, y, width, height;
		
		gtk_window_get_position(GTK_WINDOW(window_main), &x, &y);
		gtk_window_get_size(GTK_WINDOW(window_main), &width, &height);
	
		configuration.set_int_value("x", x);
		configuration.set_int_value("y", y);
		configuration.set_int_value("width", width);
		configuration.set_int_value("height", height);
	}
	
	if (hidden_cursor != NULL)
	{
		gdk_cursor_unref(hidden_cursor);
		hidden_cursor = NULL;
	}
	
	if (search_bar != NULL)
	{
		delete search_bar;
		search_bar = NULL;
	}
	
	if (scheduled_recordings_dialog != NULL)
	{
		delete scheduled_recordings_dialog;
		scheduled_recordings_dialog = NULL;
	}

	if (scheduled_recording_dialog != NULL)
	{
		delete scheduled_recording_dialog;
		scheduled_recording_dialog = NULL;
	}
	
	if (preferences_dialog != NULL)
	{
		delete preferences_dialog;
		preferences_dialog = NULL;
	}
	
	if (epg_widget != NULL)
	{
		delete epg_widget;
		epg_widget = NULL;
	}
}

gboolean MainWindow::on_window_main_delete (GtkWindow* window_main, GdkEvent* event, MainWindow* main_window)
{
	Application& application = Application::get_current();
	Configuration& configuration = application.get_configuration();
	
	gboolean quit_on_close = configuration.get_boolean_value("quit_on_close");
	gboolean has_scheduled_recordings = application.get_recording_manager().has_scheduled_recordings();
	
	main_window->hide();
	if (quit_on_close && !has_scheduled_recordings)
	{
		application.quit();
	}
	else
	{
		application.mute(true);
	}

	return TRUE;
}

gboolean MainWindow::on_window_main_motion_notify(GtkWidget* window_main, GdkEventMotion* event, MainWindow* main_window)
{
	main_window->show_cursor();
	return FALSE;
}

void MainWindow::show_controls(gboolean show)
{
	if (show)
	{
		gtk_widget_show_all(toolbar);
		gtk_widget_show_all(menubar);
		gtk_widget_show_all(statusbar);
	}
	else
	{
		gtk_widget_hide_all(toolbar);
		gtk_widget_hide_all(menubar);
		gtk_widget_hide_all(statusbar);
	}
}

void MainWindow::show_epg(gboolean show)
{
	if (show)
	{
		update_epg();
		epg_widget->show();
	}
	else
	{
		epg_widget->hide();
		search_bar->hide();
	}

	Glade& glade = Application::get_current().get_glade();
	GtkCheckMenuItem* menu_item_view_epg = GTK_CHECK_MENU_ITEM(glade.get_widget("menu_item_view_epg"));
	gboolean active = gtk_check_menu_item_get_active(menu_item_view_epg);
	if (active != show)
	{
		gtk_check_menu_item_set_active(menu_item_view_epg, show);
	}
}

void MainWindow::show_fullscreen(gboolean show)
{
	if (show)
	{
		gtk_window_fullscreen(GTK_WINDOW(window_main));
	}
	else
	{
		gtk_window_unfullscreen(GTK_WINDOW(window_main));
	}

	is_fullscreen = show;
}

void MainWindow::toggle_epg()
{	
	if (is_fullscreen)
	{
		if (get_controls_visible())
		{
			hide_controls();
		}
		else
		{
			show_epg(true);
			show_controls(true);
		}
	}
	else
	{
		if (get_epg_visible())
		{
			show_epg(false);
		}
		else
		{
			show_epg(true);
			show_controls(true);
		}
	}
}

void MainWindow::hide_controls()
{
	show_controls(false);
	show_epg(false);
}

void MainWindow::toggle_fullscreen()
{
	if (!is_fullscreen)
	{
		was_maximised = gdk_window_get_state(window_main->window) & GDK_WINDOW_STATE_MAXIMIZED;
		show_fullscreen(true);
		show_controls(false);
		show_epg(false);
	}
	else 		
	{
		show_fullscreen(false);
		show_controls(true);

		if (!was_maximised)
		{
			gtk_window_unmaximize(GTK_WINDOW(window_main)); 		
		}
	}
}

void MainWindow::on_tool_button_mute_toggled (GtkToggleToolButton *widget, MainWindow* main_window)
{
	TRY
	Application::get_current().mute(gtk_toggle_tool_button_get_active(widget));
	CATCH
}

void MainWindow::on_tool_button_record_toggled (GtkToggleToolButton* widget, MainWindow* main_window)
{
	TRY
	gboolean active = gtk_toggle_tool_button_get_active(widget);
	Application& application = Application::get_current();

	if (!active && application.is_scheduled_recording() && !application.is_shutting_down())
	{
		gint result = application.show_message_dialog(
			_("You have stopped a scheduled recording.  "\
			   "If you do not delete the scheduled recording then it will start again automatically.  "\
			   "Would you like to delete this scheduled recording?"),
			GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO);
		
		if (result == GTK_RESPONSE_YES)
		{
			application.remove_current_scheduled_recording();
		}
	}
	
	Application::get_current().record(active);
	CATCH
}

void MainWindow::on_tool_button_auto_surf_toggled (GtkToggleToolButton* widget, MainWindow* main_window)
{
	TRY
	Application::get_current().auto_surf(gtk_toggle_tool_button_get_active(widget));
	CATCH
}

void MainWindow::on_tool_button_broadcast_toggled (GtkToggleToolButton *widget, MainWindow* main_window)
{
	TRY
	Application::get_current().broadcast(gtk_toggle_tool_button_get_active(widget));
	CATCH
}

void MainWindow::on_tool_button_scheduled_recordings_clicked(GtkToolButton *widget, MainWindow* main_window)
{
	TRY
	main_window->show_scheduled_recordings_dialog();
	CATCH
}

void MainWindow::on_tool_button_hide_controls_clicked(GtkToolButton *widget, MainWindow* main_window)
{
	TRY
	Application::get_current().hide_controls();
	CATCH
}

void MainWindow::on_menu_item_file_quit_activate (GtkMenuItem *widget, MainWindow* main_window)
{
	TRY
	Application::get_current().quit();
	CATCH
}

void MainWindow::on_menu_item_view_preferences_activate (GtkMenuItem* widget, MainWindow* main_window)
{
	TRY
	main_window->show_preferences_dialog();
	CATCH
}

void MainWindow::on_menu_item_view_fullscreen_activate (GtkMenuItem* widget, MainWindow* main_window)
{
	TRY
	main_window->toggle_fullscreen();
	CATCH
}

void MainWindow::on_menu_item_view_epg_activate (GtkCheckMenuItem* widget, MainWindow* main_window)
{
	TRY	
	main_window->show_epg(gtk_check_menu_item_get_active(widget));
	CATCH
}

void MainWindow::on_menu_item_view_program_search_activate (GtkMenuItem* widget, MainWindow* main_window)
{
	TRY
	main_window->show_search();
	CATCH
}

void MainWindow::on_menu_item_audio_stream_toggled(GtkCheckMenuItem *checkmenuitem, gpointer data)
{
	TRY
	if (gtk_check_menu_item_get_active(checkmenuitem))
	{
		guint audio_pid = (guint)(glong)data;
		Log::write(_("Setting audio PID to %d"), audio_pid);
		Application& application = Application::get_current();
		Configuration& configuration = application.get_configuration();

		Engine* engine = application.get_engine();
		if (engine != NULL)
		{
			if (audio_pid == 0)
			{
				engine->set_audio_channel(-1);
				configuration.set_string_value("preferred_language", "");
			}
			else
			{
				const Stream& stream = application.get_streamer().get_stream();
				for (guint i = 0; i < stream.audio_streams.size(); i++)
				{
					AudioStream audio_stream = stream.audio_streams[i];
					if (audio_stream.pid == audio_pid)
					{
						engine->set_audio_channel(i);
						configuration.set_string_value("preferred_language", audio_stream.language);
					}
				}
			}
		}
	}
	CATCH
}

void MainWindow::on_menu_item_subtitle_stream_toggled(GtkCheckMenuItem *checkmenuitem, gpointer data)
{
	TRY
	if (gtk_check_menu_item_get_active(checkmenuitem))
	{
		guint subtitle_pid = (guint)(glong)data;		
		Log::write(_("Setting subtitle PID to %d"), subtitle_pid);
		Application& application = Application::get_current();
		Configuration& configuration = application.get_configuration();

		Engine* engine = application.get_engine();
		if (engine != NULL)
		{
			if (subtitle_pid == 0)
			{
				engine->set_subtitle_channel(-1);
				configuration.set_string_value("preferred_language", "");
			}
			else
			{
				const Stream& stream = application.get_streamer().get_stream();
				for (guint i = 0; i < stream.subtitle_streams.size(); i++)
				{
					SubtitleStream subtitle_stream = stream.subtitle_streams[i];
					if (subtitle_stream.pid == subtitle_pid)
					{
						engine->set_subtitle_channel(i);
						configuration.set_string_value("preferred_language", subtitle_stream.language);
					}
				}
			}
		}
	}
	CATCH
}

void MainWindow::on_menu_item_audio_mute_toggled(GtkCheckMenuItem *checkmenuitem, MainWindow* main_window)
{
	TRY
	Application::get_current().mute(gtk_check_menu_item_get_active(checkmenuitem));
	CATCH
}

void MainWindow::on_menu_item_help_contents_activate (GtkMenuItem* widget, MainWindow* main_window)
{
	TRY
	Application::get_current().show_help();
	CATCH
}

void MainWindow::on_menu_item_help_about_activate (GtkMenuItem* widget, MainWindow* main_window)
{
	TRY
	Application::get_current().show_about();
	CATCH
}

gboolean MainWindow::on_window_main_key_press_event(GtkWidget* window_main, GdkEventKey* event, MainWindow* main_window)
{
	TRY
	if (main_window->search_bar->is_visible())
	{
		return FALSE;
	}
	
	switch(event->keyval)
	{
		case GDK_a:
		case GDK_A:
			Application::get_current().toggle_auto_surf();
			break;
		case GDK_b:
		case GDK_B:
			Application::get_current().toggle_broadcast();
			break;
		case GDK_e:
		case GDK_E:
			Application::get_current().toggle_epg();
			break;
		case GDK_f:
		case GDK_F:
			if ((event->state & GDK_CONTROL_MASK) != 0)
			{
				main_window->show_search();
			}
			else
			{
				Application::get_current().toggle_fullscreen();
			}
			break;
		case GDK_r:
		case GDK_R:
			Application::get_current().toggle_record();
			break;
		case GDK_m:
		case GDK_M:
			Application::get_current().toggle_mute();
			break;
		case GDK_h:
		case GDK_H:
			Application::get_current().hide_controls();
			break;
		case GDK_Down:
		case GDK_Page_Down:
			{
				Application& application = Application::get_current();
				String video_channel_name = application.get_video_channel().name;
				GSList* channels = application.get_channel_manager().get_channels();
				gboolean done = (channels == NULL);
				while(!done)
				{
					Channel* channel = (Channel*)channels->data;
					channels = g_slist_next(channels);

					if (channels != NULL)
					{
						if (channel->name == video_channel_name)
						{
							Channel* next_channel = (Channel*)channels->data;
							if (next_channel != NULL)
							{
								application.change_channel(*next_channel);
							}
							done = true;
						}
					}
					done |= (channels == NULL);
				}
			}
			break;
		case GDK_Up:
		case GDK_Page_Up:
			{
				Application& application = Application::get_current();
				String video_channel_name = application.get_video_channel().name;
				GSList* channels = application.get_channel_manager().get_channels();
				Channel* prev_channel = NULL;
				gboolean done = (channels == NULL);
				while(!done)
				{
					Channel* channel = (Channel*)channels->data;

					if (prev_channel != NULL)
					{
						if (channel->name == video_channel_name)
						{
							application.change_channel(*prev_channel);
							done = true;
						}
					}

					prev_channel = channel;
					channels = g_slist_next(channels);
					done |= (channels == NULL);
				}
			}
			break;
	}
	CATCH
	
	return FALSE;
}

gboolean MainWindow::on_window_main_button_press(GtkWindow* widget, GdkEventButton *event, MainWindow* main_window)
{
	TRY
	Application& application = Application::get_current();
	if (event->button == 1)
	{				
		if (event->type == GDK_2BUTTON_PRESS)
		{
			application.toggle_fullscreen();
		}
	}
	else if (event->button == 3)
	{
		application.toggle_epg();
	}
	CATCH

	return FALSE;
}

void MainWindow::on_combo_box_record_type_changed (GtkComboBox* widget, MainWindow* main_window)
{
	TRY
	Application& application = Application::get_current();
	Glade& glade = application.get_glade();
	
	int record_type = gtk_combo_box_get_active(widget);

	if (record_type == RECORDING_TYPE_ONCE_OFF)
	{
		gtk_widget_show(glade.get_widget("label_start_date"));
		gtk_widget_show(glade.get_widget("date_edit_start_date"));
		gtk_widget_hide(glade.get_widget("label_start_day"));
		gtk_widget_hide(glade.get_widget("combo_box_start_day"));
	}
	else if (record_type == RECORDING_TYPE_EVERY_WEEK)
	{
		gtk_widget_hide(glade.get_widget("label_start_date"));
		gtk_widget_hide(glade.get_widget("date_edit_start_date"));
		gtk_widget_show(glade.get_widget("label_start_day"));
		gtk_widget_show(glade.get_widget("combo_box_start_day"));
	}
	else
	{
		gtk_widget_hide(glade.get_widget("label_start_date"));
		gtk_widget_hide(glade.get_widget("date_edit_start_date"));
		gtk_widget_hide(glade.get_widget("label_start_day"));
		gtk_widget_hide(glade.get_widget("combo_box_start_day"));
	}
	CATCH
}

void MainWindow::on_spinbutton_duration_value_changed (GtkSpinButton* widget, MainWindow* main_window)
{
	TRY
	Application& application = Application::get_current();
	Glade& glade = application.get_glade();
	
	int duration = (int)gtk_spin_button_get_value(widget);
	String text = " minutes";
	if (duration >= 60)
	{
		text += " (";
		text += EpgEvent::duration_to_string(duration * 60);
		text += ")";
	}
	gtk_label_set_text(GTK_LABEL(glade.get_widget("label_duration_text")), text.c_str());
	CATCH
}

guint MainWindow::get_dual_language_state()
{
	Glade& glade = Application::get_current().get_glade();
	guint result = ENGINE_DUAL_LANGUAGE_STATE_DISABLED;

	GtkCheckMenuItem* menu_item_audio_dual_language_left = GTK_CHECK_MENU_ITEM(glade.get_widget("menu_item_audio_dual_language_left"));
	GtkCheckMenuItem* menu_item_audio_dual_language_right = GTK_CHECK_MENU_ITEM(glade.get_widget("menu_item_audio_dual_language_right"));
	
	if (gtk_check_menu_item_get_active(menu_item_audio_dual_language_left))
	{
		result = ENGINE_DUAL_LANGUAGE_STATE_LEFT;
	}
	else if (gtk_check_menu_item_get_active(menu_item_audio_dual_language_right))
	{
		result = ENGINE_DUAL_LANGUAGE_STATE_RIGHT;
	}
	
	return result;
}

void MainWindow::set_dual_language_state(guint dual_language_state)
{	
	if (dual_language_state != get_dual_language_state ())
	{
		Glade& glade = Application::get_current().get_glade();
		GtkCheckMenuItem* check_menu_item = NULL;
		
		switch(dual_language_state)
		{
			case ENGINE_DUAL_LANGUAGE_STATE_LEFT:
				check_menu_item = GTK_CHECK_MENU_ITEM(glade.get_widget("menu_item_audio_dual_language_left"));
				break;
			case ENGINE_DUAL_LANGUAGE_STATE_RIGHT:
				check_menu_item = GTK_CHECK_MENU_ITEM(glade.get_widget("menu_item_audio_dual_language_right"));
				break;
			default:
				check_menu_item = GTK_CHECK_MENU_ITEM(glade.get_widget("menu_item_audio_dual_language_disable"));
				break;
		}
		
		gtk_check_menu_item_set_active(check_menu_item, true);
	}
}

void MainWindow::on_menu_item_audio_dual_language_changed(GtkCheckMenuItem* widget, MainWindow* main_window)
{
	TRY
	if (gtk_check_menu_item_get_active(widget))
	{
		Application::get_current().set_dual_language_state(main_window->get_dual_language_state());
	}
	CATCH
}

void MainWindow::show_search()
{
	show_epg(true);
	search_bar->show();
}

void MainWindow::update_audio_stream_menu()
{
	Application& application = Application::get_current();
	Glade& glade = application.get_glade();
	Streamer& streamer = application.get_streamer();
	const Stream& stream = streamer.get_stream();
	gboolean channel_set = false;
	
	GtkWidget* menuitem_audio_streams = glade.get_widget("menu_item_audio_streams");
	GtkWidget* menu_widget = gtk_menu_item_get_submenu(GTK_MENU_ITEM(menuitem_audio_streams));
	if (menu_widget != NULL)
	{
		audio_stream_group = NULL;
		gtk_widget_destroy(menu_widget);
	}

	guint audio_stream_size = stream.audio_streams.size();
	if (audio_stream_size > 0)
	{		
		String preferred_language = application.get_configuration().get_string_value("preferred_language");

		menu_widget = gtk_menu_new();
		for (guint i = 0; i < audio_stream_size; i++)
		{
			AudioStream audio_stream = stream.audio_streams[i];
			String label = String::format("%d: '%s' (PID %d)", i + 1, audio_stream.language.c_str(), audio_stream.pid);
			GtkWidget* menu_item = gtk_radio_menu_item_new_with_label(audio_stream_group, label.c_str());
			
			g_signal_connect( G_OBJECT ( menu_item ), "toggled", G_CALLBACK ( on_menu_item_audio_stream_toggled ), (gpointer)audio_stream.pid );
			if (!channel_set && preferred_language == audio_stream.language)
			{
				channel_set = true;
				gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item), true);
			}

			audio_stream_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (menu_item));
			gtk_menu_shell_append(GTK_MENU_SHELL(menu_widget), menu_item);
			gtk_widget_show(menu_item);
		}
		gtk_widget_show(menu_widget);
		
		gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem_audio_streams), menu_widget);
	}
}

void MainWindow::update_view_subtitle_menu()
{
	Application& application = Application::get_current();
	Glade& glade = application.get_glade();
	Streamer& streamer = application.get_streamer();
	const Stream& stream = streamer.get_stream();
	gboolean channel_set = false;

	GtkWidget* menu_item_view_subtitles = glade.get_widget("menu_item_view_subtitles");
	GtkWidget* menu_widget = gtk_menu_item_get_submenu(GTK_MENU_ITEM(menu_item_view_subtitles));
	
	if (menu_widget != NULL)
	{
		gtk_widget_destroy(menu_widget);
		subtitle_stream_group = NULL;
	}
	
	guint subtitle_stream_size = stream.subtitle_streams.size();
	if (subtitle_stream_size > 0)
	{
		menu_widget = gtk_menu_new();
		
		GtkWidget* disabled_menu_item = gtk_radio_menu_item_new_with_label(subtitle_stream_group, _("Subtitles disabled"));
		subtitle_stream_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (disabled_menu_item));
		gtk_menu_shell_append(GTK_MENU_SHELL(menu_widget), disabled_menu_item);
		gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(disabled_menu_item), true);
		g_signal_connect( G_OBJECT ( disabled_menu_item ), "toggled", G_CALLBACK ( on_menu_item_subtitle_stream_toggled ), (gpointer)0 );
		gtk_widget_show(disabled_menu_item);

		String preferred_language = application.get_configuration().get_string_value("preferred_language");
		for (guint i = 0; i < subtitle_stream_size; i++)
		{
			SubtitleStream subtitle_stream = stream.subtitle_streams[i];
			String label = String::format("%d: %s (PID %d)", i + 1, subtitle_stream.language.c_str(), subtitle_stream.pid);
			GtkWidget* menu_item = gtk_radio_menu_item_new_with_label(subtitle_stream_group, label.c_str());
			subtitle_stream_group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (menu_item));
			gtk_menu_shell_append(GTK_MENU_SHELL(menu_widget), menu_item);
			
			g_signal_connect( G_OBJECT ( menu_item ), "toggled", G_CALLBACK ( on_menu_item_subtitle_stream_toggled ), (gpointer)subtitle_stream.pid );
			if (!channel_set && preferred_language == subtitle_stream.language)
			{
				channel_set = true;
				gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item), true);
			}
			gtk_widget_show(menu_item);
		}
		gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu_item_view_subtitles), menu_widget);
		gtk_widget_show(menu_widget);
	}
}

void MainWindow::channel_change_complete()
{
	update_audio_stream_menu();
	update_view_subtitle_menu();
}

void MainWindow::change_channel_start()
{
	update_epg();
}

void MainWindow::record(gboolean state)
{
	static int context_id = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar), "Recording information");
	Application& application = Application::get_current();
	if (state)
	{
		String message = _("Recording to ");
		message += application.get_recording_file_name();
		gtk_statusbar_push(GTK_STATUSBAR(statusbar), context_id, message.c_str());
	}
	else
	{
		gtk_statusbar_pop(GTK_STATUSBAR(statusbar), context_id);
	}
	
	gtk_toggle_tool_button_set_active(
		GTK_TOGGLE_TOOL_BUTTON(application.get_glade().get_widget("tool_button_record")),
		state);
}

void MainWindow::mute(gboolean state)
{
	Glade& glade = Application::get_current().get_glade();
	
	GtkToggleToolButton* tool_button_mute = GTK_TOGGLE_TOOL_BUTTON(glade.get_widget("tool_button_mute"));
	if (gtk_toggle_tool_button_get_active(tool_button_mute) != state)
	{
		gtk_toggle_tool_button_set_active(tool_button_mute, state);
	}
	
	GtkCheckMenuItem* menu_item_audio_mute = GTK_CHECK_MENU_ITEM(glade.get_widget("menu_item_audio_mute"));
	if (gtk_check_menu_item_get_active(menu_item_audio_mute) != state)
	{
		gtk_check_menu_item_set_active(menu_item_audio_mute, state);
	}
}

void MainWindow::auto_surf(gboolean state)
{
	GtkToggleToolButton* tool_button_auto_surf = GTK_TOGGLE_TOOL_BUTTON(
		Application::get_current().get_glade().get_widget("tool_button_auto_surf"));
	
	if (gtk_toggle_tool_button_get_active(tool_button_auto_surf) != state)
	{
		gtk_toggle_tool_button_set_active(tool_button_auto_surf, state);
	}
}

void MainWindow::broadcast(gboolean state)
{
	GtkToggleToolButton* item = GTK_TOGGLE_TOOL_BUTTON(
		Application::get_current().get_glade().get_widget("tool_button_broadcast"));
	if (gtk_toggle_tool_button_get_active(item) != state)
	{
		gtk_toggle_tool_button_set_active(item, state);
	}
}

void MainWindow::update_epg()
{
	if (get_epg_visible())
	{
		epg_widget->update();
	}
	
	Application& application = Application::get_current();
	String message = application.get_status_message();
	static int context_id = gtk_statusbar_get_context_id(GTK_STATUSBAR(statusbar), "Current channel - event");	
	gtk_statusbar_pop(GTK_STATUSBAR(statusbar), context_id);
	gtk_statusbar_push(GTK_STATUSBAR(statusbar), context_id, message.c_str());
}

void MainWindow::on_timer()
{
	static time_t last_poke_time = 0;

	time_t now = DateTime::now();

	if (now - last_motion_time > CURSOR_HIDE_INTERVAL)
	{
		hide_cursor();
	}
	
	if (now - last_poke_time > POKE_INTERVAL)
	{
		if (is_visible() && !is_minimised()) 		
		{ 		
			send_fake_key();
		}
		last_poke_time = now;
	}
}

gboolean MainWindow::get_controls_visible()
{
	return GTK_WIDGET_VISIBLE(statusbar);
}

gboolean MainWindow::get_epg_visible()
{
	return epg_widget->is_visible();
}

void MainWindow::show()
{
	gtk_widget_show(GTK_WIDGET(window_main));

	Configuration& configuration = Application::get_current().get_configuration();
	
	gint x = configuration.get_int_value("x");
	gint y = configuration.get_int_value("y");

	gtk_window_move(GTK_WINDOW(window_main), x, y);

	gtk_paned_set_position(
		GTK_PANED(vpaned_main),
		configuration.get_int_value("video_position"));	
}

void MainWindow::hide()
{
	gint x = 0;
	gint y = 0;
	
	Configuration& configuration = Application::get_current().get_configuration();
	gtk_window_get_position(GTK_WINDOW(window_main), &x, &y);
	configuration.set_int_value("x", x);
	configuration.set_int_value("y", y);
	
	if (get_epg_visible())
	{
		configuration.set_int_value("video_position",
			gtk_paned_get_position(GTK_PANED(vpaned_main)));
	}
	
	gtk_widget_hide(GTK_WIDGET(window_main));
}

gboolean MainWindow::is_visible()
{
	return GTK_WIDGET_VISIBLE(GTK_WIDGET(window_main));
}

void MainWindow::send_fake_key()
{
	Display* display = GDK_DISPLAY();

	KeyCode keycode = XKeysymToKeycode(display, XK_Shift_L);
	XTestFakeKeyEvent (display, keycode, True, CurrentTime);
	XTestFakeKeyEvent (display, keycode, False, CurrentTime);
    XSync(display, False);
	XFlush(display);
}

void MainWindow::show_scheduled_recordings_dialog()
{
	gboolean fullscreen_workaround = Application::get_current().get_configuration().get_boolean_value("fullscreen_workaround");
	gboolean was_fullscreen = get_is_fullscreen();

	if (fullscreen_workaround && was_fullscreen)
	{
		maximise();
		show_fullscreen(false);
	}
	
	scheduled_recordings_dialog->show();
	
	if (fullscreen_workaround && was_fullscreen)
	{
		show_fullscreen(true);
	}
}

void MainWindow::show_scheduled_recording_dialog()
{
	gboolean fullscreen_workaround = Application::get_current().get_configuration().get_boolean_value("fullscreen_workaround");
	gboolean was_fullscreen = get_is_fullscreen();

	if (fullscreen_workaround && was_fullscreen)
	{
		maximise();
		show_fullscreen(false);
	}
	
	scheduled_recording_dialog->show();	
	
	if (fullscreen_workaround && was_fullscreen)
	{
		show_fullscreen(true);
	}
	
	update_epg();
}

void MainWindow::show_scheduled_recording_dialog(ScheduledRecording& scheduled_recording)
{
	gboolean fullscreen_workaround = Application::get_current().get_configuration().get_boolean_value("fullscreen_workaround");
	gboolean was_fullscreen = get_is_fullscreen();

	if (fullscreen_workaround && was_fullscreen)
	{
		maximise();
		show_fullscreen(false);
	}
	
	scheduled_recording_dialog->show(scheduled_recording);	
	
	if (fullscreen_workaround && was_fullscreen)
	{
		show_fullscreen(true);
	}
	
	update_epg();
}

void MainWindow::show_program_details_dialog (const EpgEvent& event)
{
	int response = 0;
	Application& application = Application::get_current();
	Glade& glade = application.get_glade();

	Configuration& configuration = application.get_configuration();
	
	String start_time = event.get_start_time_text();
	String duration = event.get_duration_text();
	
	gtk_label_set_text(GTK_LABEL(glade.get_widget("label_program_title")), event.get_title().c_str());
	gtk_label_set_text(GTK_LABEL(glade.get_widget("label_program_description")), event.get_description().c_str());
	gtk_label_set_text(GTK_LABEL(glade.get_widget("label_program_start_time")), start_time.c_str());
	gtk_label_set_text(GTK_LABEL(glade.get_widget("label_program_duration")), duration.c_str());
	
	gboolean fullscreen_workaround = configuration.get_boolean_value("fullscreen_workaround");
	gboolean was_fullscreen = get_is_fullscreen();

	if (fullscreen_workaround && was_fullscreen)
	{
		maximise();
		show_fullscreen(false);
	}
	
	response = gtk_dialog_run(GTK_DIALOG(dialog_program_details));
	gtk_widget_hide(GTK_WIDGET(dialog_program_details));
	
	if (fullscreen_workaround && was_fullscreen)
	{
		show_fullscreen(true);
	}
	
	if (response == 1)
	{	
		ScheduledRecording scheduled_recording;
		
		int record_extra_before_seconds = configuration.get_int_value("record_extra_before") * 60;
		int record_extra_after_seconds = configuration.get_int_value("record_extra_after") * 60;

		scheduled_recording.description		= event.get_title();
		scheduled_recording.channel_name	= event.get_channel().name;
		scheduled_recording.type			= RECORDING_TYPE_ONCE_OFF;
		scheduled_recording.start_time		= event.get_start_time() - record_extra_before_seconds;
		scheduled_recording.duration		= event.get_duration() + record_extra_before_seconds + record_extra_after_seconds;
		
		show_scheduled_recording_dialog(scheduled_recording);
	}	
}

void MainWindow::show_preferences_dialog ()
{
	preferences_dialog->show();
	update_epg();
}

void MainWindow::show_cursor()
{
	if (is_cursor_visible == false)
	{
		last_motion_time = time(NULL);
		gdk_window_set_cursor(event_box_video->window, NULL);
		is_cursor_visible = true;
	}
}

void MainWindow::hide_cursor()
{
	gdk_window_set_cursor(event_box_video->window, hidden_cursor);
	is_cursor_visible = false;
}

void MainWindow::set_selected_event(xmlNodePtr event)
{
	epg_widget->set_selected_event(event);
}

void MainWindow::maximise()
{
	gtk_window_maximize(GTK_WINDOW(window_main));
}

gboolean MainWindow::is_minimised() 		
{ 		
	GdkWindowState state = gdk_window_get_state(GTK_WIDGET(window_main)->window); 		
	return state & GDK_WINDOW_STATE_ICONIFIED;
}
