/*
 * Copyright (C) 2008 Michael Lamothe
 *
 * This file is part of Me TV
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Library General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor Boston, MA 02110-1301,  USA
 */

#ifndef __STREAMER_H__
#define __STREAMER_H__

#include "engine.hh"
#include "network.hh"
#include "dvb_si.hh"
#include <stdint.h>

class Stream
{
public:
	void clear()
	{
		video_streams.clear();
		audio_streams.clear();
		subtitle_streams.clear();
		teletext_streams.clear();
	}
	
	std::vector<VideoStream> video_streams;
	std::vector<AudioStream> audio_streams;
	std::vector<SubtitleStream> subtitle_streams;
	std::vector<TeletextStream> teletext_streams;
};

class Streamer
{
private:
	Engine*				engine;
	IO::Channel*		recording_file_stream;
	UdpSocket*			network_stream;
	gboolean			terminate;
	GThread*			stream_thread;
	GStaticRecMutex		mutex;
	static Streamer*	current_streamer;
	GSList*				demuxers;
	gint				CRC32[256];
	guint				pmt_pid;
	gboolean			encode_recording;
	Stream				stream;
	
	void remove_all_demuxers();
	DvbDemuxer& add_pes_demuxer(const String& demux_path, guint pid, dmx_pes_type_t pid_type, const gchar* type_text);

	gboolean is_pid_used(guint pid);
	void write(uint8_t *buf,  int size);
	void stream_loop(IO::Channel& input);
	void write(const gchar* buffer, gsize length);
	static void on_write(uint8_t *buf,  int size, void *priv);
	static gpointer stream_thread_function(Streamer* streamer);
	
	void start_stream();

	void build_pat(gchar* buffer);
	void build_pmt(gchar* buffer);

	void calculate_crc(guchar *p_begin, guchar *p_end);

public:
	Streamer();
	~Streamer();
	
	void start();
	void stop();
	
	gboolean is_recording();
	void start_recording(const String& filename);
	void stop_recording();
	
	gboolean is_broadcasting();
	void enable_broadcasting();
	void disable_broadcasting();

	const Stream& get_stream() const;
};

#endif
