/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: net.c,v 1.16 2000/11/05 17:06:08 kos Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <string.h>
#ifdef HAVE_STRINGS_H
#include <strings.h>
#endif
#include <errno.h>
#include <stdarg.h>
#include <signal.h>
#include <unistd.h>

#include "common.h"
#include "llog.h"
#include "xmalloc.h"
#include "net.h"

FILE *
net_rfp(net_conn_t *c) {
  return c->rfp;
}

FILE *
net_wfp(net_conn_t *c) {
  return c->wfp;
}

int
net_conn_setflags(net_conn_t *c, int f) {
  c->flags |= f;
  return 0;
}

int
net_conn_getflags(net_conn_t *c) {
  return c->flags;
}

int
net_destroy(net_t *n) {
  signal(SIGPIPE, n->oldsigpipe);
  if(n->type == NET_SERVER) {
    shutdown(n->sock, 2);
    close(n->sock);
  }
  xfree(n);  
  return 0;
}

int
net_conn_destroy(net_conn_t *c) {
  shutdown(c->sock, 2);
  fclose(c->wfp);
  fclose(c->rfp);
  xfree(c);
  return 0;
}

net_conn_t *
net_connect(net_t *n) {
  net_conn_t *c;

  if(n->type != NET_CLIENT) 
    return (net_conn_t *)NULL;
    
  c = xcalloc(1, sizeof(net_conn_t));
  
  if((c->sock = socket(PF_INET, SOCK_STREAM, n->proto)) < 0) {
    llog(5, "socket: %s\n", strerror(errno));
    xfree(c);
    return (net_conn_t *)NULL;
  }

  if(connect(c->sock, (struct sockaddr *)&n->sa, sizeof(struct sockaddr_in)) < 0) {
    llog(5, "connect(%s, %d): %s\n", n->addr, n->port, strerror(errno));
    close(c->sock);
    xfree(c);
    return (net_conn_t *)NULL;
  }    

  c->rfp = fdopen(c->sock, "r");
  c->wfp = fdopen(c->sock, "w");
  c->flags = NET_F_DEFAULT;
  return c;
}

net_conn_t *
net_accept(net_t *n) {
  net_conn_t *c;

  if(n->type != NET_SERVER) 
    return (net_conn_t *)NULL;
    
  c = xcalloc(1, sizeof(net_conn_t));

  memset((void *)&c->sa, 0, sizeof(struct sockaddr_in));
  c->salen = sizeof(struct sockaddr_in);
  
  if((c->sock = accept(n->sock, (struct sockaddr *)&c->sa, &c->salen)) < 0) {
    llog(5, "accept: %s\n", strerror(errno));
    return (net_conn_t *)NULL;
  }

  sprintf(c->ident, "%s:%d", inet_ntoa(c->sa.sin_addr), ntohs(c->sa.sin_port));
  c->rfp = fdopen(c->sock, "r");
  c->wfp = fdopen(c->sock, "w");
  c->flags = NET_F_DEFAULT;
  return c;
}

net_t *
net_init(int type, char *addr, int port) {
  net_t *n;
  struct protoent *pe;
  int reuseaddr;
  struct hostent *he;
  
  n = xcalloc(1, sizeof(net_t));

  if(!(pe = getprotobyname("tcp"))) {
    llog(5, "getprotobyname(tcp): failed\n");
    return (net_t *)NULL;
  }

  n->proto = pe->p_proto;
  n->type = type;
  n->port = port;
  strncpy(n->addr, addr, NET_LINELEN);
  
  switch(type & (NET_SERVER|NET_CLIENT)) {
  case NET_SERVER:

    if((n->sock = socket(PF_INET, SOCK_STREAM, pe->p_proto)) < 0) {
      llog(5, "socket: %s\n", strerror(errno));
      return (net_t *)NULL;
    }
    
    n->sa.sin_addr.s_addr = inet_addr(addr);
    n->sa.sin_port = htons(port);
    n->sa.sin_family = AF_INET;
    
    reuseaddr = 1;
    setsockopt(n->sock, SOL_SOCKET, SO_REUSEADDR, (char *)&reuseaddr, sizeof(reuseaddr));
    
    if(bind(n->sock, (struct sockaddr *)&n->sa, sizeof(struct sockaddr_in)) < 0) {
      llog(5, "bind: %s\n", strerror(errno));
      close(n->sock);
      return (net_t *)NULL;
    }
    
    if(listen(n->sock, 32)) {
      llog(5, "listen: %s", strerror(errno));
      close(n->sock);
      return (net_t *)NULL;
    }
    break;

  case NET_CLIENT:
    if(!(he = gethostbyname(addr))) {
      llog(5, "gethostbyname(%s): Failed\n", addr);
      return (net_t *)NULL;
    }
  
    memset((void *)&n->sa, 0, sizeof(struct sockaddr_in));
    n->sa.sin_addr.s_addr = ((struct in_addr *)(he->h_addr))->s_addr;
    n->sa.sin_family = AF_INET;
    n->sa.sin_port = htons(port);
    break;
  }

  n->oldsigpipe = signal(SIGPIPE, SIG_IGN);
  return n;
}

int
net_send(net_conn_t *n, msg *p) {
  int i;
  int ret = 0;
  uchar *cooklhs;
  uchar *cookrhs;
  uint cooklhsl;
  uint cookrhsl;
  uint junk1, junk2;

  cooklhs = xcalloc(1, NET_LINELEN);
  cookrhs = xcalloc(1, NET_LINELEN);
  cooklhsl = cookrhsl = NET_LINELEN;

  for(i=0; i<msg_nelems(p); i++) {
    memset(cooklhs, '\0', cooklhsl);
    memset(cookrhs, '\0', cookrhsl);

    if(textify_len(msg_rhs(p, i), strlen(msg_rhs(p, i))) > cookrhsl) {
      cookrhsl = textify_len(msg_rhs(p, i), strlen(msg_rhs(p, i)));
      cookrhs = xrealloc(cookrhs, cookrhsl);
    }
    textify(msg_rhs(p, i), msg_getlen(p, i), cookrhs, &junk2);

    if(msg_lhs(p, i)) {      
      if(textify_len(msg_lhs(p, i), strlen(msg_lhs(p, i))) > cooklhsl) {
	cooklhsl = textify_len(msg_lhs(p, i), strlen(msg_lhs(p, i)));
	cooklhs = xrealloc(cooklhs, cooklhsl);
      }
      textify(msg_lhs(p, i), msg_getlen(p, i), cooklhs, &junk1);      
      ret += (fprintf(n->wfp, "%s %s", cooklhs, cookrhs) <=0);
    } else {
      ret += (fprintf(n->wfp, "%s ", cookrhs) <=0);
    }
  }
  ret += (fwrite("\n", 1, 1, n->wfp)<=0);
  xfree(cooklhs);
  xfree(cookrhs);

  ret += (fflush(n->wfp)==EOF);
  if(ret)
    llog(3, "net_send: error %d\n", ret);

  return ret;  
}

msg *
net_recv(net_conn_t *n) {
  msg *p;
  uchar buf[NET_LINELEN];
  int len;
  char *tok;
  uchar cooked[P_LINELEN];
  uint cookedl;

  p = msg_new();
  do {
    if(!fgets(buf, NET_LINELEN, n->rfp)) {
      msg_destroy(p);
      return (msg *)NULL;
    }
    len = strlen(buf);

    llog(7, "net_recv: received %d bytes:\n", len);
    llog_hexdump(7, buf, len);

    for(tok=strtok(buf, " \r\n"); tok; tok=strtok(NULL, " \r\n")) {
      memset(cooked, '\0', P_LINELEN);
      untextify(tok, strlen(tok), cooked, &cookedl);
      msg_add(p, cooked, cookedl);
    }
  } while(len==P_LINELEN);

  return p;
}

