/* ws-queries.c
 *
 * Copyright (C) 2004 Vivien Malerba
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "ws-queries.h"
#include "workspace-page.h"
#include <string.h>
#include <glib-object.h>
#include <gtk/gtk.h>
#include <libgnomedb/libgnomedb.h>
#include "query-druid.h"
#include "query-editor.h"

/* 
 * Main static functions 
 */
static void ws_queries_class_init (WsQueriesClass * class);
static void ws_queries_init (WsQueries *ws);
static void ws_queries_dispose (GObject *object);
static void ws_queries_finalize (GObject *object);

/* WorkspacePage interface */
static void         ws_queries_page_init       (WorkspacePageIface *iface);
static gchar       *ws_queries_get_name        (WorkspacePage *iface);
static gchar       *ws_queries_get_description (WorkspacePage *iface);
static GtkWidget   *ws_queries_get_selector    (WorkspacePage *iface);
static GtkWidget   *ws_queries_get_work_area   (WorkspacePage *iface);

/* get a pointer to the parents to be able to call their destructor */
static GObjectClass  *parent_class = NULL;

/* private structure */
struct _WsQueriesPrivate
{
	Workspace      *wspace;
	MgConf         *conf;
	GtkWidget      *selector;
	GtkWidget      *work_area;

	GtkWidget      *notebook;
	GtkWidget      *description;
	GtkWidget      *sql_editor;
	GtkWidget      *query_add;
	GtkWidget      *query_edit;
	GtkWidget      *query_del;
	GtkWidget      *query_exec;	

	GObject        *sel_obj;
};

guint
ws_queries_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (WsQueriesClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) ws_queries_class_init,
			NULL,
			NULL,
			sizeof (WsQueries),
			0,
			(GInstanceInitFunc) ws_queries_init
		};

		static const GInterfaceInfo workspace_page_info = {
			(GInterfaceInitFunc) ws_queries_page_init,
			NULL,
			NULL
		};

		type = g_type_register_static (G_TYPE_OBJECT, "WsQueries", &info, 0);
		g_type_add_interface_static (type, WORKSPACE_PAGE_TYPE, &workspace_page_info);
	}
	return type;
}

static void 
ws_queries_page_init (WorkspacePageIface *iface)
{
	iface->get_name = ws_queries_get_name;
	iface->get_description = ws_queries_get_description;
	iface->get_selector = ws_queries_get_selector;
	iface->get_work_area = ws_queries_get_work_area;
}

static void
ws_queries_class_init (WsQueriesClass * class)
{
	GObjectClass   *object_class = G_OBJECT_CLASS (class);

	parent_class = g_type_class_peek_parent (class);

	object_class->dispose = ws_queries_dispose;
	object_class->finalize = ws_queries_finalize;
}

static void
ws_queries_init (WsQueries *ws)
{
	ws->priv = g_new0 (WsQueriesPrivate, 1);
}

static void ws_queries_initialize (WsQueries *ws);

/**
 * ws_queries_new
 * @conf: a #MgConf object
 *
 * Creates a new WsQueries object
 *
 * Returns: the new object
 */
GObject*
ws_queries_new (Workspace* wspace)
{
	GObject  *obj;
	WsQueries *ws;
	MgConf *conf;

	g_return_val_if_fail (wspace && IS_WORKSPACE (wspace), NULL);
	conf = mg_server_get_conf (workspace_get_server (wspace));

	obj = g_object_new (WS_QUERIES_TYPE, NULL);
	ws = WS_QUERIES (obj);
	ws->priv->conf = conf;
	ws->priv->wspace = wspace;
	g_object_ref (G_OBJECT (conf));

	ws_queries_initialize (ws);

	return obj;
}


static void
ws_queries_dispose (GObject *object)
{
	WsQueries *ws;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_WS_QUERIES (object));

	ws = WS_QUERIES (object);
	if (ws->priv) {
		if (ws->priv->selector) {
			gtk_widget_destroy (ws->priv->selector);
			ws->priv->selector = NULL;
		}

		if (ws->priv->work_area) {
			gtk_widget_destroy (ws->priv->work_area);
			ws->priv->work_area = NULL;
		}

		if (ws->priv->conf) {
			g_object_unref (G_OBJECT (ws->priv->conf));
			ws->priv->conf = NULL;
		}
	}

	/* parent class */
	parent_class->dispose (object);
}

static void
ws_queries_finalize (GObject   * object)
{
	WsQueries *ws;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_WS_QUERIES (object));

	ws = WS_QUERIES (object);
	if (ws->priv) {
		g_free (ws->priv);
		ws->priv = NULL;
	}

	/* parent class */
	parent_class->finalize (object);
}



static void selector_selection_changed_cb (MgSelector *mgsel, GObject *sel_object, WsQueries *ws);
static void query_add_clicked_cb (GtkButton *button, WsQueries *ws);
static void query_edit_clicked_cb (GtkButton *button, WsQueries *ws);
static void query_del_clicked_cb (GtkButton *button, WsQueries *ws);
static void query_exec_clicked_cb (GtkButton *button, WsQueries *ws);
static void query_updated_cb (MgConf *conf, MgQuery *query, WsQueries *ws);
static void
ws_queries_initialize (WsQueries *ws)
{
	GtkWidget *label, *vbox, *nb, *vbox2, *bbox, *button, *wid, *hbox, *vbox3;

	/* Selector part */
	wid = mg_selector_new (ws->priv->conf, NULL,
			       MG_SELECTOR_QUERIES, MG_SELECTOR_COLUMN_TYPE);
	ws->priv->selector = wid;
	g_signal_connect (G_OBJECT (ws->priv->selector), "selection_changed", 
			  G_CALLBACK (selector_selection_changed_cb), ws);

	/* WorkArea part */
	vbox = gtk_vbox_new (FALSE, 0);
	ws->priv->work_area = vbox;

	nb = gtk_notebook_new ();
	gtk_notebook_set_show_border (GTK_NOTEBOOK (nb), FALSE);
	gtk_notebook_set_show_tabs (GTK_NOTEBOOK (nb), FALSE);
	gtk_box_pack_start (GTK_BOX (vbox), nb, TRUE, TRUE, 0);
	gtk_widget_show (nb);
	ws->priv->notebook = nb;
	
	label = gtk_label_new (_("Please select a query from the list on the left,\n"
				 "or create a new one using the 'Add' button below."));
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), label, NULL);
	gtk_widget_show (label);
	
	vbox2 = gtk_vbox_new (FALSE, 5);
	gtk_notebook_append_page (GTK_NOTEBOOK (nb), vbox2, NULL);
	gtk_widget_show (vbox2);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>Description:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, FALSE, FALSE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	label = gtk_label_new (NULL);
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);
	gtk_widget_show (label);
	ws->priv->description = label;

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<b>SQL statement:</b>"));
	gtk_box_pack_start (GTK_BOX (vbox2), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	hbox = gtk_hbox_new (FALSE, 0); /* HIG */
	gtk_box_pack_start (GTK_BOX (vbox2), hbox, TRUE, TRUE, 0);
	gtk_widget_show (hbox);
	label = gtk_label_new ("    ");
	gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);
	gtk_widget_show (label);

	vbox3 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX (hbox), vbox3, TRUE, TRUE, 0);
	gtk_widget_show (vbox3);

	label = gtk_label_new (NULL);
	gtk_label_set_markup (GTK_LABEL (label), _("<small>For information only, to edit, click on the \"Edit\" button.</small>"));
	gtk_box_pack_start (GTK_BOX (vbox3), label, FALSE, FALSE, 0);
	gtk_widget_show (label);
	gtk_misc_set_alignment (GTK_MISC (label), 0., -1);

	wid = gnome_db_editor_new ();
        gnome_db_editor_set_editable (GNOME_DB_EDITOR (wid), FALSE);
        gnome_db_editor_set_highlight (GNOME_DB_EDITOR (wid), FALSE); /* CHANGE ME */
        gtk_box_pack_start (GTK_BOX (vbox3), wid, TRUE, TRUE, 0);
	gtk_widget_show (wid);
	ws->priv->sql_editor = wid;

	bbox = gtk_hbutton_box_new ();
	gtk_button_box_set_layout (GTK_BUTTON_BOX (bbox), GTK_BUTTONBOX_SPREAD);
	gtk_box_pack_start (GTK_BOX (vbox), bbox, FALSE, FALSE, 0);
	gtk_widget_show (bbox);

	button = gtk_button_new_from_stock (GTK_STOCK_ADD);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	ws->priv->query_add = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (query_add_clicked_cb), ws);

	button = gtk_button_new_with_label (_("Edit"));
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	ws->priv->query_edit = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (query_edit_clicked_cb), ws);

	button = gtk_button_new_from_stock (GTK_STOCK_DELETE);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	ws->priv->query_del = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (query_del_clicked_cb), ws);

	button = gtk_button_new_from_stock (GTK_STOCK_INDEX);
	gtk_container_add (GTK_CONTAINER (bbox), button);
	gtk_widget_show (button);
	gtk_widget_set_sensitive (button, FALSE);
	gtk_button_set_label (GTK_BUTTON (button), _("Execute"));
	ws->priv->query_exec = button;
	g_signal_connect (G_OBJECT (button), "clicked",
			  G_CALLBACK (query_exec_clicked_cb), ws);
	
	g_signal_connect (G_OBJECT (ws->priv->conf), "query_updated",
			  G_CALLBACK (query_updated_cb), ws);
}

static void query_info_display_update (MgQuery *query, WsQueries *ws);
static void
selector_selection_changed_cb (MgSelector *mgsel, GObject *sel_object, WsQueries *ws)
{
	if (sel_object && !IS_MG_QUERY (sel_object))
		return;
	
	ws->priv->sel_obj = sel_object;
	gtk_notebook_set_current_page (GTK_NOTEBOOK (ws->priv->notebook), sel_object ? 1 : 0);
	gtk_widget_set_sensitive (ws->priv->query_exec, sel_object ? TRUE : FALSE);
	gtk_widget_set_sensitive (ws->priv->query_del, sel_object ? TRUE : FALSE);
	gtk_widget_set_sensitive (ws->priv->query_edit, sel_object ? TRUE : FALSE);
	query_info_display_update ((MgQuery*) sel_object, ws);
}

static void
query_updated_cb (MgConf *conf, MgQuery *query, WsQueries *ws)
{
	if ((GObject *)query == ws->priv->sel_obj)
		query_info_display_update (query, ws);
}

static void
dialog_exec_response_cb (GtkDialog *dlg, gint response, GObject *obj)
{
	gtk_widget_destroy (GTK_WIDGET (dlg));
}

static void
query_add_clicked_cb (GtkButton *button, WsQueries *ws)
{
	GtkWidget *druid;

	druid = query_druid_new (ws->priv->conf);
	gtk_widget_show (druid);
	gnome_druid_construct_with_window (GNOME_DRUID (druid), _("New Query"), NULL, TRUE, NULL);
}

static void
query_edit_clicked_cb (GtkButton *button, WsQueries *ws)
{
	GtkWidget *dlg, *editor;
	GObject *sel_query = ws->priv->sel_obj;
	gchar *title;

	if (!sel_query)
		return;

	editor = query_editor_new (MG_QUERY (sel_query));
	gtk_widget_show (editor);
	title = g_strdup_printf (_("Edition of query '%s'"), mg_base_get_name (MG_BASE (sel_query)));
	dlg = gtk_dialog_new_with_buttons (title, NULL, 0, GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE, NULL);
	g_free (title);
	gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dlg)->vbox), editor, TRUE, TRUE, 6);
	gtk_widget_set_size_request (dlg, 550, 430);
	gtk_widget_show (dlg);
	g_signal_connect (G_OBJECT (dlg), "response",
			  G_CALLBACK (dialog_exec_response_cb), NULL);
}

static void
query_del_clicked_cb (GtkButton *button, WsQueries *ws)
{
	GObject *sel_query = ws->priv->sel_obj;
	GtkWidget *dlg;

	if (!sel_query)
		return;

	dlg = gtk_message_dialog_new (NULL, GTK_DIALOG_MODAL, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO,
				      _("Do you want to delete the '%s' query?"), 
				      mg_base_get_name (MG_BASE (sel_query)));
	if (gtk_dialog_run (GTK_DIALOG (dlg)) == GTK_RESPONSE_YES) 
		mg_base_nullify (MG_BASE (sel_query));

	gtk_widget_destroy (dlg);
}

/* extra action for the grid widgets */
static void extra_action_close_cb (GtkAction *action, GtkWidget *window);
static GtkActionEntry extra_actions[] = {
	{ "Close", GTK_STOCK_CLOSE, "_Close", NULL, "Close this window", G_CALLBACK (extra_action_close_cb)}
};

static void
query_exec_clicked_cb (GtkButton *button, WsQueries *ws)
{
	GtkWidget *window;
	MgQuery *query;
	gint result = GTK_RESPONSE_ACCEPT;
	MgContext *context;
	GtkWidget *parent_window = ws->priv->work_area;
		
	query = (MgQuery *) ws->priv->sel_obj;

	while (parent_window && !GTK_IS_WINDOW (parent_window)) 
		parent_window = gtk_widget_get_parent (parent_window);

	context = mg_entity_get_exec_context (MG_ENTITY (query));
	if (mg_context_needs_user_input (context)) {
		GtkWidget *dlg;
		MgForm *sform;
		
		dlg = mg_form_new_in_dialog (ws->priv->conf, 
					     context, GTK_WINDOW (parent_window),
					     _("Values to be filled"), 
					     _("<big><b>Required values:</b></big>\n"
					       "<small>The following values are required to "
					       "execute the query.</small>"));
		sform = g_object_get_data (G_OBJECT (dlg), "form");
		mg_form_set_entries_auto_default (sform, TRUE);
		
		gtk_widget_show (dlg);
		result = gtk_dialog_run (GTK_DIALOG (dlg));
		gtk_widget_destroy (dlg);
	}
	
	if (result == GTK_RESPONSE_ACCEPT) {
		if (mg_query_is_modif_query (query)) /* modification query */
			mg_util_query_execute_modif (query, context, TRUE, TRUE, TRUE, parent_window, NULL, NULL);
		else { /* non modification query */
			if (mg_query_is_select_query (query)) {
				GtkWidget *grid, *vbox, *toolbar;
				MgTarget *target;
				guint mode = 0;
				gchar *str;
				GtkActionGroup *grid_agroup, *extra_agroup;
				GtkUIManager *ui;

				static const gchar *grid_actions =
					"<ui>"
					"  <toolbar  name='ToolBar'>"
					"    <toolitem action='WorkWidgetNew'/>"
					"    <toolitem action='WorkWidgetDelete'/>"
					"    <toolitem action='WorkWidgetUndelete'/>"
					"    <toolitem action='WorkWidgetCommit'/>"
					"    <toolitem action='WorkWidgetReset'/>"
					"    <separator/>"
					"    <toolitem action='WorkWidgetFirstChunck'/>"
					"    <toolitem action='WorkWidgetPrevChunck'/>"
					"    <toolitem action='WorkWidgetNextChunck'/>"
					"    <toolitem action='WorkWidgetLastChunck'/>"
					"    <separator/>"
					"    <toolitem action='Close'/>"					
					"  </toolbar>"
					"</ui>";

				/* FIXME: the first target is the one that gets modified */
				GSList *targets = mg_query_get_targets (query);
				target = targets ? (MgTarget *) targets->data : NULL;
				if (targets) g_slist_free (targets);

				grid = mg_work_grid_new (query, NULL/*target*/);
				g_object_set (G_OBJECT (grid), "title_visible", FALSE, NULL);
				mg_work_widget_alldata_show_actions (MG_WORK_WIDGET (grid), FALSE);
				
				str = g_strdup_printf (_("Execution of query '%s'"), mg_base_get_name (MG_BASE (query)));
				window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
				gtk_window_set_title (GTK_WINDOW (window), str);
				g_free (str);				

				vbox = gtk_vbox_new (FALSE, 0);
				gtk_container_add (GTK_CONTAINER (window), vbox);
				gtk_widget_show (vbox);

				ui = gtk_ui_manager_new ();
				grid_agroup = mg_work_widget_get_actions_group (MG_WORK_WIDGET (grid));
				extra_agroup = gtk_action_group_new ("ExtraActions");
				gtk_action_group_add_actions (extra_agroup, extra_actions, G_N_ELEMENTS (extra_actions), window);
				gtk_ui_manager_insert_action_group (ui, grid_agroup, 0);
				gtk_ui_manager_insert_action_group (ui, extra_agroup, 0);
				gtk_ui_manager_add_ui_from_string (ui, grid_actions, -1, NULL);
				toolbar = gtk_ui_manager_get_widget (ui, "/ToolBar");
				gtk_box_pack_start (GTK_BOX (vbox), toolbar, FALSE, FALSE, 0);
				gtk_widget_show (toolbar);
				
				gtk_box_pack_start (GTK_BOX (vbox), grid, TRUE, TRUE, 0);
				gtk_widget_show (grid);
				gtk_widget_set_size_request (grid, 800, 300);
				if (target)
					mode = MG_ACTION_ASK_CONFIRM_DELETE | 
						MG_ACTION_ASK_CONFIRM_UPDATE | 
						MG_ACTION_ASK_CONFIRM_INSERT;
				
				mg_work_widget_bind_to_context_all (MG_WORK_WIDGET (grid), context);
				mg_work_widget_run (MG_WORK_WIDGET (grid), mode);
				gtk_widget_show (window);
			}
			else {
				/* the query can't be parsed! */
				GtkWidget *dlg;
				gint result;
				gchar *msg;
				const gchar *sql = mg_query_get_sql_text (query);
				msg = g_strdup_printf (_("<b><big>Execute the following query ?</big></b>\n"
							 "<small>This query can't be parsed: it may contain some "
							 "syntax or grammar which is dependant on the type of database "
							 "used, or may contain some error.</small>\n\n%s"), sql);
				dlg = gtk_message_dialog_new (GTK_WINDOW (parent_window), 0,
							      GTK_MESSAGE_QUESTION,
							      GTK_BUTTONS_YES_NO, msg);
				g_free (msg);
				gtk_label_set_use_markup (GTK_LABEL (GTK_MESSAGE_DIALOG (dlg)->label), TRUE);
				result = gtk_dialog_run (GTK_DIALOG (dlg));
				gtk_widget_destroy (dlg);
				if (result == GTK_RESPONSE_YES) {
					GError *error = NULL;
					MgResultSet *rs;

					rs = mg_server_do_query (mg_conf_get_server (ws->priv->conf), sql, 
								 MG_SERVER_QUERY_SQL, &error);
					if (error) {
						GtkWidget *dlg;
						gchar *message;
						
						message = g_strdup (error->message);
						g_error_free (error);
						dlg = gtk_message_dialog_new (GTK_WINDOW (parent_window), 0,
									      GTK_MESSAGE_ERROR,
									      GTK_BUTTONS_CLOSE,
									      message);
						g_free (message);
						gtk_dialog_run (GTK_DIALOG (dlg));
						gtk_widget_destroy (dlg);
					}
					else {
						if (rs) {
							GtkWidget *grid, *vbox, *toolbar;
							GtkActionGroup *extra_agroup;
							GtkUIManager *ui;
							static const gchar *grid_actions =
								"<ui>"
								"  <toolbar  name='ToolBar'>"
								"    <toolitem action='Close'/>"					
								"  </toolbar>"
								"</ui>";

							grid = gnome_db_grid_new_with_model (mg_resultset_get_data_model (rs));
							gtk_widget_set_size_request (grid, 800, 300);
							g_object_unref (G_OBJECT (rs));

							window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
							gtk_window_set_title (GTK_WINDOW (window), _("Execution results"));
							
							vbox = gtk_vbox_new (FALSE, 0);
							gtk_container_add (GTK_CONTAINER (window), vbox);
							gtk_widget_show (vbox);

							ui = gtk_ui_manager_new ();
							extra_agroup = gtk_action_group_new ("ExtraActions");
							gtk_action_group_add_actions (extra_agroup, extra_actions, 
										      G_N_ELEMENTS (extra_actions), window);
							gtk_ui_manager_insert_action_group (ui, extra_agroup, 0);
							gtk_ui_manager_add_ui_from_string (ui, grid_actions, -1, NULL);
							toolbar = gtk_ui_manager_get_widget (ui, "/ToolBar");
							gtk_box_pack_start (GTK_BOX (vbox), toolbar, FALSE, FALSE, 0);
							gtk_widget_show (toolbar);

							gtk_box_pack_start (GTK_BOX (vbox), grid, TRUE, TRUE, 0);
							gtk_widget_show (grid);
							gtk_widget_show (window);
						}
					}
				}
			}
		}
	}
	
	/* get rid of that context object */
	g_object_unref (G_OBJECT (context));
}

static void
extra_action_close_cb (GtkAction *action, GtkWidget *window)
{
	gtk_widget_destroy (window);
}



static void
query_info_display_update (MgQuery *query, WsQueries *ws)
{
	const gchar *str = NULL;
	gchar *title = NULL;
	
	if (query)
		str = mg_base_get_description (MG_BASE (query));
	if (str && *str) 
		gtk_label_set_text (GTK_LABEL (ws->priv->description), str);
	else
		gtk_label_set_text (GTK_LABEL (ws->priv->description), "---");

	/* global title update */
	title = ws_queries_get_description (WORKSPACE_PAGE (ws));
	g_signal_emit_by_name (G_OBJECT (ws), "description_changed", title);
	g_free (title);
	
	/* SQL version of the query */
	if (query) {
		gchar *sql;
		GError *error = NULL;
		sql = mg_renderer_render_as_sql (MG_RENDERER (query), NULL,
						 MG_RENDERER_EXTRA_PRETTY_SQL, &error);
		if (sql) {
			gnome_db_editor_set_text (GNOME_DB_EDITOR (ws->priv->sql_editor), sql, -1);
			g_free (sql);
		}
		else {
			if (error) {
				str = error->message;
				gnome_db_editor_set_text (GNOME_DB_EDITOR (ws->priv->sql_editor), str, -1);
				g_error_free (error);
			}
			else
				gnome_db_editor_set_text (GNOME_DB_EDITOR (ws->priv->sql_editor),
							  _("Non reported error"), -1);
		}
	}
	else
		gnome_db_editor_set_text (GNOME_DB_EDITOR (ws->priv->sql_editor), "", -1);	
}



/* 
 * WorkspacePage interface implementation
 */
static gchar *
ws_queries_get_name (WorkspacePage *iface)
{
	g_return_val_if_fail (iface && IS_WS_QUERIES (iface), NULL);
	g_return_val_if_fail (WS_QUERIES (iface)->priv, NULL);

	return g_strdup_printf (_("Queries"));
}

static gchar *
ws_queries_get_description (WorkspacePage *iface)
{
	gchar *str = NULL;
	WsQueries *ws;

	g_return_val_if_fail (iface && IS_WS_QUERIES (iface), NULL);
	g_return_val_if_fail (WS_QUERIES (iface)->priv, NULL);

	ws = WS_QUERIES (iface);
	if (ws->priv->sel_obj) {
		const gchar *cstr;
		 cstr =  mg_base_get_name (MG_BASE (ws->priv->sel_obj));
		 str = g_strdup_printf ("Query: <b>%s</b>", (cstr && *cstr) ? cstr : "---");
	}
	else
		str = g_strdup_printf ("<b>%s</b>", _("No query selected"));

	return str;
}

static GtkWidget *
ws_queries_get_selector (WorkspacePage *iface)
{
	g_return_val_if_fail (iface && IS_WS_QUERIES (iface), NULL);
	g_return_val_if_fail (WS_QUERIES (iface)->priv, NULL);

	return WS_QUERIES (iface)->priv->selector;
}
static GtkWidget *
ws_queries_get_work_area (WorkspacePage *iface)
{
	g_return_val_if_fail (iface && IS_WS_QUERIES (iface), NULL);
	g_return_val_if_fail (WS_QUERIES (iface)->priv, NULL);

	return WS_QUERIES (iface)->priv->work_area;
}
