/*
------------------------------------------------------------------------------
MetaCam - Extract EXIF information from digital camera files, with
support for Vendor specific blocks.
Copyright (C) 2000 Daniel Stephens (daniel@cheeseplant.org)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
------------------------------------------------------------------------------
*/

#include <iostream.h>
#include <fstream.h>
#include <stdio.h>
#include <errno.h>
#include <netinet/in.h>
#include <string.h>
#include <math.h>

#include <map>
#include <vector>
#include <string>

#include <unistd.h>
#include <getopt.h>

#include "metacam.h"
#include "dpyfuncs.h"

static const char *rcsid="$Id: metacam.cc,v 1.6 2001/12/25 20:27:42 daniel Exp $";

static void
Process_IFD(const IFD &ifd, tagmap &tag_map)
{
    for (int i=0; i<ifd.Entries(); ++i) {
	const IFDEntry &e = ifd[i];
	    idpair id(e.Tag(), e.Type());
	    tag_map[id] = new IFDEntry(e);
    }
}

option opts[] = {
    {"all", 0, NULL, 'a'},
    {"help", 0, NULL, 'h'},
    {"verbose", 0, NULL, 'v'},
    {0,0,0,0}
};

const char *resolution_unit = "Inch";

void
Display_Tags(tagmap &tag_map, knowntag *known, int verbose)
{
    resolution_unit = "Inch"; // Reset

    tagmap::iterator iter;
    for (int i=0; known[i].tag; ++i) {
	idpair id(known[i].tag, known[i].type);
	iter = tag_map.find(id);
	if (iter == tag_map.end()) {continue;}
	IFDEntry *e = (*iter).second;
	if (!e) continue;
	tag_map[id] = 0;

	if (known[i].verbose > verbose) continue;

	if (known[i].func) {
	    known[i].func(cout, known[i].name, *e);
	} else {
	    cout << known[i].name << ": ";
	    e->Output_Value(cout);
	    cout << endl;
	}

	delete e;
    }

    if (verbose > 1) {
	for (iter=tag_map.begin(); iter != tag_map.end(); ++iter) {
	    if (!(*iter).second) continue;
	    IFDEntry *e = (*iter).second ;
	    cout << "    TAG 0x" << hex << e->Tag() << dec << "-" << e->Type() << ":";
	    e->Output_Value(cout);
	    cout << endl;
	} 
    }

    
}

void
Clear_Tag_Map(tagmap &tag_map)
{
    tagmap::iterator iter;
    for (iter = tag_map.begin(); iter != tag_map.end(); ++iter) {
	if ((*iter).second) {
	    delete (*iter).second;
	    (*iter).second = 0;
	}
    }
    tag_map.clear();
}

extern int
main(int argc, char *argv[])
{
    int verbose_option=0;
    int help_option=0;
    int all_option=0;

    while (1) {
	int oidx;
	int opt = getopt_long(argc, argv, "hav", opts, &oidx);
	if (opt == -1) break;

	switch (opt) {
	case 'h': help_option = 1; break;
	case 'v': verbose_option = 1; break;
	case 'a': all_option = 1; break;
	}
    }


    if ((optind >= argc) || (help_option)) {
	cerr << "Usage: " << argv[0] << " <options> [filename(s)...]" << endl;
	cerr << endl
	     << "       -h,--help           Display this help" << endl
	     << "       -v,--verbose        Display unknown tags too" << endl
	     << "       -a,--all            Display ALL tags (implies -v)" << endl
	     << endl;
	if (!help_option) exit(2);
	exit(0);
    }

    int verbose_level = 0;
    if (verbose_option) verbose_level = 1;
    if (all_option) verbose_level = 2;

    for (int o=optind; o<argc; ++ o) {
	tagmap tag_map;

	ifstream is;
	is.open(argv[o], ios::binary || ios::in);
	if (!is.good()) {
	    int err = errno;
	    cerr << "Failed to open file '" << argv[o] << "': " << strerror(err) << endl;
	    continue;
	}

	cout << "File: " << argv[o] << endl;

	unsigned char header[2];
	is.read(header, 2);
	if (is.gcount() != 2) {
	    int err = errno;
	    cerr << "Read failed: " << strerror(err) << endl;
	    continue;
	}

	int ofs=0;

	if ((header[0] == 0xFF) && (header[1] == 0xD8)) {
	    // JPEG File
	    // Search through file for Exif header....
	    
	    int curofs = 2;
	    unsigned char scanbuf[1024];
	    int startofs = 0; // Where do we start reading into the buffer
	    bool failed_scan = false;
	    bool found_exif = true;
	    
	    while (!is.eof()) {
		if (!is.good()) {
		    int err = errno;
		    cerr << "Failure while scanning: " 
			 << strerror(err) << endl;
		    failed_scan = true;
		    break;
		}
		is.read(&scanbuf[startofs], 1024 - startofs);
		int got = is.gcount();
		if (got == 0) continue;
		if (got < 0) {
		    int err = errno;
		    cerr << "Failure while scanning: " 
			 << strerror(err) << endl;
		    failed_scan = true;
		    break;
		}

		// Pretend we read the WHOLE buffer
		got += startofs;

		// If we didn't get enough, try to get more
		if (got < 4) {
		    startofs = got;
		    continue;
		}

		for (int i=0; i<got-3; ++i) {
		    if (scanbuf[i] == 'E') {
			if ((scanbuf[i+1] == 'x') &&
			    (scanbuf[i+2] == 'i') &&
			    (scanbuf[i+3] == 'f')) {
			    
			    ofs = curofs + i + 6;
			    found_exif = true;
			}
		    }
		}

		if (found_exif) break;

		// Determine how much of buffer to keep (in case Exif
		// straddles buffer edges)

		int keep = 0;
		if (scanbuf[got-1] == 'E') {
		    keep = 1;
		} else if (scanbuf[got-2] == 'E') {
		    keep = 2;
		} else if (scanbuf[got-3] == 'E') {
		    keep = 3;
		}

		if (keep) {
		    for (int i=0; i<keep; ++i) 
			scanbuf[i] = scanbuf[got-keep+i];
		}

		curofs += (got - keep);
		startofs = keep;
	    }

	    if (failed_scan) continue;
	    is.clear(is.rdstate() & (~ios::eofbit));
	}

	tiffFile tiff(is, ofs);

	tiffUNSIGNED exif_ifd_start = 0;
	tiffUNSIGNED maker_ifd_start = 0;
	knowntag *maker_known_table = 0;

	// Get IFD0
	IFD ifd = tiff.Get_IFD();
	Process_IFD(ifd, tag_map);
	ifd = ifd.Next_IFD();

	// Look for the EXIF IFD
	idpair exif_id(0x8769, 4);
	tagmap::iterator iter = tag_map.find(exif_id);
	if (iter != tag_map.end() && ((*iter).second)) {
	    vector<unsigned long> uv = (*iter).second->Get_UVALUES();
	    exif_ifd_start = uv[0];
	    delete (*iter).second;
	    (*iter).second = 0;
	}

	string maker_name; // For later use

	idpair makername_id(0x010f, 2);
	iter = tag_map.find(makername_id);
	if (iter != tag_map.end() && ((*iter).second)) {
	    vector<string> v = (*iter).second->Get_STRINGS();
	    maker_name = v[0];
	}

	Display_Tags(tag_map, main_known, verbose_level);
	Clear_Tag_Map(tag_map);

	if (exif_ifd_start) {
	    IFD exif = tiff.Get_IFD(exif_ifd_start);
	    Process_IFD(exif, tag_map);

	    // Look for the MakerNote

	    idpair makernote_id(0x927C, 7);
	    iter = tag_map.find(makernote_id);
	    if (iter != tag_map.end() && ((*iter).second)) {
		if (strncmp(maker_name.data(), "NIKON", 5) == 0) {
		    maker_ifd_start = (*iter).second->Offset();
		    maker_known_table = nikon_known;
		    delete (*iter).second;
		    (*iter).second = 0;
		} else if (strncmp(maker_name.data(), "OLYMP", 5) == 0) {
		    maker_ifd_start = (*iter).second->Offset() + 0x08;
		    maker_known_table = olympus_known;
		    delete (*iter).second;
		    (*iter).second = 0;
		} else if (strncasecmp(maker_name.data(), "CASIO", 5) == 0) {
		    maker_ifd_start = (*iter).second->Offset();
		    maker_known_table = casio_known;
		    delete (*iter).second;
		    (*iter).second = 0;
		} else if (strncasecmp(maker_name.data(), "CANON", 5) == 0) {
		    maker_ifd_start = (*iter).second->Offset();
		    maker_known_table = canon_known;
		    delete (*iter).second;
		    (*iter).second = 0;
		}
	    }

	    cout << "  EXIF Fields --------------------------------" << endl;

	    Display_Tags(tag_map, exif_known, verbose_level);
	    Clear_Tag_Map(tag_map);
	}

	if (maker_ifd_start && maker_known_table) {
	    IFD maker = tiff.Get_IFD(maker_ifd_start);
	    Process_IFD(maker, tag_map);

	    cout << "  Manufacturer Fields ------------------------" << endl;
	    Display_Tags(tag_map, maker_known_table, verbose_level);
	    Clear_Tag_Map(tag_map);
	}
	
	cout << endl;
    }
}



// Table of tags I know about
// 1000000+ are tags in the odd Nikon data structure
// Tags at the bottom are 'real' tags I know about and want to ignore

