/*
 *  mod_bt - Making Things Better For Seeders
 *  Copyright 2006 Tyler MacDonald <tyler@yi.org>
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/* file_pool.c:
 * we may be trying to manipulate thousands of files at a time, but
 * we don't want to have them all open at once. provide a
 * oldest-use-first-out pool with read/write operations on files.
 *
 * we implement this as a linked list so that we can quickly bubble files
 * up to the top as they are used.
 */

#define BTP_FILE_POOL_PRIVATE
#include <libbtutil/const.h>
#include <libbtutil/apr.h>
#include <libbtutil/types/bt_llist.h>
#include <libbtpeer/types/btp_file_pool.h>

#include <apr.h>
#include <apr_file_io.h>
#include <apr_pools.h>
#include <apr_errno.h>
#include <stdint.h>

void btp_file_pool_use(btp_file_pool* fp, btp_file_pool_file* f) {
    bt_llist_e_link_first(f->e);
    return;
}

void btp_file_pool_add_record(btp_file_pool* fp, btp_file_pool_file* f) {
    bt_llist_e* e = bt_llist_e_new(fp, f, sizeof(btp_file_pool_file));
    f->e = e;
    bt_llist_e_link_first(e);
    return;
}

void btp_file_pool_del_record(btp_file_pool* fp, btp_file_pool_file* f) {
    apr_file_close(f->fh);
    bt_llist_e_unlink(f->e);
    return;
}

apr_status_t btp_file_pool_add(
    btp_file_pool* fp, const char* path, apr_file_t **fh
) {
    apr_file_t *rfh = NULL;
    char err[255];
    apr_status_t rv = apr_file_open(
        &rfh, path,
        APR_READ | APR_WRITE | APR_CREATE, APR_FPROT_OS_DEFAULT,
        fp->p
    );
    
    if(rv == APR_SUCCESS) {
        btp_file_pool_file *f = apr_palloc(fp->p, sizeof(btp_file_pool_file));
        BT_STRCPY(f->path, path);
        f->fh = rfh;
        *fh = rfh;
        if(fp->n >= fp->max) btp_file_pool_del_record(
            fp, (btp_file_pool_file*)fp->last->d
        );
        btp_file_pool_add_record(fp, f);
    } else {
        fprintf(
            stderr, "btp_file_pool_add(%s): %s\n",
            path, apr_strerror(rv, err, sizeof(err))
        );
    }

    return rv;
}

apr_status_t btp_file_pool_get(
    btp_file_pool* fp, const char* path, apr_file_t **fh
) {
    bt_llist_e* e;
    btp_file_pool_file* c;
    
    for(e=fp->first; e; e=e->next) {
        c = (btp_file_pool_file*)e->d;
        if(!strcmp(path, c->path)) {
            btp_file_pool_use(fp, c);
            *fh = c->fh;
            return APR_SUCCESS;
        }
    }
    
    return btp_file_pool_add(fp, path, fh);
}
