
#include <libbtutil.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <malloc.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>

#include <apr.h>
#include <apr_pools.h>
#include <apr_errno.h>
#include <apr_strings.h>
#include <apr_file_info.h>
#include <apr_file_io.h>

static char errorstr[BT_SHORT_STRING];
static char* bt_metainfo_load(apr_pool_t* p, const char* path);

int bt_metainfo_pos2file(bt_metainfo* info, bt_off_t pos, bt_off_t* offset) {
    int i;
    bt_off_t offset_i = pos;
    
    for(i=0; i<info->file_count; i++) {
        if(offset_i < info->files[i].length)
            break;
        offset_i -= info->files[i].length;
    }
    
    if(i >= info->file_count) {
        fprintf(
            stderr,
            "bt_metainfo_pos2file: offset " BT_OFF_T_FMT " > torrent size "
            BT_SIZE_T_FMT "\n",
            pos, info->total_size
        );
        return -1;        
    }

    if(offset) *offset = offset_i;
    return i;
}

bt_metainfo* bt_metainfo_parse(apr_pool_t* p, const char* path) {
    char*               buf;
    bt_bcode            bcode, *info, *list, *val;
    int                 i;
    apr_pool_t*         temp = NULL;
    bt_metainfo*        metainfo;
    int rv;
       
    if(apr_pool_create(&temp, p) != APR_SUCCESS) {
        fprintf(stderr, "Error creating metainfo's temporary pool!\n");
        goto err;
    }

    metainfo = apr_pcalloc(p, sizeof(bt_metainfo));
    strncpy(metainfo->torrent, path, sizeof(metainfo->torrent) - 1);

    if(!(buf = bt_metainfo_load(temp, path)))
        goto err;
    
    /* Parse bencoded infos */
    if((rv = bt_bcode_decode(temp, buf, &bcode, NULL)) != BT_BCODE_SUCCESS) {
        fprintf(
            stderr, "Error while parsing bcoded data: %s\n",
            bt_bcode_error(rv)
        );
        goto err;
    }

    /* Get info hash */
    if(!(info = bt_bcode_find(&bcode, "info"))) {
        fprintf(stderr, "Could not find \"info\" dictionary\n");
        goto err;
    }
    
    BT_SHA1(
        (uint8_t*)info->begin, (long)info->end - (long)info->begin,
        metainfo->hash
    );


    if(!(val = bt_bcode_find(&bcode, "announce" ))) {
        fprintf(stderr, "No \"announce\" entry\n");
        goto err;
    }
    
    if(
        bt_url2args(
            val->val.s.s, metainfo->tracker_address, &(metainfo->tracker_port),
            metainfo->tracker_announce
        )
        != APR_SUCCESS
    )
        goto err;

    /* Piece length */
    if(!(val = bt_bcode_find(info, "piece length"))) {
        fprintf(stderr, "No \"piece length\" entry\n");
        goto err;
    }
    
    metainfo->piece_size = val->val.i;

    /* Hashes */
    val = bt_bcode_find(info, "pieces");
    if(val->val.s.i % BT_INFOHASH_LEN) {
        fprintf(
            stderr, "Invalid \"pieces\" string (size is %"APR_SIZE_T_FMT")\n",
            val->val.s.i
        );
        goto err;
    }
    
    metainfo->piece_count = val->val.s.i / BT_INFOHASH_LEN;
    metainfo->pieces = apr_pmemdup(p, val->val.s.s, val->val.s.i);

    /* TODO add more tests so we don't crash on weird files */

    metainfo->total_size = 0;
    
    if((val = bt_bcode_find(&bcode, "creation date")))
        metainfo->creation_date = val->val.i;
    
    if((list = bt_bcode_find(info, "files"))) {
        /* multiple files */
        int j;
        char *name;

        val = bt_bcode_find(info, "name");
        name = bt_strcat_utf8(temp, "", val->val.s.s);
        BT_STRCPY(metainfo->name, name);

        metainfo->file_count = list->val.l.count;
        metainfo->files      = apr_pcalloc(
            p, metainfo->file_count * sizeof(bt_metainfo_finfo)
        );

        for(i=0; i<list->val.l.count; i++) {
            val = bt_bcode_find(&list->val.l.vals[i], "path");
            name = bt_strcat_utf8(temp, "", metainfo->name);
            
            for(j=0; j<val->val.l.count; j++) {
                name = bt_strcat_utf8(temp, name, "/");
                name = bt_strcat_utf8(temp, name, val->val.l.vals[j].val.s.s);
                BT_STRCPY(metainfo->files[i].name, name);
            }
            
            val = bt_bcode_find(&list->val.l.vals[i], "length");
            metainfo->files[i].length  = val->val.i;
            metainfo->total_size       += val->val.i;
        }
    } else {
        /* single file */
        char* name;
        
        metainfo->file_count    = 1;
        metainfo->files         = apr_pcalloc(p, sizeof(bt_metainfo_finfo));

        val = bt_bcode_find(info, "name");
        name = bt_strcat_utf8(p, "", val->val.s.s);
        
        BT_STRCPY(metainfo->files[0].name, name);
        BT_STRCPY(metainfo->name, name);
        
        val = bt_bcode_find(info, "length");
        metainfo->files[0].length  = val->val.i;
        metainfo->total_size       += val->val.i;
    }

    if(
        (uint64_t)metainfo->piece_count !=
        (metainfo->total_size + metainfo->piece_size - 1)
            / metainfo->piece_size
    ) {
        fprintf(stderr, "Size of hashes and files don't match\n");
        goto err;
    }
    
    apr_pool_destroy(temp);
    return metainfo;
    
    err:
    if(temp) apr_pool_destroy(temp);
    if(p) apr_pool_destroy(p);
    return NULL;
}

static inline bt_bcode* bt_metainfo_bcode_path(
    bt_metainfo* metainfo, const char* path_s, apr_pool_t* p
) {
    int len = 0;
    char* work = apr_pstrdup(p, path_s);
    char* pos;
    char* hop;
    int ret;
    bt_bcode* path = bt_bcode_new_list(p, BT_TINY_STRING);
    
    pos = work + strlen(metainfo->name) + 1;

    while(*pos == '/')
        pos++;
    
    while((hop = strchr(pos, '/'))) {
        while(*hop == '/') {
            *hop = '\0';
            hop++;
        }
        
        if(*pos) {
            if(
                (
                    ret = bt_bcode_list_add(
                        path, len, bt_bcode_new_str(p, pos)
                    )
                ) != APR_SUCCESS
            ) {
                fprintf(
                    stderr, "bt_metainfo_bcode_path: failed to add "
                    "path element \"%s\" for path \"%s\"\n",
                    pos, path_s
                );
                return NULL;
            }
            
            len++;

            if(len >= BT_TINY_STRING) {
                fprintf(
                    stderr,
                    "bt_metainfo_bcode_path: more than %d path "
                    "elements in %s\n",
                    BT_TINY_STRING, path_s
                );
                return NULL;
            }
        }
        
        pos = hop;
    }
    
    if(*pos) {
        bt_bcode_list_add(path, len, bt_bcode_new_str(p, pos));
        len++;
    }
    
    if(len) {
        path->val.l.count = len;
        return path;
    } else {
        fprintf(
            stderr,
            "bt_metainfo_bcode_path: path \"%s\" doesn't have any elements!\n",
            path_s
        );
        return NULL;
    }
}

static inline apr_status_t bt_metainfo_bcode_file(
    bt_metainfo* metainfo,
    bt_metainfo_finfo finfo, bt_bcode* bcode, apr_pool_t* p, unsigned int pos
) {
    int ret;
    bt_bcode* file = bt_bcode_new_dict(p, 2);
    
    if(
        (ret = bt_bcode_dict_add(
            p, file, 0, "length", bt_bcode_new_int(p, finfo.length)
        ))
        == APR_SUCCESS
    ) {
        if(
            (ret = bt_bcode_dict_add(
                p, file, 1, "path",
                bt_metainfo_bcode_path(metainfo, finfo.name, p)
            ))
            == APR_SUCCESS
        )
            ret = bt_bcode_list_add(bcode, pos, file);
        else
            fprintf(
                stderr,
                "bt_metainfo_bcode_file: Attempting to "
                "bcode file \"%s\" failed\n",
                finfo.name
            );
    }
    
    return ret;
}

static inline apr_status_t bt_metainfo_bcode_files(
    bt_metainfo* metainfo, bt_bcode* bcode, apr_pool_t* p, unsigned int pos
) {
    apr_status_t ret = APR_ENOENT;
    unsigned int i;
    bt_bcode* files = bt_bcode_new_list(p, metainfo->file_count);
    
    for(i=0; i<metainfo->file_count; i++) {
        if(
            (ret =  bt_metainfo_bcode_file(
                metainfo, metainfo->files[i], files, p, i
            ))
            != APR_SUCCESS
        )
            return ret;
    }
    
    if(ret == APR_SUCCESS)
        ret = bt_bcode_dict_add(p, bcode, pos, "files", files);        
    
    return ret;
}

static inline apr_status_t bt_metainfo_bcode_pieces(
    bt_metainfo* metainfo, bt_bcode* bcode, apr_pool_t* p, unsigned int pos
) {
    bt_bcode* pieces;
    
    if((pieces = bt_bcode_new_strn(
        p, metainfo->pieces, BT_HASH_LEN * metainfo->piece_count
    ))) {
        return bt_bcode_dict_add(p, bcode, pos, "pieces", pieces);
    } else {
        fprintf(
            stderr, "bt_metainfo_bcode_pieces: bcodeing piece hashes failed\n"
        );
        return APR_ENOENT;
    }
}

static inline apr_status_t bt_metainfo_bcode_info(
    bt_metainfo* metainfo, bt_bcode* bcode, apr_pool_t* p, unsigned int pos
) {
    apr_status_t ret;
    int mul = metainfo->file_count > 1 ? 1 : 0;
    bt_bcode* info = bt_bcode_new_dict(p, 4);
    
    if(
        (ret = bt_bcode_dict_add(
            p, info, 2, "piece length",
            bt_bcode_new_int(p, metainfo->piece_size)
        )) == APR_SUCCESS
    ) {
        if(mul) {
            ret = bt_metainfo_bcode_files(metainfo, info, p, 0);
            if(ret != APR_SUCCESS)
                fprintf(
                    stderr,
                    "bt_metainfo_bcode_info: failed to add files block\n"
                );
        } else {
            ret = bt_bcode_dict_add(
                p, info, 0, "length", bt_bcode_new_int(p, metainfo->total_size)
            );
            if(ret != APR_SUCCESS)
                fprintf(
                    stderr,
                    "bt_metainfo_bcode_info: failed to add length\n"
                );
        }
        
        if(ret == APR_SUCCESS) {
            if(
                (ret = bt_bcode_dict_add(
                    p, info, 1, "name", bt_bcode_new_str(p, metainfo->name)
                ))
                == APR_SUCCESS
            ) {
                if(
                    (ret = bt_metainfo_bcode_pieces(metainfo, info, p, 3))
                    == APR_SUCCESS
                ) {
                    ret = bt_bcode_dict_add(p, bcode, pos, "info", info);
                    if(ret != APR_SUCCESS) {
                        fprintf(
                            stderr,
                            "bt_metainfo_bcode_info: failed to add "
                            "info block\n"
                        );
                    }
                } else {
                    fprintf(
                        stderr,
                        "bt_metainfo_bcode_info: failed to get "
                        "bcoded piece info\n"
                    );
                }
            } else {
                fprintf(
                    stderr,
                    "bt_metainfo_bcode_info: failed to add name\n"
                );
            }
        }
    } else {
        fprintf(
            stderr, "bt_metainfo_bcode_info: failed to add piece length\n"
        );
    }
    
    return ret;
}

apr_status_t bt_metainfo_encode(
    bt_metainfo* metainfo, apr_pool_t* p, bt_bcode** bcode_p
) {
    apr_status_t ret;
    int cd = metainfo->creation_date ? 1 : 0;
    bt_bcode* bcode = bt_bcode_new_dict(p, 2 + cd);

    if(
        (ret = bt_bcode_dict_add(
            p, bcode, 0, "announce",
            bt_bcode_new_str(p, bt_args2url(
                p, metainfo->tracker_address, metainfo->tracker_port,
                metainfo->tracker_announce
            ))
        )) == APR_SUCCESS
    ) {
        if(cd) {
            ret = bt_bcode_dict_add(
                p, bcode, 1, "creation date",
                bt_bcode_new_int(p, metainfo->creation_date)
            );
        }
        
        if(
            ret == APR_SUCCESS &&
            (
                (ret =  bt_metainfo_bcode_info(metainfo, bcode, p, 1 + cd))
                == APR_SUCCESS
            )
        ) {
            *bcode_p = bcode;
            return APR_SUCCESS;
        } else {
            fprintf(stderr, "bt_metainfo_encode: failed to add info block\n");
        }
    } else {
        fprintf(stderr, "bt_metainfo_encode: failed to add announce url\n");
    }
    
    return ret;
}

apr_status_t bt_metainfo_save(
    bt_metainfo* metainfo, apr_pool_t* p, const char* path
) {
    int ret;
    bt_bcode* bcode;
    apr_pool_t* pool = NULL;
    
    if((ret = apr_pool_create(&pool, p)) == APR_SUCCESS) {
        if(
            (ret = bt_metainfo_encode(metainfo, pool, &bcode))
            == APR_SUCCESS
        ) {
            char* buf;
            apr_size_t len;
            if(
                (ret = bt_bcode_encode(p, bcode, &buf, &len, NULL))
                == BT_BCODE_SUCCESS
            ) {
                apr_file_t* file;
                if(
                    (ret = apr_file_open(
                        &file, path, APR_WRITE | APR_CREATE | APR_TRUNCATE,
                        APR_FPROT_OS_DEFAULT, p
                    ))
                    == APR_SUCCESS
                ) {
                    apr_size_t actual;
                    ret = apr_file_write_full(file, buf, len, &actual);
                    apr_file_close(file);
                }
            } else {
                fprintf(
                    stderr, "bt_metainfo_save: encoding: %s\n",
                    bt_bcode_error(ret)
                );
            }
        }
    }
    
    if(pool) apr_pool_destroy(pool);
    return ret;
}

static char* bt_metainfo_load(apr_pool_t* p, const char* path) {
    char*               buf;
    apr_file_t*         file;
    apr_size_t          len;
    struct stat         sinfo;
    int                 rv;

    bzero(&sinfo, sizeof(sinfo));

    if(stat(path, &sinfo)) {
        fprintf(stderr, "Failed to stat %s\n", path);
        return NULL;
    }
    
    if(!S_ISREG(sinfo.st_mode)) {
        fprintf(stderr, "%s: Not a regular file.\n", path);
        return NULL;
    }
    
    if(sinfo.st_size > BT_MAX_METAINFO_SIZE) {
        fprintf(
            stderr, "%s: file to large ("BT_OFF_T_FMT" bytes > %u bytes)",
            path, sinfo.st_size, BT_MAX_METAINFO_SIZE
        );
        return NULL;
    }
    
    /* Load the torrent file into our buffer */
    if((rv = apr_file_open(
        &file, path, APR_READ | APR_BINARY, APR_OS_DEFAULT, p)) != APR_SUCCESS
    ) {
        fprintf(
            stderr, "Could not open file (%s): %s\n", path,
            apr_strerror(rv, errorstr, BT_SHORT_STRING)
        );
        return NULL;
    }
    
    len = sinfo.st_size;
    buf = apr_palloc(p, len);
    
    if((rv = apr_file_read(file, buf, &len)) != APR_SUCCESS) {
        fprintf(
            stderr, "Failed to read " BT_SIZE_T_FMT " bytes from %s: %s\n",
            (bt_size_t) sinfo.st_size, path,
            apr_strerror(rv, errorstr, BT_SHORT_STRING)
        );
        return NULL;
    }
    
    if(len != sinfo.st_size) {
        fprintf(
            stderr,
            "%s: expected " BT_SIZE_T_FMT " bytes, but got "
            BT_SIZE_T_FMT "!\n", path,
            (bt_size_t) sinfo.st_size, (bt_size_t) len
        );
        return NULL;
    }
    
    apr_file_close(file);
    
    return buf;
}

static inline bt_metainfo_finfo bt_metainfo_finfo_stat(
    const char* name, struct stat sb
) {
    bt_metainfo_finfo rv;
    rv.length = sb.st_size;
    BT_STRCPY(rv.name, name);
    return rv;
}

static apr_status_t bt_metainfo_init_files_recurse(
    bt_metainfo* info, apr_pool_t* p,
    bt_metainfo_finfo* files,
    const char* path, int* fn, const char* pre
) {
    int ret;
    apr_dir_t* dir = NULL;
    char err[255];
    
    /* using both sb and finfo is an annoying hack to get around an APR problem
     * when APR is compiled without large file support, it has problems
     * linking to programs that are compiled with large file support because
     * it's headers trust the compiler's definition before APR's.
     */
    
    apr_finfo_t finf = {0};
    struct stat sb;
    const char* dir_path = pre ? apr_pstrcat(p, path, "/", pre, NULL) : path;
    const char* prep = pre ? apr_pstrcat(p, pre, "/", NULL) : "";
       
    if((ret = apr_dir_open(&dir, dir_path, p)) == APR_SUCCESS) {
        while(ret == APR_SUCCESS) {
            if((ret = apr_dir_read(&finf, 0, dir)) == APR_SUCCESS) {
                char* fpath;
                
                if (!strcmp(finf.name, ".") || !strcmp(finf.name, "..")) {
                    continue;
                }
                
                fpath = apr_pstrcat(p, dir_path, "/", finf.name, NULL);
            
                if(finf.filetype == APR_DIR) {
                    ret = bt_metainfo_init_files_recurse(
                        info, p, files, path, fn,
                        apr_pstrcat(p, prep, finf.name, NULL)
                    );
                } else if(finf.filetype == APR_REG) {
                    
                    if(stat(fpath, &sb)) {
                        /* TODO: fix this return value */
                        fprintf(
                            stderr, "bt_metainfo_init_files_recurse: "
                            "failed to stat %s: %s\n",
                            fpath, strerror(errno)
                        );
                        ret = APR_FROM_OS_ERROR(errno);
                    } else if(*fn >= BT_METAINFO_MAX_FILES) {
                        fprintf(
                            stderr, "bt_metainfo_init_files_recurse: "
                            "more than %i files in %s!\n",
                            BT_METAINFO_MAX_FILES, dir_path
                        );
                        ret = APR_ENOMEM;
                    } else {
                        files[*fn] = bt_metainfo_finfo_stat(
                            apr_pstrcat(
                                p, info->name, "/", prep, finf.name, NULL
                            ),
                            sb
                        );
                        *fn = *fn + 1;
                    }
                } else {
                    fprintf(
                        stderr, "bt_metainfo_init_files_recurse: "
                        "\"%s\" is not a file or a directory!\n",
                        fpath
                    );
                    ret = APR_EINVAL;
                }
            } else {
                if(ret != APR_ENOENT) {
                    fprintf(
                        stderr, "bt_metainfo_init_files_recurse: "
                        "failed to read from directory %s: %s\n",
                        dir_path, apr_strerror(ret, err, sizeof(err))
                    );
                }
            }
        }
        
        if(ret == APR_ENOENT)
            ret = APR_SUCCESS;
        
        if(ret != APR_SUCCESS) {
            fprintf(
                stderr, "failed to read directory \"%s\": %s\n",
                dir_path, apr_strerror(ret, err, sizeof(err))
            );
        }
        
        apr_dir_close(dir);
    } else {
        fprintf(
            stderr, "failed to open directory \"%s\": %s\n",
            dir_path, apr_strerror(ret, err, sizeof(err))
        );
    }
    
    return ret;
}

static int bt_metainfo_cmp_finfo(const void* a, const void* b) {
    return strcmp(((bt_metainfo_finfo*)a)->name, ((bt_metainfo_finfo*)b)->name);
}

static apr_status_t bt_metainfo_init_files_dir(
    bt_metainfo* info, apr_pool_t* p, const char* path
) {
    bt_metainfo_finfo files[BT_METAINFO_MAX_FILES];
    int fn = 0;
    int ret;

    if(
        (ret = bt_metainfo_init_files_recurse(info, p, files, path, &fn, NULL))
        == APR_SUCCESS
    ) {
        int i;
        info->total_size = 0;
        for(i=0;i<fn;i++)
            info->total_size += files[i].length;
                
        qsort(files, fn, sizeof(bt_metainfo_finfo), bt_metainfo_cmp_finfo);
        info->file_count = fn;
        info->files = apr_palloc(p, sizeof(bt_metainfo_finfo) * fn);
        memcpy(info->files, files, sizeof(bt_metainfo_finfo) * fn);
    }
    
    return ret;
}
    
apr_status_t bt_metainfo_init_files(
    bt_metainfo* info, apr_pool_t* p, const char* path
) {
    struct stat sb;
    apr_status_t rv;
    char* ppath = apr_pstrdup(p, path);
    char* xpath = rindex(ppath, '/');
    const char* dpath = xpath ? xpath + 1 : ppath;
    
    if(xpath)
        *xpath = 0;
    
        
    if(stat(path, &sb)) 
        return APR_FROM_OS_ERROR(errno);

    BT_STRCPY(info->name, dpath);
    
    if(S_ISDIR(sb.st_mode)) {
        rv = bt_metainfo_init_files_dir(info, p, path);
    } else {
        rv = APR_SUCCESS;
        info->file_count = 1;
        info->files = apr_palloc(p, sizeof(bt_metainfo_finfo));
        info->files[0] = bt_metainfo_finfo_stat(dpath, sb);
        info->total_size = info->files[0].length;
    }
    
    if(rv == APR_SUCCESS && info->piece_size) {
        /* If we have a piece size, update the piece count
         * and allocate a new piece buffer
         */
        info->piece_count = info->total_size / info->piece_size;
        if(info->total_size % info->piece_size)
            info->piece_count++;
        info->pieces = apr_pcalloc(p, BT_INFOHASH_LEN * info->piece_count);
    }
        
    return rv;
}
